/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: basic_status.cpp,v 1.9 2013/10/09 14:18:13 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "basic_status.h"
#include "sql_connection.h"
#include "sql_privileges_flags.h"


BasicStatus::BasicStatus(ServerInfo *info, QWidget *parent)
	:QWidget(parent), SqlService(info, "basic")
{
	SqlQuery::Scope	scope = SqlQuery::S_GLOBAL;
	SqlQuery::Table	table = SqlQuery::T_STATUS;
	QPalette	plt;

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	area = new QScrollArea(this);
	area->setWidgetResizable(TRUE);
	area->setAlignment(Qt::AlignHCenter | Qt::AlignTop);
	area->setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
	area->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
	layout->addWidget(area, 1);

	box = new QDialogButtonBox(QDialogButtonBox::Close,
		Qt::Horizontal, this);
	connect(box, SIGNAL(rejected()), SIGNAL(reject()));
	layout->addWidget(box, 0, Qt::AlignBottom);

	sdb = new QPushButton(QIcon(":/icons/shutdown.png"),
		tr("Shutdown"), this);
	sdb->setEnabled(FALSE);
	plt = sdb->palette();
	plt.setBrush(QPalette::Button, BRUSH_SHUTDOWN);
	sdb->setPalette(plt);
	connect(sdb, SIGNAL(clicked()), SIGNAL(shutdown()));
	box->addButton(sdb, QDialogButtonBox::ActionRole);

	rb = new QPushButton(QIcon(":/icons/table.png"), tr("Refresh"), this);
	rb->setEnabled(FALSE);
	connect(rb, SIGNAL(clicked()), SIGNAL(refresh()));
	box->addButton(rb, QDialogButtonBox::ActionRole);

	status = new QWidget(area);
	area->setWidget(status);

	form = new QFormLayout(status);
	form->setHorizontalSpacing(status->fontMetrics().lineSpacing());
	form->setFormAlignment(Qt::AlignTop | Qt::AlignLeft);

	q = new QueriesItem(this);
	form->addRow(tr("Queries"), q);
	vect.append(q);

	q_avg = new QLabel("0.0", status);
	form->addRow(tr("Average queries per second"), q_avg);

	addStatus(scope, table, "Slow_queries", tr("Slow queries"));
	addStatus(scope, table, "Questions", tr("Questions"));
	addStatus(scope, table, "Threads_running", tr("Running threads"));
	addStatus(scope, table, "Threads_connected", tr("Connected threads"));
	addStatus(scope, table, "Open_files", tr("Open files"));
	addStatus(scope, table, "Open_tables", tr("Open tables"));
	addStatus(scope, table, "Aborted_clients", tr("Aborted clients"));
	addStatus(scope, table, "Aborted_connects", tr("Aborted connections"));
	addStatus(scope, table, "Access_denied_errors",
		tr("Access denied errors"));
	addStatus(scope, table, "Compression", tr("Protocol compression"));
	addStatus(scope, table, "Bytes_received",
		tr("Received traffic (Bytes)"));
	addStatus(scope, table, "Bytes_sent", tr("Sent traffic (Bytes)"));

	uptime = new BasicStatusItem(scope, table, "Uptime", this);
	form->addRow(tr("Server uptime (s)"), uptime);
	vect.append(uptime);

	qsize = new QLabel("0", this);
	form->addRow(tr("Number of queries in the queue"), qsize);
}

BasicStatus::~BasicStatus()
{

	destroyVector();
	delete qsize;
	delete q_avg;
	delete form;
	delete status;
	delete rb;
	delete sdb;
	delete box;
	delete area;
	delete layout;
}


void
BasicStatus::destroyIt()
{

	/* There is nothing to do here */
}


void
BasicStatus::timeout()
{
	BasicStatusVector::iterator	it;
	BasicStatusItem			*item;

	qsize->setText(QString::number(connection()->size()));
	for (it = vect.begin(); it != vect.end(); ++it) {
		item = *it;
		item->timeout();
		if (item == q)
			computeAvgQ();
	}
}


QString
BasicStatus::tabName()
{

	return (tr("Basic"));
}


void
BasicStatus::addStatus(SqlQuery::Scope scope, SqlQuery::Table table,
	const QString &name, const QString &descr)
{
	BasicStatusItem	*item;

	item = new BasicStatusItem(scope, table, name, this);
	form->addRow(descr, item);
	vect.append(item);
}


void
BasicStatus::destroyVector()
{
	BasicStatusVector::iterator	it;
	BasicStatusItem			*item;

	for (it = vect.begin(); it != vect.end(); ++it) {
		item = *it;
		if (item == q)
			delete q;
		else
			delete item;
	}
	vect.clear();
}


/* The algorithm was taken from MariaDB-10.0.3 */
void
BasicStatus::computeAvgQ()
{
	quint64	tmp = uptime->text().toULongLong();

	if (tmp > 0) {
		double	diff = (double)q->queries() / (double)tmp;

		q_avg->setText(QString::number(diff, 'g', 4));
	}
}


SqlConnection *
BasicStatus::connection()const
{

	return (SqlService::connection());
}


void
BasicStatus::privilegesChanged()
{

	sdb->setEnabled(info()->priv.isSet(PRIV_SHUTDOWN));
	rb->setEnabled(info()->priv.isSet(PRIV_RELOAD));
}

//-----------------------------------------------------------------------------

BasicStatusItem::BasicStatusItem(SqlQuery::Scope scope, SqlQuery::Table table,
	const QString &name, QWidget *parent)
	:QLabel(parent)
{
	BasicStatus	*status = static_cast<BasicStatus *>(parent);
	SqlConnection	*conn = status->connection();

	query = new SqlQuery(conn);
	query->connect(this, SLOT(readData()));
	query->prepare(SqlQuery::query(name, table, scope));

	setText(Core::text(Core::Undefined));
}

BasicStatusItem::~BasicStatusItem()
{

	delete query;
}


void
BasicStatusItem::timeout()
{

	query->exec();
}


void
BasicStatusItem::readData()
{

	if (query->next())
		setText(query->value(0).toString());
	query->free();
}

//-----------------------------------------------------------------------------

QueriesItem::QueriesItem(QWidget *parent)
	:BasicStatusItem(SqlQuery::S_GLOBAL, SqlQuery::T_STATUS,
		"Queries", parent)
{

	setText("0");
}

QueriesItem::~QueriesItem()
{

}


quint64
QueriesItem::queries()const
{

	return (text().toULongLong());
}

/* EOF */
