/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: settings.cpp,v 1.6 2013/10/09 14:18:13 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "settings.h"
#include "sql_service.h"


#define	CFG_FORMAT		QSettings::NativeFormat
#define	CFG_SCOPE		QSettings::UserScope
#define	CFG_ORGANIZATION	QCoreApplication::organizationName()
#define	CFG_PROGRAM		QCoreApplication::applicationName()

#define	CFG_MYSQL_GROUP		"mysql"
#define	CFG_MYSQL_SERVER	"server"
#define	CFG_PLUGIN_GROUP	"plugins"
#define	CFG_PLUGIN		"plugin"
#define	CFG_PLUGIN_FILE		"file"

#define	CFG_WINDOW_GROUP	"window"
#define	CFG_WINDOW_GEOMETRY	"geometry"



Settings::Settings()
{

	setTranslation(FALSE);
}

Settings::~Settings()
{

	destroyVector();
}


void
Settings::load()
{
	QSettings	cfg(CFG_FORMAT, CFG_SCOPE, CFG_ORGANIZATION,
				CFG_PROGRAM);

	loadServers(cfg);
	loadPlugins(cfg);
	loadGeometry(cfg);
}


void
Settings::save()
{
	QSettings	cfg(CFG_FORMAT, CFG_SCOPE, CFG_ORGANIZATION,
				CFG_PROGRAM);

	saveServers(cfg);
	savePlugins(cfg);
	saveGeometry(cfg);
}


void
Settings::setGeometry(const QRect &g)
{

	geom = g;
}


void
Settings::setTranslation(bool tr)
{

	p_tr = tr;
}


QRect
Settings::geometry()const
{

	return (geom);
}


bool
Settings::hasTranslation()const
{

	return (p_tr);
}


Settings::operator SqlAccountVector*()
{

	return (&vect);
}


Settings::operator PluginData*()
{

	return (&plg_list);
}


void
Settings::destroyVector()
{
	SqlAccountVector::iterator	it;
	SqlAccount			*ptr;

	for (it = vect.begin(); it != vect.end(); ++it) {
		ptr = *it;
		FATAL(ptr == NULL);
		delete ptr;
	}
}


void
Settings::loadServers(QSettings &cfg)
{
	int		i;
	SqlAccount	acc;

	cfg.beginGroup(CFG_MYSQL_GROUP);
	cfg.beginReadArray(CFG_MYSQL_SERVER);
	for (i = 0; ; ++i) {
		cfg.setArrayIndex(i);
		if (acc.load(cfg))
			vect.append(new SqlAccount(acc));
		else
			break;
	}
	cfg.endArray();
	cfg.endGroup();
}


void
Settings::loadPlugins(QSettings &cfg)
{
	int	i;

	cfg.beginGroup(CFG_PLUGIN_GROUP);
	cfg.beginReadArray(CFG_PLUGIN);
	for (i = 0; ; ++i) {
		cfg.setArrayIndex(i);
		if (!plg_list.load(cfg))
			break;
	}
	cfg.endArray();
	cfg.endGroup();
}


void
Settings::loadGeometry(QSettings &cfg)
{
	QRect	def_geom(QPoint(0, 0), DEFAULT_WINDOW_SIZE);

	cfg.beginGroup(CFG_WINDOW_GROUP);
	geom = cfg.value(CFG_WINDOW_GEOMETRY, def_geom).toRect();
	cfg.endGroup();
}


void
Settings::saveServers(QSettings &cfg)
{
	SqlAccountVector::const_iterator	it;
	int		i = 0;

	cfg.remove(CFG_MYSQL_GROUP);
	cfg.beginGroup(CFG_MYSQL_GROUP);
	cfg.beginWriteArray(CFG_MYSQL_SERVER);
	for (it = vect.constBegin(); it != vect.constEnd(); ++it, ++i) {
		cfg.setArrayIndex(i);
		(*it)->save(cfg);
	}
	cfg.endArray();
	cfg.endGroup();
}


void
Settings::savePlugins(QSettings &cfg)
{

	cfg.remove(CFG_PLUGIN_GROUP);
	cfg.beginGroup(CFG_PLUGIN_GROUP);
	cfg.beginWriteArray(CFG_PLUGIN);
	plg_list.save(cfg);
	cfg.endArray();
	cfg.endGroup();
}


void
Settings::saveGeometry(QSettings &cfg)
{

	cfg.beginGroup(CFG_WINDOW_GROUP);
	cfg.setValue(CFG_WINDOW_GEOMETRY, geom);
	cfg.endGroup();
}

//-----------------------------------------------------------------------------

PluginData::PluginData()
{
}

PluginData::~PluginData()
{

	destroyVector();
}


void
PluginData::destroyVector()
{
	PluginVector::iterator	it;
	QPluginLoader		*plg;

	for (it = vect.begin(); it != vect.end(); ++it) {
		plg = *it;
		plg->unload();
		delete plg;
	}
}


QPluginLoader *
PluginData::load(const QString &file, QString *error)
{
	QPluginLoader	*ret;
	ServicePlugin	*plg;

	ret = new QPluginLoader();
	FATAL(ret == NULL);
	ret->setFileName(file);
	if (!ret->load()) {
		if (error != NULL)
			*error = ret->errorString();
		SAFE_DELETE(ret);
	} else {
		/* loaded */
		plg = PLUGIN_CAST(ret);
		if (plg == NULL) {
			/* it is not mine! */
			if (error != NULL)
				*error = QObject::tr("The file %1 is not my "
					"plugin").arg(file);
			ret->unload();
			SAFE_DELETE(ret);
		}
	}

	return (ret);
}


void
PluginData::insert(unsigned int pos, QPluginLoader *plg)
{
	ServicePlugin	*srv = PLUGIN_CAST(plg);

	vect.insert(pos, plg);
	map.insert(srv->name(), plg);
}


/* Returns a position where a new `name` should be inserted */
unsigned int
PluginData::findPosition(const QString &name)const
{
	unsigned int			ret = 0, i = 0;
	PluginVector::const_iterator	it;
	ServicePlugin			*plg;

	for (it = vect.constBegin(); it != vect.constEnd(); ++it, ++i) {
		plg = PLUGIN_CAST(*it);
		FATAL(plg == NULL);
		if (plg->name().compare(name, Qt::CaseInsensitive) <= 0) {
			ret = i;
			break;
		}
	}

	return (ret);
}


int
PluginData::size()const
{

	return (vect.size());
}


QPluginLoader *
PluginData::at(unsigned int pos)
{
	QPluginLoader	*ret;

	if (pos < (unsigned int)vect.size())
		ret = vect.at(pos);
	else
		ret = NULL;

	return (ret);
}


QPluginLoader *
PluginData::lookup(const QString &name)
{

	return (map.value(name, NULL));
}


void
PluginData::save(QSettings &cfg)
{
	PluginVector::const_iterator	it;
	QPluginLoader			*plg;
	unsigned int			i = 0;

	for (it = vect.constBegin(); it != vect.constEnd(); ++it, ++i) {
		plg = *it;
		FATAL(plg == NULL);
		cfg.setArrayIndex(i);
		cfg.setValue(CFG_PLUGIN_FILE, plg->fileName());
	}
}


bool
PluginData::load(QSettings &cfg)
{
	QPluginLoader	*plg;
	bool		ret;
	QVariant	null(QString::null);
	QString		file(cfg.value(CFG_PLUGIN_FILE, null)
				.toString());
	unsigned int	pos;
	ServicePlugin	*srv;

	if (file.isEmpty())
		ret = FALSE;
	else if ((plg = load(file)) == NULL)
		ret = FALSE;
	else if ((srv = PLUGIN_CAST(plg)) != NULL) {
		pos = findPosition(srv->name());
		insert(pos, plg);
		ret = TRUE;
	} else
		ret = FALSE;

	return (ret);
}


void
PluginData::remove(unsigned int pos)
{
	QPluginLoader	*plg = vect.at(pos);
	ServicePlugin	*srv = PLUGIN_CAST(plg);

	vect.remove(pos);
	map.remove(srv->name());
	plg->unload();
	delete plg;
}


bool
PluginData::isEmpty()const
{

	return (vect.empty());
}


PluginData::operator PluginVector*()
{

	return (&vect);
}

/* EOF */
