/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: sql_account_model.cpp,v 1.7 2013/08/20 10:12:39 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "sql_account_model.h"

#define	LAB_SERVER	0
#define	LAB_USER	1
#define	LAB_PASSWORD	2
#define	LAB_SSL		3
#define	LAB_STATUS	4
#define	LAB_TOTAL	(LAB_STATUS + 1)


SqlAccountModel::SqlAccountModel(SqlAccountVector *ptr, QObject *parent)
	:QAbstractTableModel(parent), vect(ptr)
{
	SqlAccountVector::const_iterator	it;
	SqlAccount				*acc;

	for (it = vect->constBegin(); it != vect->constEnd(); ++it) {
		acc = *it;
		map.insert(acc->account(), acc);
	}
}

SqlAccountModel::~SqlAccountModel()
{
}


QVariant
SqlAccountModel::data(const QModelIndex &ndx, int role)const
{
	QVariant	ret;
	int		row = ndx.row();
	SqlAccount	*acc;

	if ((row >= 0) && (row < vect->size())) {
		acc = vect->at(row);
		switch (role) {
			case Qt::DisplayRole:
				ret = displayData(acc, ndx.column());
				break;
			case Qt::BackgroundRole:
				ret = backgroundData(acc);
				break;
		}
	}

	return (ret);
}


QVariant
SqlAccountModel::headerData(int sct, Qt::Orientation ort, int role)const
{
	QVariant	ret;

	if ((ort == Qt::Horizontal) && (role == Qt::DisplayRole))
		switch (sct) {
			case LAB_SERVER:
				ret = tr("Server");
				break;
			case LAB_USER:
				ret = tr("Username");
				break;
			case LAB_PASSWORD:
				ret = tr("Password");
				break;
			case LAB_SSL:
				ret = tr("SSL");
				break;
			case LAB_STATUS:
				ret = tr("Status");
				break;
		}

	return (ret);
}


int
SqlAccountModel::columnCount(const QModelIndex&)const
{

	return (LAB_TOTAL);
}


int
SqlAccountModel::rowCount(const QModelIndex&)const
{

	return (vect->size());
}


bool
SqlAccountModel::exists(const SqlAccount &acc)const
{

	return (map.contains(acc.account()));
}


SqlAccount *
SqlAccountModel::remove(const SqlAccount &acc)
{
	int		i;
	QString		key(acc.account());
	SqlAccount	*ret = map.value(key, NULL), *tmp;
	SqlAccountVector::iterator	it;

	if (ret != NULL) {
		map.remove(key);
		for (i = 0, it = vect->begin(); it != vect->end(); ++it, ++i) {
			tmp = *it;
			if (tmp == ret) {
				beginRemoveRows(QModelIndex(), i, i);
				vect->erase(it);
				endRemoveRows();
				break;
			}
		}
	}

	return (ret);
}


SqlAccount *
SqlAccountModel::account(const QModelIndex &ndx)const
{
	int		row = ndx.row();
	SqlAccount	*ret;

	if ((row < 0) || (row >= vect->size()))
		ret = NULL;
	else
		ret = vect->at(row);

	return (ret);
}


SqlAccount *
SqlAccountModel::account(const SqlAccount &acc)const
{

	return (map.value(acc.account(), NULL));
}


void
SqlAccountModel::insert(const SqlAccount &acc)
{
	SqlAccount	*ptr;
	int		pos;

	if (vect->empty()) {
		pos = 0;
	} else {
		pos = findLocation(acc);
	}

	ptr = new SqlAccount(acc);
	map.insert(ptr->account(), ptr);
	beginInsertRows(QModelIndex(), pos, pos);
	vect->insert(pos, ptr);
	endInsertRows();
}


void
SqlAccountModel::doConnection(const QModelIndex &ndx)
{
	SqlAccount	*ptr = account(ndx);
	int		row;
	QModelIndex	mch;

	if (ptr != NULL) {
		emit actionConnect(ptr);
		row = ndx.row();
		mch = index(row, LAB_STATUS);
		emit dataChanged(mch, mch);
	}
}


QVariant
SqlAccountModel::displayData(const SqlAccount *acc, int column)const
{
	static QString	connected(tr("Connected")),
			disconnected(tr("Disconnected"));
	QVariant	ret;

	switch (column) {
		case LAB_SERVER:
			ret = acc->host();
			break;
		case LAB_USER:
			ret = acc->user();
			break;
		case LAB_PASSWORD:
			ret = Core::hasValue(acc->hasPassword());
			break;
		case LAB_SSL:
			ret = Core::hasValue(acc->isSet(SQL_SSL));
			break;
		case LAB_STATUS:
			ret = (acc->hasStatus(ACC_CONNECTED)?
				connected:disconnected);
			break;
	}

	return (ret);
}


QVariant
SqlAccountModel::backgroundData(const SqlAccount *acc)const
{
	QVariant	ret;
	bool		pwd = acc->hasPassword(),
			usr = !acc->user().isEmpty(),
			ssl = acc->isSet(SQL_SSL);

	if (acc->hasStatus(ACC_CONNECTED))
		ret = BRUSH_CONNECTED;
	else if (usr && pwd && ssl) {
		/* nothing to do here */
	} else if (!usr && !pwd && !ssl)
		ret = BRUSH_NO_SECURITY;
	else if ((!pwd && !ssl) || (!pwd && ssl))
		ret = BRUSH_LOW_SECURITY;
	else if (!ssl)
		ret = BRUSH_MID_SECURITY;

	return (ret);
}


int
SqlAccountModel::findLocation(const SqlAccount &acc)const
{
	SqlAccountVector::const_iterator	it;
	int					ret = vect->size(), i = 0;
	SqlAccount				*ptr;
	QString					tmp1, tmp2, pattern("%1@%2");

	tmp1 = pattern.arg(acc.host()).arg(acc.user());
	for (it = vect->constBegin(); it != vect->constEnd(); ++it, ++i) {
		ptr = *it;
		FATAL(ptr == NULL);
		tmp2 = pattern.arg(ptr->host()).arg(ptr->user());
		if (tmp1 <= tmp2) {
			ret = i;
			break;
		}
	}

	return (ret);
}


void
SqlAccountModel::accountChanged(SqlAccount *acc)
{
	int		pos = vect->indexOf(acc);
	QModelIndex	ndx;

	if (pos >= 0) {
		ndx = index(pos, LAB_STATUS);
		emit dataChanged(ndx, ndx);
	}
}

/* EOF */
