/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: sql_config.cpp,v 1.7 2013/08/20 10:12:39 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "sql_config.h"
#include "sql_connection.h"


#define	CFG_SQL_HOST		"host"
#define	CFG_SQL_PORT		"port"
#define	CFG_SQL_USER		"user"
#define	CFG_SQL_PASS		"password"
#define	CFG_SQL_FLAGS		"flags"
#define	CFG_SSL_KEY		"key"
#define	CFG_SSL_CERT		"cert"
#define	CFG_SSL_CA		"ca"
#define	CFG_SSL_CAPATH		"capath"
#define	CFG_SSL_CIPHER		"cipher"
#define	CFG_TIMEOUT_READ	"rtimeout"
#define	CFG_TIMEOUT_WRITE	"wtimeout"
#define	CFG_TIMEOUT_CONN	"ctimeout"


SqlConfig::SqlConfig()
	:SqlUser()
{
}

SqlConfig::SqlConfig(const SqlConfig &cfg)
	:SqlUser()
{

	setConfig(cfg);
}

SqlConfig::~SqlConfig()
{
}


void
SqlConfig::setPort(int port)
{

	if ((port <= 0) || (port > 65535))
		port = SqlConnection::defaultPort();
	sql_port = port;
}


void
SqlConfig::setReadTimeout(int to)
{

	sql_read_timeout = to;
}


void
SqlConfig::setWriteTimeout(int to)
{

	sql_write_timeout = to;
}


void
SqlConfig::setConnectTimeout(int to)
{

	sql_conn_timeout = to;
}


void
SqlConfig::setConfig(const SqlConfig &cfg)
{

	setHost(cfg.host());
	setUser(cfg.user());
	setPassword(cfg.password());
	setPort(cfg.port());
	setReadTimeout(cfg.readTimeout());
	setWriteTimeout(cfg.writeTimeout());
	setConnectTimeout(cfg.connectTimeout());
	sql_flags = cfg.flags() & SQL_FLAGS_MASK;
	setSSL(cfg.key(), cfg.cert(), cfg.ca(), cfg.caPath(), cfg.cipher());
}


void
SqlConfig::setFlag(flags_t flag, bool on)
{

	if (flag != 0) {
		if (on)
			sql_flags |= flag;
		else
			sql_flags &= (~flag);
	}
}


void
SqlConfig::setSSL(const QString &key, const QString &cert, const QString &ca,
	const QString &capath, const QString &cipher)
{

	ssl_key = key;
	ssl_cert = cert;
	ssl_ca = ca;
	ssl_capath = capath;
	ssl_cipher = cipher;
}


void
SqlConfig::save(QSettings &cfg)
{

	cfg.setValue(CFG_SQL_HOST, host());
	cfg.setValue(CFG_SQL_PORT, port());
	cfg.setValue(CFG_TIMEOUT_READ, readTimeout());
	cfg.setValue(CFG_TIMEOUT_WRITE, writeTimeout());
	cfg.setValue(CFG_TIMEOUT_CONN, connectTimeout());
	cfg.setValue(CFG_SQL_USER, user());
	cfg.setValue(CFG_SQL_PASS, encrypted());
	cfg.setValue(CFG_SQL_FLAGS, flags());
	if (isSet(SQL_SSL)) {
		cfg.setValue(CFG_SSL_KEY, key());
		cfg.setValue(CFG_SSL_CERT, cert());
		cfg.setValue(CFG_SSL_CA, ca());
		cfg.setValue(CFG_SSL_CAPATH, caPath());
		cfg.setValue(CFG_SSL_CIPHER, cipher());
	}
}


bool
SqlConfig::load(QSettings &cfg)
{
	bool	ret = cfg.contains(CFG_SQL_HOST);
	QString	null(QString::null);

	if (ret) {
		setHost(cfg.value(CFG_SQL_HOST, SqlConnection::defaultHost())
			.toString());
		setPort(cfg.value(CFG_SQL_PORT, SqlConnection::defaultPort())
			.toInt());
		setReadTimeout(cfg.value(CFG_TIMEOUT_READ, 0).toInt());
		setWriteTimeout(cfg.value(CFG_TIMEOUT_WRITE, 0).toInt());
		setConnectTimeout(cfg.value(CFG_TIMEOUT_CONN, 0).toInt());
		setUser(cfg.value(CFG_SQL_USER, SqlConnection::defaultUser())
			.toString());
		setPassword(decrypted(cfg.value(CFG_SQL_PASS, null)
			.toByteArray()));
		sql_flags = cfg.value(CFG_SQL_FLAGS, 0).toUInt() &
			SQL_FLAGS_MASK;
		if (isSet(SQL_SSL)) {
			/* SSL properties */
			setSSL(cfg.value(CFG_SSL_KEY, null).toString(),
				cfg.value(CFG_SSL_CERT, null).toString(),
				cfg.value(CFG_SSL_CA, null).toString(),
				cfg.value(CFG_SSL_CAPATH, null).toString(),
				cfg.value(CFG_SSL_CIPHER, null).toString());
		}
	}

	return (ret);
}


int
SqlConfig::port()const
{

	return (sql_port);
}


int
SqlConfig::readTimeout()const
{

	return (sql_read_timeout);
}


int
SqlConfig::writeTimeout()const
{

	return (sql_write_timeout);
}


int
SqlConfig::connectTimeout()const
{

	return (sql_conn_timeout);
}


flags_t
SqlConfig::flags()const
{

	return (sql_flags);
}


bool
SqlConfig::isSet(flags_t flag)const
{

	return ((flags() & flag) != 0);
}


QString
SqlConfig::key()const
{

	return (ssl_key);
}


QString
SqlConfig::cert()const
{

	return (ssl_cert);
}


QString
SqlConfig::ca()const
{

	return (ssl_ca);
}


QString
SqlConfig::caPath()const
{

	return (ssl_capath);
}


QString
SqlConfig::cipher()const
{

	return (ssl_cipher);
}


QString
SqlConfig::account()const
{
	QString		ret, haddr, acc;
	QHostAddress	addr;

	if (addr.setAddress(host())) {
		/* IP! */
		if (addr.protocol() == QAbstractSocket::IPv6Protocol) {
			/* IPv6! */
			haddr = QString("[%1]").arg(host());
		} else {
			/* IPv4 or unknown protocol */
			haddr = host();
		}
	} else {
		/* hostname */
		haddr = host();
	}

	acc = accountFormat(FALSE).arg(user()).arg(haddr);
	if (port() == SqlConnection::defaultPort())
		ret = acc;
	else
		ret = QString("%1:%2").arg(acc).arg(port());

	return (ret.toLower());
}


SqlConfig&
SqlConfig::operator=(const SqlConfig &cfg)
{

	setConfig(cfg);

	return (*this);
}


bool
SqlConfig::operator!=(const SqlConfig &cfg)const
{
	Qt::CaseSensitivity	how = Qt::CaseInsensitive;
	bool			ret;

 	ret = (QString::compare(host(), cfg.host(), how) != 0);
 	if (ret)
 		ret = (QString::compare(user(), cfg.user(), how) != 0);

	return (ret);
}


QByteArray
SqlConfig::encrypted()const
{

	return (qCompress(password().toUtf8().toBase64().toHex()));
}


QString
SqlConfig::decrypted(const QByteArray &pwd)const
{

	return (QByteArray::fromBase64(QByteArray::fromHex(qUncompress(pwd))));
}

/* EOF */
