/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: sql_config_dialog.cpp,v 1.7 2013/10/30 07:20:44 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "sql_config_dialog.h"
#include "sql_connection.h"
#include "sql_error.h"

#define	BUTTON_SAVE	QDialogButtonBox::Save
#define	BUTTON_CANCEL	QDialogButtonBox::Cancel
#define	BUTTONS		(BUTTON_SAVE | BUTTON_CANCEL)


SqlConfigDialog::SqlConfigDialog(QWidget *parent)
	:QDialog(parent)
{

	layout = new QVBoxLayout(this);
	/* MARGINS_ZERO(layout); */

	win = new SqlConfigWidget(this);
	connect(win, SIGNAL(changed()), this, SLOT(canSave()));
	layout->addWidget(win, 1);

	dlg = new QDialogButtonBox(BUTTONS, Qt::Horizontal, this);
	connect(dlg, SIGNAL(rejected()), this, SLOT(askReject()));
	connect(dlg, SIGNAL(accepted()), this, SLOT(askAccept()));
	layout->addWidget(dlg, 0, Qt::AlignBottom);

	tryConn = new QPushButton(tr("Try connection"), this);
	connect(tryConn, SIGNAL(clicked()), this, SLOT(tryConnection()));
	dlg->addButton(tryConn, QDialogButtonBox::ActionRole);

	canSave(FALSE);
}

SqlConfigDialog::~SqlConfigDialog()
{

	delete tryConn;
	delete dlg;
	delete win;
	delete layout;
}


void
SqlConfigDialog::keyPressEvent(QKeyEvent *e)
{

	if (e->key() == Qt::Key_Escape)
		askReject();
	else
		QDialog::keyPressEvent(e);
}


void
SqlConfigDialog::closeEvent(QCloseEvent *e)
{

	e->ignore();
	askReject();
}


void
SqlConfigDialog::askReject()
{
	bool	done = TRUE;

	if (isWindowModified()) {
		done = (QMessageBox::question(this,
			tr("Discard configuration"),
			tr("All changes will be lost!\n"
			"Do you want to discard the data?"),
			QMessageBox::Discard | QMessageBox::Cancel,
			QMessageBox::Cancel) == QMessageBox::Discard);
	}
	if (done)
		reject();
}


/* Enables or disables "Save" button */
void
SqlConfigDialog::canSave(bool mod)
{
	QPushButton	*btn = dlg->button(BUTTON_SAVE);
	bool		ena = win->canAccept();

	if (btn != NULL)
		btn->setEnabled(ena);

	tryConn->setEnabled(ena);
	setWindowModified(mod);
}


/* Tries to connect to the selected SQL server */
void
SqlConfigDialog::tryConnection()
{
	SqlConfig	cfg;
	SqlError	err;

	config(cfg);
	if (SqlConnection::tryConnection(cfg, err))
		QMessageBox::information(this, tr("Connected"),
			tr("The connection established successfully"));
	else
		QMessageBox::critical(this, tr("Connection error"),
			tr("Connection to %1 failed\n%2\nError code %3")
			.arg(cfg.account()).arg(err.error()).arg(err.errcode()));
}


void
SqlConfigDialog::config(SqlConfig &cfg)
{

	win->config(cfg);
}


void
SqlConfigDialog::setConfig(const SqlConfig &cfg)
{

	win->setConfig(cfg);
	canSave(FALSE);
}


void
SqlConfigDialog::askAccept()
{
	bool	usr = win->hasUser(),
		pwd = win->hasPassword(),
		wish = TRUE;
	QString	title(tr("Security problem"));

	if (!usr && !pwd) {
		/* no user and no password! */
		wish = (QMessageBox::question(this, title,
			tr("I hope it is a mistake because a connection "
			"without user name and password is absolutely "
			"insecure and very dangerous for data safety.\n"
			"Are you sure that I should use it as is?"),
			QMessageBox::Yes | QMessageBox::No,
			QMessageBox::No) == QMessageBox::Yes);
	} else if (!pwd) {
		/* no password */
		wish = (QMessageBox::question(this, title,
			tr("Did you forget the password?\n"
			"If not, it is insecure and dangerous!\n"
			"Are you sure that I should use it as is?"),
			QMessageBox::Yes | QMessageBox::No,
			QMessageBox::No) == QMessageBox::Yes);
	}

	if (wish)
		accept();
}

/* EOF */
