/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: sql_config_widget.cpp,v 1.8 2013/10/09 14:18:13 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "sql_config_widget.h"
#include "sql_connection.h"

#define	SPECIAL_VALUE	0
#define	MAX_TIMEOUT	3600


SqlConfigWidget::SqlConfigWidget(QWidget *parent)
	:QWidget(parent)
{

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	form = new QFormLayout();
	layout->addLayout(form);
	MARGINS_ZERO(form);

	hostEdit = new QLineEdit(SqlConnection::defaultHost(), this);
	hostEdit->selectAll();
	connect(hostEdit, SIGNAL(textChanged(const QString&)),
		this, SLOT(dataChanged(const QString&)));
	form->addRow(tr("Host name"), hostEdit);

	userEdit = new QLineEdit(SqlConnection::defaultUser(), this);
	connect(userEdit, SIGNAL(textChanged(const QString&)),
		this, SLOT(dataChanged(const QString&)));
	form->addRow(tr("User name"), userEdit);

	passEdit = new QLineEdit(this);
	connect(passEdit, SIGNAL(textChanged(const QString&)),
		this, SLOT(dataChanged(const QString&)));
	form->addRow(tr("User's password"), passEdit);

	portEdit = new QSpinBox(this);
	portEdit->setRange(SPECIAL_VALUE, 65535);
	portEdit->setSpecialValueText(tr("default"));
	portEdit->setValue(portEdit->minimum());
	connect(portEdit, SIGNAL(valueChanged(const QString&)),
		this, SLOT(dataChanged(const QString&)));
	form->addRow(tr("Server port"), portEdit);

	readEdit = new QSpinBox(this);
	readEdit->setRange(SPECIAL_VALUE, MAX_TIMEOUT);
	readEdit->setSpecialValueText(tr("default"));
	readEdit->setValue(readEdit->minimum());
	connect(readEdit, SIGNAL(valueChanged(const QString&)),
		this, SLOT(dataChanged(const QString&)));
	form->addRow(tr("Read timeout"), readEdit);

	writeEdit = new QSpinBox(this);
	writeEdit->setRange(SPECIAL_VALUE, MAX_TIMEOUT);
	writeEdit->setSpecialValueText(tr("default"));
	writeEdit->setValue(writeEdit->minimum());
	connect(writeEdit, SIGNAL(valueChanged(const QString&)),
		this, SLOT(dataChanged(const QString&)));
	form->addRow(tr("Write timeout"), writeEdit);

	connEdit = new QSpinBox(this);
	connEdit->setRange(SPECIAL_VALUE, MAX_TIMEOUT);
	connEdit->setSpecialValueText(tr("default"));
	connEdit->setValue(connEdit->minimum());
	connect(connEdit, SIGNAL(valueChanged(const QString&)),
		this, SLOT(dataChanged(const QString&)));
	form->addRow(tr("Connect timeout"), connEdit);

	flagsForm = new SqlConfigFlags(this);
	connect(flagsForm, SIGNAL(changed()), SIGNAL(changed()));
	connect(flagsForm, SIGNAL(askClicked(bool)),
		this, SLOT(askChanged(bool)));
	layout->addWidget(flagsForm);

	sslForm = new SqlConfigSSL(this);
	connect(sslForm, SIGNAL(changed()), SIGNAL(changed()));
	layout->addWidget(sslForm);

	connect(flagsForm, SIGNAL(sslClicked(bool)),
		this, SLOT(dataChanged(bool)));
	sslForm->setEnabled(flagsForm->sslIsChecked());
	askChanged(flagsForm->askIsChecked());
}

SqlConfigWidget::~SqlConfigWidget()
{

	delete sslForm;
	delete flagsForm;
	delete connEdit;
	delete writeEdit;
	delete readEdit;
	delete portEdit;
	delete passEdit;
	delete userEdit;
	delete hostEdit;
	delete form;
	delete layout;
}


bool
SqlConfigWidget::canAccept()const
{
	bool	ret = !hostEdit->text().isEmpty();

	return (ret);
}


void
SqlConfigWidget::dataChanged(const QString&)
{

	emit changed();
}


void
SqlConfigWidget::dataChanged(bool ena)
{

	sslForm->setEnabled(ena);
	emit changed();
}


void
SqlConfigWidget::askChanged(bool ena)
{

	if (ena) {
		userEdit->clear();
		passEdit->clear();
	}
	userEdit->setDisabled(ena);
	passEdit->setDisabled(ena);
	emit changed();
}


void
SqlConfigWidget::config(SqlConfig &cfg)
{

	cfg.setHost(hostEdit->text());
	cfg.setUser(userEdit->text());
	cfg.setPassword(passEdit->text());
	cfg.setPort(portEdit->value());
	flagsForm->flags(cfg);
	if (cfg.isSet(SQL_SSL))
		sslForm->ssl(cfg);
}


void
SqlConfigWidget::setConfig(const SqlConfig &cfg)
{
	int	port;
	bool	ena = cfg.isSet(SQL_SSL);

	hostEdit->setText(cfg.host());
	userEdit->setText(cfg.user());
	passEdit->setText(cfg.password());
	port = cfg.port();
	if (port == SqlConnection::defaultPort())
		port = SPECIAL_VALUE;
	portEdit->setValue(port);
	flagsForm->setFlags(cfg);
	askChanged(flagsForm->askIsChecked());
	sslForm->setEnabled(ena);
	if (ena)
		sslForm->setSSL(cfg);
}


bool
SqlConfigWidget::hasUser()const
{
	bool	ret = flagsForm->askIsChecked();

	if (ret != TRUE)
		ret = !userEdit->text().isEmpty();

	return (ret);
}


bool
SqlConfigWidget::hasPassword()const
{
	bool	ret = flagsForm->askIsChecked();

	if (ret != TRUE)
		ret = !passEdit->text().isEmpty();

	return (ret);
}

//-----------------------------------------------------------------------------

SqlConfigFlags::SqlConfigFlags(QWidget *parent)
	:QGroupBox(tr("Connection options"), parent)
{

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	reconnect = new QCheckBox(tr("Reconnect if the connection lost"),
		this);
	connect(reconnect, SIGNAL(clicked()), SIGNAL(changed()));
	layout->addWidget(reconnect);

	compress = new QCheckBox(tr("Use compressed protocol"), this);
	connect(compress, SIGNAL(clicked()), SIGNAL(changed()));
	layout->addWidget(compress);

	ssl = new QCheckBox(tr("Use security socket layer (SSL)"), this);
	connect(ssl, SIGNAL(clicked(bool)), SIGNAL(sslClicked(bool)));
	connect(ssl, SIGNAL(clicked()), SIGNAL(changed()));
	layout->addWidget(ssl);

	alogin = new QCheckBox(tr("Auto login at startup time"), this);
	connect(alogin, SIGNAL(clicked()), SIGNAL(changed()));
	layout->addWidget(alogin);

	askLogin = new QCheckBox(tr("Ask account properties at start time"),
		this);
	askLogin->setEnabled(FALSE);
	connect(askLogin, SIGNAL(clicked(bool)), SIGNAL(askClicked(bool)));
	connect(askLogin, SIGNAL(clicked()), SIGNAL(changed()));
	layout->addWidget(askLogin);
}

SqlConfigFlags::~SqlConfigFlags()
{

	delete askLogin;
	delete alogin;
	delete ssl;
	delete compress;
	delete reconnect;
	delete layout;
}


bool
SqlConfigFlags::sslIsChecked()const
{

	return (ssl->isChecked());
}


bool
SqlConfigFlags::askIsChecked()const
{

	return (askLogin->isChecked());
}


void
SqlConfigFlags::flags(SqlConfig &cfg)
{

	cfg.setFlag(SQL_RECONNECT, reconnect->isChecked());
	cfg.setFlag(SQL_COMPRESS, compress->isChecked());
	cfg.setFlag(SQL_SSL, ssl->isChecked());
	cfg.setFlag(SQL_AUTO_LOGIN, alogin->isChecked());
	cfg.setFlag(SQL_ASK_ACCOUNT, askLogin->isChecked());
}


void
SqlConfigFlags::setFlags(const SqlConfig &cfg)
{

	reconnect->setChecked(cfg.isSet(SQL_RECONNECT));
	compress->setChecked(cfg.isSet(SQL_COMPRESS));
	ssl->setChecked(cfg.isSet(SQL_SSL));
	alogin->setChecked(cfg.isSet(SQL_AUTO_LOGIN));
	askLogin->setChecked(cfg.isSet(SQL_ASK_ACCOUNT));
	askLogin->setChecked(FALSE);	/* dummy in this version */
}

//-----------------------------------------------------------------------------

SqlConfigSSL::SqlConfigSSL(QWidget *parent)
	:QGroupBox(tr("SSL options"), parent)
{

	layout = new QFormLayout(this);
	MARGINS_ZERO(layout);

	sslKey = new FileChooser(tr("Select a key file"), 0, this);
	connect(sslKey, SIGNAL(changed()), SIGNAL(changed()));
	layout->addRow(tr("key file"), sslKey);

	sslCert = new FileChooser(tr("Select a certificate file"), 0, this);
	connect(sslCert, SIGNAL(changed()), SIGNAL(changed()));
	layout->addRow(tr("Certificate file"), sslCert);

	sslCA = new FileChooser(tr("Select a certificate authority file"), 0,
		this);
	connect(sslCA, SIGNAL(changed()), SIGNAL(changed()));
	layout->addRow(tr("Certificate authority file"), sslCA);

	sslCaPath = new FileChooser(tr("Select a directory with trusted SSL "
		"CA certificates"), QFileDialog::ShowDirsOnly, this);
	connect(sslCaPath, SIGNAL(changed()), SIGNAL(changed()));
	layout->addRow(tr("Directory that contains trusted SSL CA "
		"certificates in PEM format"), sslCaPath);

	sslCipher = new QLineEdit(this);
	connect(sslCipher, SIGNAL(textChanged(const QString&)),
		this, SLOT(dataChanged(const QString&)));
	layout->addRow(tr("Permissible ciphers to use for SSL encryption"),
		sslCipher);
}

SqlConfigSSL::~SqlConfigSSL()
{

	delete sslCipher;
	delete sslCaPath;
	delete sslCA;
	delete sslCert;
	delete sslKey;
	delete layout;
}


void
SqlConfigSSL::ssl(SqlConfig &cfg)
{

	cfg.setSSL(sslKey->text(), sslCert->text(), sslCA->text(),
		sslCaPath->text(), sslCipher->text());
}


void
SqlConfigSSL::setSSL(const SqlConfig &cfg)
{

	sslKey->setText(cfg.key());
	sslCert->setText(cfg.cert());
	sslCA->setText(cfg.ca());
	sslCaPath->setText(cfg.caPath());
	sslCipher->setText(cfg.cipher());
}


void
SqlConfigSSL::dataChanged(const QString&)
{

	emit changed();
}

//-----------------------------------------------------------------------------

FileChooser::FileChooser(const QString &ttl, QFileDialog::Options o,
	QWidget *parent)
	:QWidget(parent), title(ttl), opt(o)
{
	QFont	fnt(font());

	layout = new QHBoxLayout(this);
	MARGINS_ZERO(layout);

	labFile = new QLabel(this);
	fnt.setBold(TRUE);
	fnt.setItalic(TRUE);
	labFile->setFont(fnt);
	layout->addWidget(labFile, 1);

	chooser = new QPushButton(tr("Choose"), this);
	connect(chooser, SIGNAL(clicked()), this, SLOT(chooseFile()));
	layout->addWidget(chooser, 0, Qt::AlignRight);
}

FileChooser::~FileChooser()
{

	delete chooser;
	delete labFile;
	delete layout;
}


void
FileChooser::chooseFile()
{
	QString	file;

	if (opt == 0)
		file = QFileDialog::getOpenFileName(this, title, QString::null,
			QString::null, NULL, QFILEDIALOG_READONLY);
	else
		file = QFileDialog::getExistingDirectory(this, title,
			QString::null, opt | QFILEDIALOG_READONLY);

	if (!file.isEmpty()) {
		labFile->setText(file);
		emit changed();
	}
}


void
FileChooser::setText(const QString &txt)
{

	labFile->setText(txt);
}


QString
FileChooser::text()const
{

	return (labFile->text());
}

/* EOF */
