/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: sql_connection.cpp,v 1.16 2013/09/19 12:40:38 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "sql_connection.h"
#include "sql_config.h"
#include "sql_error.h"
#include "sql_privileges_flags.h"
#include "sql_connection_private.h"


#define	DEFAULT_HOST	"localhost"
#define	DEFAULT_USER	"user"

#define	PRIV_QUERY(priv, flags, flag, tflag)	do {			\
	if ((flags).isSet(flag)) {					\
		if (!(priv).isEmpty())					\
			(priv) += ", ";					\
		(priv) += tflag;					\
	}								\
} while (0)

#define	PRIV_BASIC(priv, flags)	do {					\
	PRIV_QUERY(priv, flags, PRIV_SELECT, "SELECT");			\
	PRIV_QUERY(priv, flags, PRIV_INSERT, "INSERT");			\
	PRIV_QUERY(priv, flags, PRIV_UPDATE, "UPDATE");			\
	PRIV_QUERY(priv, flags, PRIV_DELETE, "DELETE");			\
	PRIV_QUERY(priv, flags, PRIV_CREATE, "CREATE");			\
	PRIV_QUERY(priv, flags, PRIV_DROP, "DROP");			\
	PRIV_QUERY(priv, flags, PRIV_RELOAD, "RELOAD");			\
	PRIV_QUERY(priv, flags, PRIV_SHUTDOWN, "SHUTDOWN");		\
	PRIV_QUERY(priv, flags, PRIV_PROCESS, "PROCESS");		\
	PRIV_QUERY(priv, flags, PRIV_FILE, "FILE");			\
	PRIV_QUERY(priv, flags, PRIV_REFERENCES, "REFERENCES");		\
	PRIV_QUERY(priv, flags, PRIV_INDEX, "INDEX");			\
	PRIV_QUERY(priv, flags, PRIV_ALTER, "ALTER");			\
	PRIV_QUERY(priv, flags, PRIV_SHOW_DB, "SHOW DATABASES");	\
	PRIV_QUERY(priv, flags, PRIV_SUPER, "SUPER");			\
	PRIV_QUERY(priv, flags, PRIV_CREATE_TMP_TABLE,			\
		"CREATE TEMPORARY TABLES");				\
	PRIV_QUERY(priv, flags, PRIV_LOCK_TABLES, "LOCK TABLES");	\
	PRIV_QUERY(priv, flags, PRIV_EXECUTE, "EXECUTE");		\
	PRIV_QUERY(priv, flags, PRIV_REPL_SLAVE, "REPLICATION SLAVE");	\
	PRIV_QUERY(priv, flags, PRIV_REPL_CLIENT, "REPLICATION CLIENT");\
	PRIV_QUERY(priv, flags, PRIV_CREATE_VIEW, "CREATE VIEW");	\
	PRIV_QUERY(priv, flags, PRIV_SHOW_VIEW, "SHOW VIEW");		\
	PRIV_QUERY(priv, flags, PRIV_CREATE_ROUTINE, "CREATE ROUTINE");	\
	PRIV_QUERY(priv, flags, PRIV_ALTER_ROUTINE, "ALTER ROUTINE");	\
	PRIV_QUERY(priv, flags, PRIV_CREATE_USER, "CREATE USER");	\
	PRIV_QUERY(priv, flags, PRIV_EVENT, "EVENT");			\
	PRIV_QUERY(priv, flags, PRIV_TRIGGER, "TRIGGER");		\
	PRIV_QUERY(priv, flags, PRIV_CREATE_TABLESPACE,			\
		"CREATE TABLESPACE");					\
} while (0)

#define	PRIV_COLUMN(priv, flags, column)	do {			\
	PRIV_QUERY(priv, flags, PRIV_SELECT,				\
		QString("SELECT(%1)").arg(column));			\
	PRIV_QUERY(priv, flags, PRIV_INSERT,				\
		QString("INSERT(%1)").arg(column));			\
	PRIV_QUERY(priv, flags, PRIV_UPDATE,				\
		QString("UPDATE(%1)").arg(column));			\
	PRIV_QUERY(priv, flags, PRIV_REFERENCES,			\
		QString("REFERENCES(%1)").arg(column));			\
} while (0)

typedef	struct {
	QString	object;
	QString	account;
	QString	column;
} xchg_t;

static QString	grant(const SqlPrivilegesFlags&, const xchg_t *);
static QString	revoke(const SqlPrivilegesFlags&, const xchg_t *);


SqlConnection::SqlConnection()
{

	thr = new QThread();

	priv = new SqlConnection_Private(this);
	priv->moveToThread(thr);
}

SqlConnection::~SqlConnection()
{

	close();
	delete priv;
	delete thr;
}


void
SqlConnection::connect(const char *signal, QObject *target, const char *member)
{

	QObject::connect(priv, signal, target, member);
}


void
SqlConnection::connect(const SqlConfig *cfg)
{

	thr->start();
	priv->startConnection(cfg);
}


void
SqlConnection::connect()
{

	thr->start();
	priv->startConnection();
}


void
SqlConnection::close()
{

	priv->close();
	thr->quit();
	thr->wait();
}


quint64
SqlConnection::checksum(const QString &table)
{
	quint64	ret;

	if (table.isEmpty())
		ret = 0;
	else
		ret = priv->checksum(table);

	return (ret);
}


quint64
SqlConnection::checksum(GrantInfo::Type type)
{

	return (priv->checksum(type));
}


SqlError
SqlConnection::refresh(unsigned int flags)
{

	return (priv->refresh(flags));
}


SqlError
SqlConnection::shutdown()
{

	return (priv->shutdown());
}


unsigned long
SqlConnection::threadId()
{

	return (priv->threadId());
}


SqlError
SqlConnection::kill(unsigned long id)
{

	return (priv->kill(id));
}


SqlError
SqlConnection::exec(const QString &q)
{

	return (priv->exec(q.toUtf8()));
}


SqlError
SqlConnection::exec(const QStringList &list)
{
	SqlError			ret;
	QStringList::const_iterator	it;

	for (it = list.constBegin(); it != list.constEnd(); ++it) {
		ret = exec(*it);
		if (ret != 0)
			break;
	}

	return (ret);
}


QString
SqlConnection::account()const
{

	return (priv->account());
}


unsigned long
SqlConnection::serverVersion()
{

	return (priv->serverVersion());
}


/* Returns number of queries in the queue */
int
SqlConnection::size()
{

	return (priv->size());
}


SqlPrivilegesMap
SqlConnection::privileges(const GrantInfo &info)
{

	return (priv->privileges(info));
}


SqlConnection::operator SqlConnection_Private*()
{

	return (priv);
}


bool
SqlConnection::tryConnection(const SqlConfig &cfg, SqlError &err)
{
	SqlConnection_Private	p(NULL);

	return (p.tryConnection(cfg, err));
}


int
SqlConnection::defaultPort()
{

	return (MYSQL_PORT);
}


QString
SqlConnection::defaultHost()
{

	return (DEFAULT_HOST);
}


QString
SqlConnection::defaultUser()
{
	const char	*user = getenv("LOGNAME");

	if ((user == NULL) || (*user == '\0'))
		user = DEFAULT_USER;

	return (user);
}

//-----------------------------------------------------------------------------

GrantInfo::GrantInfo()
{

	type = G_GLOBAL;
	host.clear();
	user.clear();
	db.clear();
	table.clear();
	column.clear();
	proc.clear();
	func.clear();
	proxy_host.clear();
	proxy_user.clear();
}


void
GrantInfo::commit(QStringList &queries, const GrantInfo &info,
	const SqlPrivilegesFlags &was, const SqlPrivilegesFlags &cur)
{
	sql_privileges_t	p_grant(cur & ~was),
				p_revoke(was & ~cur);
	SqlUser			user;
	QString			query, pt;
	xchg_t			xchg;

	user.setUser(info.user);
	user.setHost(info.host);
	xchg.account = user.account(TRUE);

	switch (info.type) {
		case G_GLOBAL:
			xchg.object = "*.*";
			break;
		case G_DATABASE:
			xchg.object = info.db;
			break;
		case G_COLUMN:
			xchg.column = info.column;
		case G_TABLE:
			xchg.object = QString("%1.%2").arg(info.db)
				.arg(info.table);
			break;
		case G_PROC:
			if (info.proc.isEmpty() && !info.func.isEmpty()) {
				query = info.func;
				pt = "FUNCTION";
			} else if (!info.proc.isEmpty() &&
				info.func.isEmpty()) {
				query = info.proc;
				pt = "PROCEDURE";
			} else
				abort();
			xchg.object = QString("%1 %2.%3").arg(pt).arg(info.db)
				.arg(query);
			break;
		default:
			abort();
	}

	query = grant(p_grant, &xchg);
	if (!query.isEmpty())
		queries << query;
	query = revoke(p_revoke, &xchg);
	if (!query.isEmpty())
		queries << query;
}


QString
GrantInfo::systemTable(GrantInfo::Type type)
{
	QString	ret;

	switch (type) {
		case G_GLOBAL:
			ret = Core::systemTable("user");
			break;
		case G_DATABASE:
			ret = Core::systemTable("db");
			break;
		case G_TABLE:
			ret = Core::systemTable("tables_priv");
			break;
		case G_COLUMN:
			ret = Core::systemTable("columns_priv");
			break;
		case G_PROC:
			ret = Core::systemTable("procs_priv");
			break;
		case G_PROXY:
			ret = Core::systemTable("proxies_priv");
			break;
		default:
			abort();
	}

	return (ret);
}

//-----------------------------------------------------------------------------

QString
grant(const SqlPrivilegesFlags &flags, const xchg_t *xchg)
{
	QString			ret, priv(QString::null);
	sql_privileges_t	mask1 = PRIV_ACCESS_DENIED | PRIV_GRANT,
				mask = (PRIV_MASK & ~mask1);

	if ((flags.flags() & mask) == mask)
		priv = "ALL";
	else if (!xchg->column.isEmpty())
		PRIV_COLUMN(priv, flags, xchg->column);
	else
		PRIV_BASIC(priv, flags);

	if (!priv.isEmpty()) {
		ret = QString("GRANT %1 ON %2 TO %3")
			.arg(priv).arg(xchg->object).arg(xchg->account);
		if (flags.isSet(PRIV_GRANT))
			ret += " WITH GRANT OPTION";
	}

	return (ret);
}


QString
revoke(const SqlPrivilegesFlags &flags, const xchg_t *xchg)
{
	QString			ret(QString::null), priv(QString::null);
	sql_privileges_t	mask1 = PRIV_ACCESS_DENIED | PRIV_GRANT,
				mask = (PRIV_MASK & ~mask1);

	if ((flags.flags() & mask) == mask) {
		priv = "ALL";
		PRIV_QUERY(priv, flags, PRIV_GRANT, "GRANT OPTION");
	} else if (!xchg->column.isEmpty())
		PRIV_COLUMN(priv, flags, xchg->column);
	else {
		PRIV_BASIC(priv, flags);
		PRIV_QUERY(priv, flags, PRIV_GRANT, "GRANT OPTION");
	}

	if (!priv.isEmpty())
		ret = QString("REVOKE %1 ON %2 FROM %3")
			.arg(priv).arg(xchg->object).arg(xchg->account);

	return (ret);
}

/* EOF */
