/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: sql_connection_private.cpp,v 1.22 2013/11/05 12:57:08 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include <errmsg.h>

#include "sql_connection.h"
#include "sql_connection_private.h"
#include "sql_query_private.h"


#define	IF_DATA(array)	(((array).size() == 0)?NULL:(array).constData())
#define	ST_CLOSING	0x01

#define	SET_PRIV_FLAG(priv, access, flag)	do {			\
	if ((access) == 'Y')						\
		(priv) += (flag);					\
} while (0)


static const char	*default_charset = "utf8";


SqlConnection_Private::SqlConnection_Private(SqlConnection *ptr,
	QObject *parent)
	:QObject(parent), conn(ptr), mysql(NULL), status(0), next_id(0),
	my_acc(QString::null)
{

	queue = new SqlQueue();

	timer = new QTimer(this);
	timer->start(0);
	connect(timer, SIGNAL(timeout()), this, SLOT(readQueue()));
}

SqlConnection_Private::~SqlConnection_Private()
{

	delete timer;
	delete queue;
	close();
}


void
SqlConnection_Private::close()
{
	SqlQueryMap::iterator	it;
	SqlQuery_Private	*ptr;
	bool			emt = FALSE;

	for (it = map.begin(); it != map.end(); ++it) {
		ptr = *it;
		ptr->setReprepare();
	}

	mtx_conn.lock();
	if (mysql != NULL) {
		mysql_close(mysql);
		mysql = NULL;
		emt = TRUE;
	}
	mtx_conn.unlock();

	if (emt)
		emit closed();
}


/* Called from the main thread, use mutexes, and copy data! */
void
SqlConnection_Private::startConnection(const SqlConfig *cfg)
{

	mtx_cfg.lock();
	config.setConfig(*cfg);
	mtx_cfg.unlock();

	startConnection();
}


/* Called from the main thread to reconnect */
void
SqlConnection_Private::startConnection()
{

	QTimer::singleShot(0, this, SLOT(doConnect()));
}


void
SqlConnection_Private::removeQuery(unsigned int id)
{
	SqlQuery_Private	*query = map.value(id, NULL);

	if (query != NULL) {
/*		disconnect(this, SIGNAL(closed()),
			query, SLOT(setReprepare()));*/
		map.remove(id);
		next_id = id;
	}
}


void
SqlConnection_Private::exec(SqlQuery_Private *query)
{

	queue->insert(query);
}


SqlError
SqlConnection_Private::exec(MYSQL_STMT *stmt)
{
	int		result;
	SqlError	err;

	mtx_conn.lock();
	result = mysql_stmt_execute(stmt);
	if (result == 0)
		result = mysql_stmt_store_result(stmt);
	if (result != 0)
		getError(err, stmt);
	mtx_conn.unlock();
	errorActions(err);

	return (err);
}


SqlError
SqlConnection_Private::exec(const QByteArray &q)
{
	int		result;
	SqlError	err;
	MYSQL_RES	*res = NULL;

	mtx_conn.lock();
	if (mysql != NULL) {
		result = mysql_real_query(mysql, q.constData(), q.size());
		if (result == 0)
			res = mysql_store_result(mysql);
		else
			getError(err, mysql);
	}
	mtx_conn.unlock();
	if (res != NULL)
		mysql_free_result(res);

	return (err);
}


void
SqlConnection_Private::close(MYSQL_STMT *stmt)
{

	/* FIXME: dead lock, why? */
	mtx_conn.lock();
	mysql_stmt_close(stmt);
	mtx_conn.unlock();
}


void
SqlConnection_Private::readQueue()
{
	SqlQuery_Private	*query;

	query = queue->query();
	if (query != NULL)
		query->execute();
}


/* Establish a new connection to a MySQL server */
void
SqlConnection_Private::doConnect()
{
	SqlConfig	cfg;
	SqlError	err;
	MYSQL		*tmp;

	mtx_cfg.lock();
	cfg.setConfig(config);
	mtx_cfg.unlock();
	close();

	tmp = createConnection(cfg, err);
	if (tmp == NULL)
		emit error(err);
	else {
		mtx_conn.lock();
		mysql = tmp;
		mtx_conn.unlock();
		emit connected();
	}
}

#if	0
void
SqlConnection_Private::doClose()
{
	bool	emt;

	mtx_conn.lock();
	emt = (mysql != NULL);
	if (emt == TRUE) {
		mysql_close(mysql);
		mysql = NULL;
	}
	mtx_conn.unlock();

	if (emt == TRUE)
		connectionClosed();

	mtx_stat.lock();
	status &= ~ST_CLOSING;
	cnd_stat.wakeOne();
	mtx_stat.unlock();
}
#endif

MYSQL *
SqlConnection_Private::createConnection(const SqlConfig &cfg, SqlError &err)
{
	MYSQL		*init, *ret = NULL;
	MYSQL_RES	*res;
	MYSQL_ROW	row;
	bool		empty = TRUE;

	init = mysql_init(NULL);
	if (init != NULL) {
		setOptions(init, cfg);
		ret = mysql_real_connect(init, cfg.host().toUtf8().constData(),
			cfg.user().toUtf8().constData(),
			cfg.password().toUtf8().constData(),
			NULL, cfg.port(), NULL, 0);
		if (ret == NULL) {
			getError(err, init);
			mysql_close(init);
		} else if (mysql_query(ret, "select user()") == 0) {
			res = mysql_store_result(ret);
			if (res != NULL) {
				if ((row = mysql_fetch_row(res)) != NULL) {
					mtx_cfg.lock();
					my_acc = row[0];
					mtx_cfg.unlock();
				}
				mysql_free_result(res);
			}
			mtx_cfg.lock();
			empty = my_acc.isEmpty();
			mtx_cfg.unlock();
			if (empty)
				goto HAXEP;
		} else {
			/* Wow! we did not read the account! */
			HAXEP:
			mtx_cfg.lock();
			my_acc = "unknown@unknown";
			mtx_cfg.unlock();
		}
	}

	return (ret);
}


unsigned int
SqlConnection_Private::createId()
{
	unsigned int	ret;

	for (ret = next_id; map.contains(ret); ++ret)
		;

	next_id = ret + 1;

	return (ret);
}


bool
SqlConnection_Private::tryConnection(const SqlConfig &cfg, SqlError &err)
{
	MYSQL	*init, *sql;
	bool	ret;

	init = mysql_init(NULL);
	if (init == NULL)
		ret = FALSE;
	else {
		setOptions(init, cfg);
		sql = mysql_real_connect(init, cfg.host().toUtf8().constData(),
			cfg.user().toUtf8().constData(),
			cfg.password().toUtf8().constData(),
			NULL, cfg.port(), NULL, 0);
		if (sql == NULL) {
			getError(err, init);
			mysql_close(init);
			ret = FALSE;
		} else {
			err.clear();
			mysql_close(sql);
			ret = TRUE;
		}
	}

	return (ret);
}


unsigned int
SqlConnection_Private::insertQuery(SqlQuery_Private *query)
{
	unsigned int	id = createId();

	map[id] = query;

	return (id);
}


MYSQL_STMT *
SqlConnection_Private::createStmt()
{
	MYSQL_STMT	*ret;

	mtx_conn.lock();
	if (mysql != NULL)
		ret = mysql_stmt_init(mysql);
	else
		ret = NULL;
	mtx_conn.unlock();

	return (ret);
}


int
SqlConnection_Private::prepare(MYSQL_STMT *stmt, const QByteArray &query)
{
	int		ret;
	SqlError	err;

	mtx_conn.lock();
	ret = mysql_stmt_prepare(stmt, query.constData(), query.size());
	if (ret != 0) {
		getError(err, stmt);
		cerr<<err<<endl;
	}
	mtx_conn.unlock();

	return (ret);
}


quint64
SqlConnection_Private::checksum(const QString &table)
{
	quint64	ret = 0;

	mtx_conn.lock();
	if (mysql != NULL) {
		QByteArray	query(QString("checksum table %1")
					.arg(table).toUtf8());
		const char	*qdata = query.constData();
		unsigned long	qsize = query.size();
		MYSQL_RES	*res;
		MYSQL_ROW	row;

		if (mysql_real_query(mysql, qdata, qsize) == 0)
			res = mysql_store_result(mysql);
		else
			res = NULL;
		mtx_conn.unlock();

		if (res != NULL) {
			row = mysql_fetch_row(res);
			if ((row != NULL) && (row[1] != NULL))
				ret = strtoull(row[1], NULL, 10);
			mysql_free_result(res);
		}
	} else
		mtx_conn.unlock();

	return (ret);
}


/* Returns checksum for the specified privileges table */
quint64
SqlConnection_Private::checksum(GrantInfo::Type type)
{

	return (checksum(GrantInfo::systemTable(type)));
}


SqlError
SqlConnection_Private::refresh(unsigned int flags)
{
	SqlError	ret;

	mtx_conn.lock();
	if (mysql_refresh(mysql, flags) != 0)
		getError(ret, mysql);
	mtx_conn.unlock();

	return (ret);
}


SqlError
SqlConnection_Private::shutdown()
{
	SqlError	ret;

	mtx_conn.lock();
	if (mysql_shutdown(mysql, SHUTDOWN_DEFAULT) != 0)
		getError(ret, mysql);
	mtx_conn.unlock();

	return (ret);
}


unsigned long
SqlConnection_Private::threadId()
{
	unsigned long	ret;
	bool		ena;

	mtx_conn.lock();
	ena = (mysql != NULL);
	if (ena) {
		ret = mysql_thread_id(mysql);
		mtx_conn.unlock();
	} else {
		mtx_conn.unlock();
		ret = 0;
	}

	return (ret);
}


SqlConnection *
SqlConnection_Private::connection()const
{

	return (conn);
}


SqlError
SqlConnection_Private::kill(unsigned long id)
{
	SqlError	ret;

	mtx_conn.lock();
	if (mysql != NULL) {
		if (mysql_kill(mysql, id) != 0)
			getError(ret, mysql);
	}
	mtx_conn.unlock();

	return (ret);
}


QString
SqlConnection_Private::account()
{
	QString	ret;

	mtx_cfg.lock();
	ret = my_acc;
	mtx_cfg.unlock();

	return (ret);
}


unsigned long
SqlConnection_Private::serverVersion()
{
	unsigned long	ret = 0;

	mtx_conn.lock();
	if (mysql != NULL)
		ret = mysql_get_server_version(mysql);
	mtx_conn.unlock();

	return (ret);
}


/* Returns number of queries in the queue */
int
SqlConnection_Private::size()
{

	return (queue->size());
}


SqlPrivilegesMap
SqlConnection_Private::privileges(const GrantInfo &info)
{
	SqlPrivilegesMap	ret;

	switch (info.type) {
		case GrantInfo::G_GLOBAL:
			loadGlobalPrivileges(ret, info);
			break;
		case GrantInfo::G_DATABASE:
			loadDatabasePrivileges(ret, info);
			break;
		case GrantInfo::G_TABLE:
			loadTablePrivileges(ret, info);
			break;
		case GrantInfo::G_COLUMN:
			loadColumnPrivileges(ret, info);
			break;
		case GrantInfo::G_PROC:
			loadProcPrivileges(ret, info);
			break;
		default:
			abort();
	}

	return (ret);
}


void
SqlConnection_Private::setOptions(MYSQL *init, const SqlConfig &cfg)
{
	my_bool		reconnect = TRUE,
			truncate = TRUE;
	unsigned int	rtimeout, wtimeout, ctimeout;
	int		timeout;

	if (cfg.isSet(SQL_RECONNECT)) {
		/* reconnect if the connection lost */
		mysql_options(init, MYSQL_OPT_RECONNECT, &reconnect);
	}

	if (cfg.isSet(SQL_COMPRESS)) {
		/* use compressed protocol */
		mysql_options(init, MYSQL_OPT_COMPRESS, NULL);
	}

	if (cfg.isSet(SQL_SSL)) {
		QByteArray	key(cfg.key().toUtf8()),
				cert(cfg.cert().toUtf8()),
				ca(cfg.ca().toUtf8()),
				capath(cfg.caPath().toUtf8()),
				cipher(cfg.cipher().toUtf8());
		const char	*key_ptr = IF_DATA(key),
				*cert_ptr = IF_DATA(cert),
				*ca_ptr = IF_DATA(ca),
				*capath_ptr = IF_DATA(capath),
				*cipher_ptr = IF_DATA(cipher);

		mysql_ssl_set(init, key_ptr, cert_ptr, ca_ptr,
			capath_ptr, cipher_ptr);
	}

	mysql_options(init, MYSQL_SET_CHARSET_NAME, default_charset);
	mysql_options(init, MYSQL_REPORT_DATA_TRUNCATION, &truncate);

	timeout = cfg.readTimeout();
	if (timeout > 0) {
		rtimeout = timeout;
		mysql_options(init, MYSQL_OPT_READ_TIMEOUT, &rtimeout);
	}

	timeout = cfg.writeTimeout();
	if (timeout > 0) {
		wtimeout = timeout;
		mysql_options(init, MYSQL_OPT_WRITE_TIMEOUT, &wtimeout);
	}

	timeout = cfg.connectTimeout();
	if (timeout > 0) {
		ctimeout = timeout;
		mysql_options(init, MYSQL_OPT_CONNECT_TIMEOUT, &ctimeout);
	}
}


void
SqlConnection_Private::getError(SqlError &err, MYSQL *init)
{

	err.setError(mysql_errno(init), mysql_error(init));
}


void
SqlConnection_Private::getError(SqlError &err, MYSQL_STMT *stmt)
{

	err.setError(mysql_stmt_errno(stmt), mysql_stmt_error(stmt));
}


void
SqlConnection_Private::errorActions(const SqlError &err)
{

	if (err == CR_SERVER_LOST)
		close();
}


void
SqlConnection_Private::loadGlobalPrivileges(SqlPrivilegesMap &fmap,
	const GrantInfo &info)
{
	unsigned long		vers = serverVersion();
	QStringList		fields;
	QString			query("SELECT %1 from %2 where "
					"'%3' like Host && '%4' like User");
	QByteArray		qbin;
	int			result, i;
	MYSQL_RES		*res = NULL;
	MYSQL_ROW		row;
	SqlPrivilegesFlags	priv;

	/* TODO: split columns by the server version! */
	fields<<FLD_SELECT_PRIV
		<<FLD_INSERT_PRIV
		<<FLD_UPDATE_PRIV
		<<FLD_DELETE_PRIV
		<<FLD_CREATE_PRIV
		<<FLD_DROP_PRIV
		<<FLD_RELOAD_PRIV
		<<FLD_SHUTDOWN_PRIV
		<<FLD_PROCESS_PRIV
		<<FLD_FILE_PRIV
		<<FLD_GRANT_PRIV
		<<FLD_REFERENCES_PRIV
		<<FLD_INDEX_PRIV
		<<FLD_ALTER_PRIV
		<<FLD_SHOW_DB_PRIV
		<<FLD_SUPER_PRIV
		<<FLD_CREATE_TMP_TABLE_PRIV
		<<FLD_LOCK_TABLES_PRIV
		<<FLD_EXECUTE_PRIV
		<<FLD_REPL_SLAVE_PRIV
		<<FLD_REPL_CLIENT_PRIV
		<<FLD_CREATE_VIEW_PRIV
		<<FLD_SHOW_VIEW_PRIV
		<<FLD_CREATE_ROUTINE_PRIV
		<<FLD_ALTER_ROUTINE_PRIV
		<<FLD_CREATE_USER_PRIV
		<<FLD_EVENT_PRIV
		<<FLD_TRIGGER_PRIV;
	if (vers >= SQL_VERSION(5, 3, 0))
		fields<<FLD_CREATE_TABLESPACE_PRIV;
	qbin = query.arg(fields.join(", "))
		.arg(GrantInfo::systemTable(info.type))
		.arg(info.host).arg(info.user).toUtf8();

	mtx_conn.lock();
	if (mysql != NULL) {
		result = mysql_real_query(mysql, qbin.constData(),
			qbin.size());
		if (result == 0)
			res = mysql_store_result(mysql);
		else
			priv += PRIV_ACCESS_DENIED;
	} else
		priv += PRIV_NOT_READY;
	mtx_conn.unlock();
	if (res != NULL) {
		if ((row = mysql_fetch_row(res)) != NULL) {
			i = 0;
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_SELECT);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_INSERT);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_UPDATE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_DELETE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_CREATE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_DROP);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_RELOAD);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_SHUTDOWN);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_PROCESS);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_FILE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_GRANT);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_REFERENCES);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_INDEX);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_ALTER);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_SHOW_DB);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_SUPER);
			SET_PRIV_FLAG(priv, *(row[i++]),
				PRIV_CREATE_TMP_TABLE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_LOCK_TABLES);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_EXECUTE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_REPL_SLAVE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_REPL_CLIENT);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_CREATE_VIEW);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_SHOW_VIEW);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_CREATE_ROUTINE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_ALTER_ROUTINE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_CREATE_USER);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_EVENT);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_TRIGGER);
			SET_PRIV_FLAG(priv, *(row[i++]),
				PRIV_CREATE_TABLESPACE);
		}
		mysql_free_result(res);
	}
	fmap["*.*"] = priv;
}


void
SqlConnection_Private::loadDatabasePrivileges(SqlPrivilegesMap &fmap,
	const GrantInfo &info)
{
//	unsigned long		vers = serverVersion();
	QStringList		fields;
	QString			query("SELECT Db, %1 from %2 where "
					"Host = '%3' && User = '%4'");
	QByteArray		qbin;
	int			result, i;
	MYSQL_RES		*res = NULL;
	MYSQL_ROW		row;
	QString			level;
	SqlPrivilegesFlags	priv;

	/* TODO: split columns by the server version! */
	fields<<FLD_SELECT_PRIV
		<<FLD_INSERT_PRIV
		<<FLD_UPDATE_PRIV
		<<FLD_DELETE_PRIV
		<<FLD_CREATE_PRIV
		<<FLD_DROP_PRIV
		<<FLD_GRANT_PRIV
		<<FLD_REFERENCES_PRIV
		<<FLD_INDEX_PRIV
		<<FLD_ALTER_PRIV
		<<FLD_CREATE_TMP_TABLE_PRIV
		<<FLD_LOCK_TABLES_PRIV
		<<FLD_CREATE_VIEW_PRIV
		<<FLD_SHOW_VIEW_PRIV
		<<FLD_CREATE_ROUTINE_PRIV
		<<FLD_ALTER_ROUTINE_PRIV
		<<FLD_EXECUTE_PRIV
		<<FLD_EVENT_PRIV
		<<FLD_TRIGGER_PRIV;
	qbin = query.arg(fields.join(", "))
		.arg(GrantInfo::systemTable(info.type))
		.arg(info.host).arg(info.user).toUtf8();

	mtx_conn.lock();
	if (mysql != NULL) {
		result = mysql_real_query(mysql, qbin.constData(),
			qbin.size());
		if (result == 0)
			res = mysql_store_result(mysql);
	}
	mtx_conn.unlock();
	if (res != NULL) {
		while ((row = mysql_fetch_row(res)) != NULL) {
			i = 0; priv = PRIV_USAGE;
			level = QString("%1.*").arg(row[i++]);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_SELECT);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_INSERT);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_UPDATE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_DELETE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_CREATE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_DROP);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_GRANT);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_REFERENCES);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_INDEX);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_ALTER);
			SET_PRIV_FLAG(priv, *(row[i++]),
				PRIV_CREATE_TMP_TABLE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_LOCK_TABLES);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_CREATE_VIEW);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_SHOW_VIEW);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_CREATE_ROUTINE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_ALTER_ROUTINE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_EXECUTE);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_EVENT);
			SET_PRIV_FLAG(priv, *(row[i++]), PRIV_TRIGGER);
			fmap[level] = priv;
		}
		mysql_free_result(res);
	}
}


void
SqlConnection_Private::loadTablePrivileges(SqlPrivilegesMap &fmap,
	const GrantInfo &info)
{
//	unsigned long		vers = serverVersion();
	QString			query("SELECT Db, Table_name, Table_priv "
					"from %1 where "
					"Host = '%2' && User = '%3'");
	QByteArray		qbin;
	int			result, i;
	MYSQL_RES		*res = NULL;
	MYSQL_ROW		row;
	QString			level, db, tbl;
	SqlPrivilegesFlags	priv;

	qbin = query.arg(GrantInfo::systemTable(info.type))
		.arg(info.host).arg(info.user).toUtf8();

	mtx_conn.lock();
	if (mysql != NULL) {
		result = mysql_real_query(mysql, qbin.constData(),
			qbin.size());
		if (result == 0)
			res = mysql_store_result(mysql);
	}
	mtx_conn.unlock();
	if (res != NULL) {
		while ((row = mysql_fetch_row(res)) != NULL) {
			i = 0; priv = PRIV_USAGE;
			db = row[i++];
			tbl = row[i++];
			level = QString("%1.%2").arg(db).arg(tbl);
			parsePrivileges(priv, row[i++]);
			fmap[level] = priv;
		}
		mysql_free_result(res);
	}
}


void
SqlConnection_Private::loadColumnPrivileges(SqlPrivilegesMap &fmap,
	const GrantInfo &info)
{
//	unsigned long		vers = serverVersion();
	QString			query("SELECT Db, Table_name, Column_name, "
					"Column_priv from %1 where "
					"Host = '%2' && User = '%3'");
	QByteArray		qbin;
	int			result, i;
	MYSQL_RES		*res = NULL;
	MYSQL_ROW		row;
	QString			level, db, tbl, col;
	SqlPrivilegesFlags	priv;

	qbin = query.arg(GrantInfo::systemTable(info.type))
		.arg(info.host).arg(info.user).toUtf8();

	mtx_conn.lock();
	if (mysql != NULL) {
		result = mysql_real_query(mysql, qbin.constData(),
			qbin.size());
		if (result == 0)
			res = mysql_store_result(mysql);
	}
	mtx_conn.unlock();
	if (res != NULL) {
		while ((row = mysql_fetch_row(res)) != NULL) {
			i = 0; priv = PRIV_USAGE;
			db = row[i++];
			tbl = row[i++];
			col = row[i++];
			level = QString("%1.%2.%3").arg(db).arg(tbl).arg(col);
			parsePrivileges(priv, row[i++]);
			fmap[level] = priv;
		}
		mysql_free_result(res);
	}
}


void
SqlConnection_Private::loadProcPrivileges(SqlPrivilegesMap &fmap,
	const GrantInfo &info)
{
//	unsigned long		vers = serverVersion();
	QString			query("SELECT Db, Routine_name, Routine_type, "
					"Proc_priv from %1 where "
					"Host = '%2' && User = '%3'");
	QByteArray		qbin;
	int			result, i;
	MYSQL_RES		*res = NULL;
	MYSQL_ROW		row;
	QString			level, db, rname;
	SqlPrivilegesFlags	priv;

	qbin = query.arg(GrantInfo::systemTable(info.type))
		.arg(info.host).arg(info.user).toUtf8();

	mtx_conn.lock();
	if (mysql != NULL) {
		result = mysql_real_query(mysql, qbin.constData(),
			qbin.size());
		if (result == 0)
			res = mysql_store_result(mysql);
	}
	mtx_conn.unlock();
	if (res != NULL) {
		while ((row = mysql_fetch_row(res)) != NULL) {
			i = 0; priv = PRIV_USAGE;
			db = row[i++];
			rname = row[i++];
			priv.setProcType(procType(row[i++]));
			level = QString("%1.%2").arg(db).arg(rname);
			parsePrivileges(priv, row[i++]);
			fmap[level] = priv;
		}
		mysql_free_result(res);
	}
}


void
SqlConnection_Private::parsePrivileges(SqlPrivilegesFlags &priv,
	const QString &str)
{
	QStringList	list(str.split(',', QString::SkipEmptyParts));
	QString		tmp;
	Qt::CaseSensitivity	sens = Qt::CaseInsensitive;
	QStringList::const_iterator	it;

	for (it = list.constBegin(); it != list.constEnd(); ++it) {
		tmp = *it;
		if (tmp.compare("Select", sens) == 0)
			priv += PRIV_SELECT;
		else if (tmp.compare("Insert", sens) == 0)
			priv += PRIV_INSERT;
		else if (tmp.compare("Update", sens) == 0)
			priv += PRIV_UPDATE;
		else if (tmp.compare("Delete", sens) == 0)
			priv += PRIV_DELETE;
		else if (tmp.compare("Create", sens) == 0)
			priv += PRIV_CREATE;
		else if (tmp.compare("Drop", sens) == 0)
			priv += PRIV_DROP;
		else if (tmp.compare("Grant", sens) == 0)
			priv += PRIV_GRANT;
		else if (tmp.compare("References", sens) == 0)
			priv += PRIV_REFERENCES;
		else if (tmp.compare("Index", sens) == 0)
			priv += PRIV_INDEX;
		else if (tmp.compare("Alter", sens) == 0)
			priv += PRIV_ALTER;
		else if (tmp.compare("Create View", sens) == 0)
			priv += PRIV_CREATE_VIEW;
		else if (tmp.compare("Show View", sens) == 0)
			priv += PRIV_SHOW_VIEW;
		else if (tmp.compare("Trigger", sens) == 0)
			priv += PRIV_TRIGGER;
		else if (tmp.compare("Execute", sens) == 0)
			priv += PRIV_EXECUTE;
		else if (tmp.compare("Alter Routine", sens) == 0)
			priv += PRIV_ALTER_ROUTINE;
		else {
			/* unknown privileges for this level */
			abort();
		}
	}
}


SqlPrivilegesFlags::ProcType
SqlConnection_Private::procType(const QString &type)const
{
	SqlPrivilegesFlags::ProcType	ret;

	if (type.compare("FUNCTION", Qt::CaseInsensitive) == 0)
		ret = SqlPrivilegesFlags::P_FUNC;
	else if (type.compare("PROCEDURE", Qt::CaseInsensitive) == 0)
		ret = SqlPrivilegesFlags::P_PROC;
	else
		ret = SqlPrivilegesFlags::P_NONE;

	return (ret);
}

//-----------------------------------------------------------------------------

SqlQueue::SqlQueue()
{
}

SqlQueue::~SqlQueue()
{
}


void
SqlQueue::insert(SqlQuery_Private *query)
{

	mtx.lock();
	queue.enqueue(query);
	cnd.wakeOne();
	mtx.unlock();
}


SqlQuery_Private *
SqlQueue::query()
{
	SqlQuery_Private	*ret;

	mtx.lock();
	if (queue.isEmpty())
		cnd.wait(&mtx, CONDITION_TIMEOUT);
	if (queue.isEmpty())
		ret = NULL;
	else
		ret = queue.dequeue();
	mtx.unlock();

	return (ret);
}


int
SqlQueue::size()
{
	int	ret;

	mtx.lock();
	ret = queue.size();
	mtx.unlock();

	return (ret);
}

/* EOF */
