/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: sql_query_private.cpp,v 1.7 2013/08/01 14:01:08 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "sql_query_private.h"
#include "sql_connection_private.h"

#define	S_PREPARED	0x01
#define	S_QUEUED	0x02

#define	IS_PREPARED	(p_status & S_PREPARED)
#define	IS_QUEUED	(p_status & S_QUEUED)

#ifdef	QMYADMIN_DEBUG
#define	QUERY_DEBUG()	do {						\
	cout<<__FUNCTION__						\
		<<": this = "<<this					\
		<<"; id = "<<p_id					\
		<<"; status = 0x"<<hex<<status()			\
		<<endl;							\
} while (0)
#else	/* QMYADMIN_DEBUG */
#define	QUERY_DEBUG()
#endif	/* QMYADMIN_DEBUG */


class SqlValue;

typedef	union {
	void		*basic;		/* basic pointer to zero */
	QByteArray	*str;		/* string data */
	long long int	ll_val;
	long int	l_val;
} my_union;
typedef	QVector<MYSQL_BIND>		BindVector;
typedef	QVector<SqlValue *>		ValueVector;
typedef QMap<QString, SqlValue *>	ValueMap;

//-----------------------------------------------------------------------------

class SqlValue
{
public:
	SqlValue(MYSQL_FIELD *, MYSQL_BIND *);
	~SqlValue();

	my_bool		error()const;
	int		fetch(MYSQL_STMT *, unsigned int);
	QVariant	value()const;

private:
	enum enum_field_types	p_type;
	my_union		data;
	unsigned int		p_flags;
	my_bool			p_null;
	my_bool			p_error;
	unsigned long		p_length;
	MYSQL_BIND		*p_bind;

	void	initData(MYSQL_FIELD *);
	void	destroyData();
	void	bindString();
	void	realloc();
	void	reallocString();
};

//-----------------------------------------------------------------------------

class SqlResult
{
public:
	SqlResult(MYSQL_RES *);
	~SqlResult();

	void	bind(unsigned long, QString&);

	int		fetch(MYSQL_STMT *);
	QVariant	value(unsigned long)const;
	QVariant	value(const QString&)const;

	operator MYSQL_BIND*();

private:
	BindVector	*vect;
	ValueVector	*val;
	ValueMap	*map;
};

//-----------------------------------------------------------------------------

SqlQuery_Private::SqlQuery_Private(SqlConnection_Private *priv)
	:QObject(NULL), conn(priv), stmt(NULL), p_result(NULL), p_status(0)
{

	p_id = conn->insertQuery(this);
}

SqlQuery_Private::~SqlQuery_Private()
{

	conn->removeQuery(p_id);
	close();
}


void
SqlQuery_Private::prepare(const QString &q)
{
	bool	p;

	mtx.lock();
	p = IS_PREPARED;
	mtx.unlock();
	if (p != TRUE) {
		p_query = q;
		reprepare();
	}
}


void
SqlQuery_Private::exec()
{
	bool	q;

	mtx.lock();
	if ((q = IS_QUEUED) != TRUE)
		p_status |= S_QUEUED;
	mtx.unlock();
	if (q != TRUE)
		conn->exec(this);
}


void
SqlQuery_Private::execute()
{
	SqlError	err;
	bool		p, q;

	mtx.lock();
	p = IS_PREPARED;
	if ((q = IS_QUEUED) == TRUE)
		p_status &= ~S_QUEUED;
	mtx.unlock();
	if (p != TRUE)
		reprepare();
	
	if ((stmt != NULL) && q) {
		err = conn->exec(stmt);
		if (err == 0)
			emit ready();
	}
}


void
SqlQuery_Private::close()
{

	if (stmt != NULL) {
		setReprepare();
		p_query.clear();
		SAFE_DELETE(p_result);
	}
}


void
SqlQuery_Private::free()
{

	if (stmt != NULL)
		mysql_stmt_free_result(stmt);
}


void
SqlQuery_Private::bind(unsigned long pos, QString &str)
{

	p_result->bind(pos, str);
}


void
SqlQuery_Private::setReprepare()
{

	if (stmt != NULL) {
		conn->close(stmt);
		stmt = NULL;
	}
	mtx.lock();
	p_status &= ~S_PREPARED;
	mtx.unlock();
}


void
SqlQuery_Private::reprepare()
{
	int	result;

	close();
	if (stmt == NULL)
		stmt = conn->createStmt();
	result = (stmt != NULL);

	if (result != 0) {
		result = conn->prepare(stmt, p_query.toUtf8());
		if (result == 0) {
			mtx.lock();
			p_status |= S_PREPARED;
			mtx.unlock();
			bindArray();
		}
	}
}


bool
SqlQuery_Private::next()
{
	bool	ret;

	if (stmt != NULL)
		ret = (fetch() == 0);
	else
		ret = FALSE;

	return (ret);
}


QVariant
SqlQuery_Private::value(unsigned long pos)const
{
	QVariant	ret;

	if (p_result != NULL)
		ret = p_result->value(pos);

	return (ret);
}


QVariant
SqlQuery_Private::value(const QString &name)const
{
	QVariant	ret;

	if (p_result != NULL)
		ret = p_result->value(name);

	return (ret);
}


SqlConnection *
SqlQuery_Private::connection()const
{

	return (conn->connection());
}

void
SqlQuery_Private::bindArray()
{
	MYSQL_RES	*result;

	if (p_result == NULL) {
		result = mysql_stmt_result_metadata(stmt);
		FATAL(result == NULL);
		p_result = new SqlResult(result);
		mysql_free_result(result);
	}

	if ((stmt != NULL) && (mysql_stmt_bind_result(stmt, *p_result) != 0))
		abort();
}


int
SqlQuery_Private::fetch()
{
	int	ret;

	ret = mysql_stmt_fetch(stmt);
	if (ret == MYSQL_DATA_TRUNCATED) {
		ret = p_result->fetch(stmt);
		bindArray();
	}

	return (ret);
}


int
SqlQuery_Private::status()
{
	int	ret;

	mtx.lock();
	ret = p_status;
	mtx.unlock();

	return (ret);
}

//-----------------------------------------------------------------------------

SqlValue::SqlValue(MYSQL_FIELD *field, MYSQL_BIND *bind)
{

	p_type = field->type;
	p_flags = field->flags;
	p_length = 0;
	p_error = 0;
	p_null = 0;
	p_bind = bind;
	p_bind->error = &p_error;
	p_bind->is_null = &p_null;
	p_bind->length = &p_length;
	p_bind->buffer_type = p_type;

	data.basic = NULL;
	initData(field);
}

SqlValue::~SqlValue()
{

	destroyData();
}


void
SqlValue::initData(MYSQL_FIELD *)
{

	switch (p_type) {
		case MYSQL_TYPE_VAR_STRING:
		case MYSQL_TYPE_STRING:
		case MYSQL_TYPE_BLOB:
			bindString();
			break;
		case MYSQL_TYPE_LONGLONG:
			p_bind->buffer = &data.ll_val;
			break;
		case MYSQL_TYPE_LONG:
			p_bind->buffer = &data.l_val;
			break;
		default:
			abort();
	}
}


void
SqlValue::destroyData()
{

	switch (p_type) {
		case MYSQL_TYPE_VAR_STRING:
		case MYSQL_TYPE_STRING:
		case MYSQL_TYPE_BLOB:
			SAFE_DELETE(data.str);
			break;
		case MYSQL_TYPE_LONGLONG:
		case MYSQL_TYPE_LONG:
			/* no data to destroy */
			break;
		default:
			abort();
	}
}


void
SqlValue::bindString()
{

	p_bind->buffer = NULL;
}


void
SqlValue::realloc()
{

	switch (p_type) {
		case MYSQL_TYPE_VAR_STRING:
		case MYSQL_TYPE_STRING:
		case MYSQL_TYPE_BLOB:
			reallocString();
			break;
		default:
			abort();
	}
}


void
SqlValue::reallocString()
{

	if (data.str == NULL)
		data.str = new QByteArray();

	if (p_length > static_cast<unsigned long>(data.str->size())) {
		data.str->resize(p_length);
		p_bind->buffer = data.str->data();
		p_bind->buffer_length = data.str->size();
	}
}


my_bool
SqlValue::error()const
{

	return (p_error);
}


int
SqlValue::fetch(MYSQL_STMT *stmt, unsigned int pos)
{
	int	ret;

	realloc();

	p_error = 0;
	ret = mysql_stmt_fetch_column(stmt, p_bind, pos, 0);

	return (ret);
}


QVariant
SqlValue::value()const
{
	QVariant	ret;

	switch (p_type) {
		case MYSQL_TYPE_VAR_STRING:
		case MYSQL_TYPE_STRING:
		case MYSQL_TYPE_BLOB:
			if ((data.str != NULL) && (data.str->size() > 0) &&
				(p_null == 0))
				ret = QString(data.str->left(p_length));
			break;
		case MYSQL_TYPE_LONGLONG:
			ret = data.ll_val;
			break;
		case MYSQL_TYPE_LONG:
			ret = (qlonglong)data.l_val;
			break;
		default:
			abort();
	}

	return (ret);
}

//-----------------------------------------------------------------------------

SqlResult::SqlResult(MYSQL_RES *result)
{
	unsigned long	i, fields = mysql_num_fields(result);
	MYSQL_FIELD	*fld = mysql_fetch_fields(result);
	MYSQL_BIND	*bind;
	SqlValue	*value;

	vect = new BindVector(fields);
	bind = vect->data();
	memset(bind, 0, vect->size() * sizeof(*bind));

	val = new ValueVector();
	map = new ValueMap();

	for (i = 0; i < fields; ++i, ++fld, ++bind) {
		value = new SqlValue(fld, bind);
		val->append(value);
		map->insert(QString(fld->name).toLower(), value);
	}
}

SqlResult::~SqlResult()
{
	ValueVector::iterator	it;

	for (it = val->begin(); it != val->end(); ++it)
		delete (*it);

	delete map;
	delete val;
	delete vect;
}


void
SqlResult::bind(unsigned long, QString&)
{

/*	if (pos >= vect.size())
		vect.*/
}


int
SqlResult::fetch(MYSQL_STMT *stmt)
{
	ValueVector::iterator	it;
	SqlValue		*ptr;
	int			ret = 0;
	unsigned long		pos;

	for (pos = 0, it = val->begin(); it != val->end(); ++it, ++pos) {
		ptr = *it;
		if (ptr->error()) {
			ret |= ptr->fetch(stmt, pos);
			if (ret != 0)
				break;
		}
	}

	return (ret);
}


QVariant
SqlResult::value(unsigned long pos)const
{
	QVariant	ret;

	if (pos < static_cast<unsigned long>(val->size()))
		ret = val->at(pos)->value();

	return (ret);
}


QVariant
SqlResult::value(const QString &name)const
{
	QVariant	ret;
	SqlValue	*ptr = map->value(name.toLower(), NULL);

	if (ptr != NULL)
		ret = ptr->value();

	return (ret);
}


SqlResult::operator MYSQL_BIND*()
{

	return (vect->data());
}

/* EOF */
