/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: sql_server.cpp,v 1.19 2013/11/05 07:42:05 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include <mysql.h>

#include "sql_server.h"
#include "sql_account.h"
#include "sql_connection.h"
#include "basic_status.h"
#include "sql_error.h"
#include "sql_query.h"
#include "settings.h"


#define	SET_REFRESH(flags, box, flag)	do {				\
	if ((box)->isChecked())						\
		(flags) |= (flag);					\
} while (0)


SqlServer::SqlServer(SqlAccount *ptr, PluginData *plg_ptr,
	QAction *act, QWidget *parent)
	:QWidget(parent), account(ptr), plugins(plg_ptr), srv_act(act)
{

	connect(account, SIGNAL(close()), this, SLOT(accountDestroyed()));
	info.connection = new SqlConnection();
	info.connection->connect(SIGNAL(connected()), this, SLOT(connected()));
	info.connection->connect(SIGNAL(closed()), this, SLOT(closed()));
	info.connection->connect(SIGNAL(error(const SqlError&)),
		this, SLOT(error(const SqlError&)));

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	tab = new QTabWidget(this);
	connect(tab, SIGNAL(currentChanged(int)),
		this, SLOT(windowChanged(int)));
	layout->addWidget(tab, 1);

	addStatus();
	addPlugins();

	timer = new QTimer(this);
	connect(timer, SIGNAL(timeout()), this, SLOT(timeout()));

	info.connection->connect(account);
	setCursor(Qt::WaitCursor);
	setEnabled(FALSE);
}

SqlServer::~SqlServer()
{

	account->setService(lastService());
	SAFE_DELETE(timer);
	SAFE_DELETE(srv_act);
	delete status;
	delete tab;
	delete layout;
	delete info.connection;
}


void
SqlServer::accountDestroyed()
{

	closed();
	emit reject(this);
}


void
SqlServer::connected()
{

	info.me = SqlUser::split(info.connection->account());
	unsetCursor();
	setEnabled(TRUE);
	account->setStatus(ACC_CONNECTED, TRUE);
	emit changed(account);
	RELOAD_TIMER(timer);
	crc_global = 0;
	timeout();
	setService(account->service());
}


void
SqlServer::closed()
{

	setEnabled(FALSE);
	if ((timer != NULL) && timer->isActive())
		timer->stop();
	if (account->status() & ACC_CONNECTED) {
		account->setStatus(ACC_CONNECTED, FALSE);
		emit changed(account);
	}
	if (timer != NULL)
		QTimer::singleShot(0, this, SLOT(reconnect()));
}


void
SqlServer::error(const SqlError &err)
{

	unsetCursor();
	err.show(this);
	emit reject(this);
}


void
SqlServer::timeout()
{
	static QString	table(GrantInfo::systemTable(GrantInfo::G_GLOBAL));
	QWidget		*win = tab->currentWidget();
	SqlService	*srv = service_cast(win);
	quint64		crc = info.connection->checksum(table);

	if (crc != crc_global) {
		/* crc of the global table changed */
		reloadPrivileges();
		if (info.priv != PRIV_NOT_READY)
			crc_global = crc;
	}
	if (srv != NULL) {
		if (win->isEnabled())
			srv->timeout();
	} else
		windowTimeout();
}


void
SqlServer::refresh()
{
	RefreshDialog	*dlg;
	unsigned int	flags = 0;
	SqlError	err;

	dlg = new RefreshDialog(this);
	FATAL(dlg == NULL);

	if (dlg->exec())
		flags = dlg->flags();
	delete dlg;

	if (flags != 0) {
		err = info.connection->refresh(flags);
		if (err != 0)
			error(err);
	}
}


void
SqlServer::shutdown()
{
	Confirm		q, w;
	QString		host(account->host());
	SqlError	err;

	q.title = QString::null;
	q.text = tr("Do you really want to shutdown the server %1?")
		.arg(host);
	w.title = QString::null;
	w.text = tr("I hope you understand that the server %1 will be "
		"unavailable for all users, also some incomplete "
		"transactions could be lost.\n"
		"Continue anyway?").arg(host);

	if (Core::doubleConfirm(q, w, this)) {
		err = info.connection->shutdown();
		if (err == 0) {
			closeServer();
			/*
			 * this object does not exist anymore
			 * the last report here: killed ;-)
			 */
			QMessageBox::information(NULL, tr("Shutdown complete"),
				tr("The server %1 has shuted down "
				"successfuly").arg(host));
		} else
			error(err);
	}
}


void
SqlServer::addStatus()
{

	status = new BasicStatus(&info, this);
	tab->addTab(status, status->tabName());
	connect(status, SIGNAL(reject()), this, SLOT(closeServer()));
	connect(status, SIGNAL(shutdown()), this, SLOT(shutdown()));
	connect(status, SIGNAL(refresh()), this, SLOT(refresh()));
	status->privilegesChanged();
}


void
SqlServer::addPlugins()
{
	PluginVector::const_iterator	it;
	PluginVector	*vect = *plugins;

	for (it = vect->constBegin(); it != vect->constEnd(); ++it)
		addPlugin(*it);
}


void
SqlServer::addPlugin(QPluginLoader *plg)
{
	ServicePlugin	*srv;
	SqlService	*ssrv;
	QWidget		*win;

	FATAL(plg == NULL);
	srv = PLUGIN_CAST(plg);
	FATAL(srv == NULL);
	win = srv->create(&info, tab);
	ssrv = service_cast(win);
	FATAL(ssrv == NULL);
	tab->addTab(win, ssrv->tabName());
	ssrv->privilegesChanged();
}


void
SqlServer::closeServer()
{

	emit reject(this);
}


void
SqlServer::windowTimeout()
{
	int		i, cnt = tab->count();
	SqlService	*srv;

	for (i = 0; i < cnt; ++i) {
		srv = service_cast(tab->widget(i));
		if (srv != NULL)
			srv->timeout();
	}
}


void
SqlServer::reloadPrivileges()
{
	GrantInfo		gi;
	SqlPrivilegesFlags	pf;

	gi.type = GrantInfo::G_GLOBAL;
	gi.host = info.me.host();
	gi.user = info.me.user();
	pf = info.connection->privileges(gi)["*.*"];
	if (info.priv != pf) {
		info.priv = pf;
		privilegesChanged();
	}
}


void
SqlServer::privilegesChanged()
{
	int		i, cnt = tab->count();
	SqlService	*srv;

	for (i = 0; i < cnt; ++i) {
		srv = service_cast(tab->widget(i));
		if (srv != NULL)
			srv->privilegesChanged();
	}
}


void
SqlServer::setService(const QString &sname)
{
	int		i, cnt = tab->count();
	QWidget		*win;
	SqlService	*srv;

	for (i = 0; i < cnt; ++i) {
		win = tab->widget(i);
		srv = service_cast(win);
		if ((srv != NULL) && (sname.compare(srv->service(),
			Qt::CaseInsensitive) == 0)) {
			tab->setCurrentWidget(win);
			break;
		}
	}
}


void
SqlServer::windowChanged(int)
{

	timeout();
}


void
SqlServer::reconnect()
{
	SqlError	err;

	if (SqlConnection::tryConnection(*account, err) != TRUE) {
		/* ask  to reconnect */
		QMessageBox::StandardButton	result;
		QString	host(account->host());

		unsetCursor();
		result = QMessageBox::question(this, tr("Lost connection"),
			tr("Connection to %1 lost and a new try to connect "
			"failed.\n"
			"Do you want to try to reconnect or "
			"just close this window?").arg(host),
			QMessageBox::Yes | QMessageBox::Close |
			QMessageBox::Cancel, QMessageBox::Yes);
		switch (result) {
			case QMessageBox::Yes:
				/* reconnect */
				QTimer::singleShot(0, this, SLOT(reconnect()));
				break;
			case QMessageBox::Close:
				emit reject(this);
				break;
			default:
				/* nothing to do here */
				break;
		}
	} else {
		/* connected */
		info.connection->connect();
	}
}


QAction *
SqlServer::action()const
{

	return (srv_act);
}


QString
SqlServer::lastService()const
{
	QString	ret;
	SqlService	*srv = service_cast(tab->currentWidget());

	if (srv != NULL)
		ret = srv->service();

	return (ret);
}

//-----------------------------------------------------------------------------

RefreshDialog::RefreshDialog(QWidget *parent)
	:QDialog(parent)
{

	setWindowTitle(tr("Flushing and reseting"));
	layout = new QVBoxLayout(this);

	grant = new QCheckBox(tr("Refresh the grant table"), this);
	layout->addWidget(grant);

	log = new QCheckBox(tr("Flush the logs"), this);
	layout->addWidget(log);

	tables = new QCheckBox(tr("Flush the table cache"), this);
	layout->addWidget(tables);

	hosts = new QCheckBox(tr("Flush the host cache"), this);
	layout->addWidget(hosts);

	status = new QCheckBox(tr("Reset status variables"), this);
	layout->addWidget(status);

	threads = new QCheckBox(tr("Flush the thread cache"), this);
	layout->addWidget(threads);

	slave = new QCheckBox(tr("On a slave replication server, reset the "
		"master server information and restart the slave"), this);
	layout->addWidget(slave);

	master = new QCheckBox(tr("On a master replication server, remove the "
		"binary log files listed in the binary log index and truncate "
		"the index file"), this);
	layout->addWidget(master);

	dlg = new QDialogButtonBox(QDialogButtonBox::Ok |
		QDialogButtonBox::Cancel, Qt::Horizontal, this);
	connect(dlg, SIGNAL(accepted()), this, SLOT(accept()));
	connect(dlg, SIGNAL(rejected()), this, SLOT(reject()));
	layout->addWidget(dlg, 0, Qt::AlignBottom);
}

RefreshDialog::~RefreshDialog()
{

	delete dlg;
	delete master;
	delete slave;
	delete threads;
	delete status;
	delete hosts;
	delete tables;
	delete log;
	delete grant;
	delete layout;
}


unsigned int
RefreshDialog::flags()const
{
	unsigned int	ret = 0;

	SET_REFRESH(ret, grant, REFRESH_GRANT);
	SET_REFRESH(ret, log, REFRESH_LOG);
	SET_REFRESH(ret, tables, REFRESH_TABLES);
	SET_REFRESH(ret, hosts, REFRESH_HOSTS);
	SET_REFRESH(ret, status, REFRESH_STATUS);
	SET_REFRESH(ret, threads, REFRESH_THREADS);
	SET_REFRESH(ret, slave, REFRESH_SLAVE);
	SET_REFRESH(ret, master, REFRESH_MASTER);

	return (ret);
}

/* EOF */
