/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: workspace.cpp,v 1.12 2013/10/09 14:18:13 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "workspace.h"
#include "settings.h"
#include "sub_window.h"
#include "sql_account_editor.h"
#include "sql_server.h"
#include "plugin_editor.h"


Workspace::Workspace(Settings *ptr, QMenu **sptr, QWidget *parent)
	:QWidget(parent), cfg(ptr), server_ptr(sptr), sub_list(NULL),
	sub_plug(NULL)
{

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	area = new QMdiArea(this);
	connect(area, SIGNAL(subWindowActivated(QMdiSubWindow *)),
		this, SLOT(showMessage(QMdiSubWindow *)));
	layout->addWidget(area, 1);
	area->setViewMode(QMdiArea::SubWindowView);
	area->setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
	area->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);

	QTimer::singleShot(0, this, SLOT(connectServers()));
}

Workspace::~Workspace()
{

	delete area;
	delete layout;
}


void
Workspace::serverList()
{
	SqlAccountEditor	*edit;
	int			size;

	if (sub_list == NULL) {
		/* no such window, create it */
		sub_list = new SubWindow(area, 0);
		connect(sub_list, SIGNAL(closed(const SubWindow *)),
			this, SLOT(windowDestroyed(const SubWindow *)));
		sub_list->setAttribute(Qt::WA_DeleteOnClose, TRUE);
		edit = new SqlAccountEditor(*cfg);
		connect(edit, SIGNAL(reject()),
			this, SLOT(destroyServerList()));
		connect(edit, SIGNAL(doConnection(SqlAccount *)),
			this, SLOT(changeConnection(SqlAccount *)));
		sub_list->setWidget(edit);
		sub_list->setWindowIcon(QIcon(":/icons/manager.png"));
		sub_list->setWindowTitle(tr("Manager's accounts"));
		sub_list->show();
		size = fontMetrics().lineSpacing() * 40;
		sub_list->setMinimumSize(size, size * 3 / 4);
	} else {
		/* raise the window */
		area->setActiveSubWindow(sub_list);
	}
}


void
Workspace::pluginList()
{
	PluginEditor	*edit;
	int		size;

	if (sub_plug == NULL) {
		/* no such window, create it */
		sub_plug = new SubWindow(area, 0);
		connect(sub_plug, SIGNAL(closed(const SubWindow *)),
			this, SLOT(windowDestroyed(const SubWindow *)));
		sub_plug->setAttribute(Qt::WA_DeleteOnClose, TRUE);
		edit = new PluginEditor(*cfg);
		connect(edit, SIGNAL(reject()),
			this, SLOT(destroyPluginList()));
		connect(edit, SIGNAL(loaded(QPluginLoader *)),
			this, SLOT(addPlugin(QPluginLoader *)));
		sub_plug->setWidget(edit);
		sub_plug->setWindowIcon(QIcon(":/icons/plugin.png"));
		sub_plug->setWindowTitle(tr("Plugins of extensions"));
		sub_plug->show();
		size = fontMetrics().lineSpacing() * 40;
		sub_plug->setMinimumSize(size, size * 3 / 4);
	} else {
		/* raise the window */
		area->setActiveSubWindow(sub_plug);
	}
}


void
Workspace::windowDestroyed(const SubWindow *win)
{

	if (win == sub_list) {
		/* server list destroyed */
		sub_list = NULL;
	} else if (win == sub_plug) {
		/* plugin list destroyed */
		sub_plug = NULL;
	}
}


void
Workspace::destroyServerList()
{

	IF_DELETE(sub_list);
}


void
Workspace::destroyPluginList()
{

	IF_DELETE(sub_plug);
}


void
Workspace::showMessage(QMdiSubWindow *win)
{

	if (win == NULL)
		emit message(QString::null);
	else {
		/* server */
		emit message(win->windowTitle());
	}
}


/* Connects or disconnects specified account */
void
Workspace::changeConnection(SqlAccount *ptr)
{
/*	QWidget		*view = area->viewport();
	int		wdt = view->width(),
			hgt = view->height();*/
	SubWindow	*sw;
	SqlServer	*srv;
	QAction		*act;

	FATAL(*server_ptr == NULL);
	if (ptr->hasStatus(ACC_CONNECTED)) {
		ptr->closeConnection();
	} else {
		sw = new SubWindow(area, 0);
		connect(sw, SIGNAL(closed(const SubWindow *)),
			this, SLOT(windowDestroyed(const SubWindow *)));
		sw->setAttribute(Qt::WA_DeleteOnClose, TRUE);
		act = (*server_ptr)->addAction(ptr->account());
		srv = new SqlServer(ptr, *cfg, act);
		connect(srv, SIGNAL(reject(const QWidget *)),
			this, SLOT(destroyServer(const QWidget *)));
		connect(srv, SIGNAL(changed(SqlAccount *)),
			this, SLOT(serverChanged(SqlAccount *)));
		sw->setWidget(srv);
		sw->setWindowIcon(QIcon(":/icons/server.png"));
		sw->setWindowTitle(ptr->account());
		/*sw->show();
		sw->resize(wdt * 9 / 10, hgt * 9 / 10);*/
		sw->showMaximized();
	}
}


void
Workspace::connectServers()
{
	SqlAccountVector		*vect = *cfg;
	SqlAccount			*ptr;
	SqlAccountVector::iterator	it;

	for (it = vect->begin(); it != vect->end(); ++it) {
		ptr = *it;
		if (ptr->isSet(SQL_AUTO_LOGIN))
			changeConnection(ptr);
	}
}


void
Workspace::destroyServer(const QWidget *widget)
{
	QList<QMdiSubWindow *>	list(area->subWindowList());
	SubWindow		*win;
	QList<QMdiSubWindow *>::iterator	it;

	for (it = list.begin(); it != list.end(); ++it) {
		win = static_cast<SubWindow *>(*it);
		if (win->widget() == widget) {
			/* the window is found! */
			delete win;
		}
	}
}


void
Workspace::addPlugin(QPluginLoader *plg)
{
	QList<QMdiSubWindow *>::iterator	it;
	QList<QMdiSubWindow *>	list(area->subWindowList());
	QMdiSubWindow		*win;
	SqlServer		*srv;

	for (it = list.begin(); it != list.end(); ++it) {
		win = *it;
		srv = qobject_cast<SqlServer *>(win->widget());
		if (srv != NULL)
			srv->addPlugin(plg);
	}
}


/* this slot called when a server changes data in the account list */
void
Workspace::serverChanged(SqlAccount *acc)
{
	SqlAccountEditor	*edit;

	if (sub_list != NULL) {
		edit = qobject_cast<SqlAccountEditor *>(sub_list->widget());
		if (edit != NULL)
			edit->accountChanged(acc);
	}
}


void
Workspace::raiseServer(QAction *act)
{
	QList<QMdiSubWindow *>	list(area->subWindowList());
	SubWindow		*win;
	SqlServer		*srv;
	QList<QMdiSubWindow *>::iterator	it;

	for (it = list.begin(); it != list.end(); ++it) {
		win = static_cast<SubWindow *>(*it);
		srv = qobject_cast<SqlServer *>(win->widget());
		if ((srv != NULL) && (srv->action() == act)) {
			area->setActiveSubWindow(win);
			break;
		}
	}
}

/* EOF */
