/*-
 * Copyright (c) 2013 by SilverSoft.Net
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: charset_model.cpp,v 1.3 2013/08/28 12:18:14 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "charset_model.h"
#include "sql_connection.h"
#include "sql_query.h"
#include "sql_error.h"
#include "plg_gui.h"


#define	LAB_CHARSET		0
#define	LAB_COLLATE		1
#define	LAB_MAXLEN		2
#define	LAB_DESCRIPTION		3
#define	LAB_TOTAL		(LAB_DESCRIPTION + 1)


CharsetModel::CharsetModel(SqlConnection *conn, QObject *parent)
	:QAbstractTableModel(parent)
{

	query = new SqlQuery(conn);
	query->prepare(QString("select CHARACTER_SET_NAME, "
		"DEFAULT_COLLATE_NAME, MAXLEN, DESCRIPTION from %1")
		.arg(Core::informationSchema("CHARACTER_SETS")));

	query->connect(this, SLOT(reloadData()));
}

CharsetModel::~CharsetModel()
{
	CharsetVector::iterator	it;
	Charset			*cset;

	for (it = vect.begin(); it != vect.end(); ++it) {
		cset = *it;
		delete cset;
	}

	delete query;
}


QVariant
CharsetModel::data(const QModelIndex &ndx, int role)const
{
	QVariant	ret;
	Charset		*cset = charset(ndx);

	if (cset != NULL)
		switch (role) {
			case Qt::DisplayRole:
				ret = displayData(cset, ndx.column());
				break;
			case Qt::BackgroundRole:
				ret = backgroundData(cset);
				break;
		}

	return (ret);
}


QVariant
CharsetModel::headerData(int section, Qt::Orientation ort, int role)const
{
	QVariant	ret;

	if ((ort == Qt::Horizontal) && (role == Qt::DisplayRole))
		switch (section) {
			case LAB_CHARSET:
				ret = tr("Charset");
				break;
			case LAB_COLLATE:
				ret = tr("Collate");
				break;
			case LAB_MAXLEN:
				ret = tr("MaxLen");
				break;
			case LAB_DESCRIPTION:
				ret = tr("Description");
				break;
		}

	return (ret);
}


int
CharsetModel::columnCount(const QModelIndex&)const
{

	return (LAB_TOTAL);
}


int
CharsetModel::rowCount(const QModelIndex&)const
{

	return (vect.size());
}


void
CharsetModel::markAll()
{
	CharsetVector::iterator	it;

	for (it = vect.begin(); it != vect.end(); ++it)
		(*it)->setMarked(TRUE);
}


void
CharsetModel::deleteMarked(unsigned long &changes)
{
	CharsetVector::iterator	it;
	Charset			*cset;
	int			pos;

	for (pos = 0, it = vect.begin(); it != vect.end(); ) {
		cset = *it;
		if (cset->isMarked()) {
			beginRemoveRows(QModelIndex(), pos, pos);
			it = vect.erase(it);
			endRemoveRows();
			map.remove(cset->charset());
			delete cset;
			changes++;
		} else {
			++pos;
			++it;
		}
	}
}


void
CharsetModel::reloadData()
{
	Charset		*cset, tmp;
	int		pos;
	unsigned long	changes = 0;

	markAll();
	while (query->next()) {
		copyData(tmp, query);
		cset = map.value(tmp.charset(), NULL);
		if (cset == NULL) {
			/* create a new charset */
			cset = new Charset(tmp);
			pos = findPosition(cset->charset());
			beginInsertRows(QModelIndex(), pos, pos);
			vect.insert(pos, cset);
			endInsertRows();
			map.insert(cset->charset(), cset);
			changes++;
		} else
			updateData(cset, tmp, changes);
	}
	query->free();
	deleteMarked(changes);
	if (changes > 0)
		emit modelChanged();
}


void
CharsetModel::timeout()
{

	query->exec();
}


void
CharsetModel::clearSelection()
{
	CharsetVector::iterator	it;

	for (it = vect.begin(); it != vect.end(); ++it)
		(*it)->setFlags(SELECTED, FALSE);
	emit dataChanged(index(0, LAB_CHARSET),
		index(vect.size(), LAB_DESCRIPTION));
}


void
CharsetModel::selectCharset(const QString &cname, QModelIndex &ndx)
{
	CharsetVector::iterator	it;
	Charset			*cset;
	int			was, pos, now;
	QModelIndex		nl, nr;

	for (pos = 0, it = vect.begin(); it != vect.end(); ++it, ++pos) {
		cset = *it;
		was = cset->flags();
		cset->setFlags(SELECTED, cset->charset() == cname);
		now = cset->flags();
		if (now != was) {
			nl = index(pos, LAB_CHARSET);
			nr = index(pos, LAB_DESCRIPTION);
			if ((now & SELECTED) && !ndx.isValid())
				ndx = nl;
			emit dataChanged(nl, nr);
		}
	}
}


int
CharsetModel::findPosition(const QString &name)
{
	CharsetVector::const_iterator	it;
	int				ret = vect.size(), i;

	for (i = 0, it = vect.constBegin(); it != vect.constEnd(); ++it, ++i) {
		if ((*it)->charset() > name) {
			ret = i;
			break;
		}
	}

	return (ret);
}


void
CharsetModel::copyData(Charset &cset, SqlQuery *q)
{

	cset.setCharset(q->value(0).toString());
	cset.setCollate(q->value(1).toString());
	cset.setMaxLen(q->value(2).toLongLong());
	cset.setDescription(q->value(3).toString());
}


void
CharsetModel::updateData(Charset *ptr, const Charset &cset,
	unsigned long &changes)
{
	int	pos = -1;

	ptr->setMarked(FALSE);

	if (ptr->charset() != cset.charset()) {
		ptr->setCharset(cset.charset());
		changed(pos, LAB_CHARSET, ptr, changes);
	}

	if (ptr->collate() != cset.collate()) {
		ptr->setCollate(cset.collate());
		changed(pos, LAB_COLLATE, ptr, changes);
	}

	if (ptr->maxLen() != cset.maxLen()) {
		ptr->setMaxLen(cset.maxLen());
		changed(pos, LAB_MAXLEN, ptr, changes);
	}

	if (ptr->description() != cset.description()) {
		ptr->setDescription(cset.description());
		changed(pos, LAB_DESCRIPTION, ptr, changes);
	}
}


void
CharsetModel::changed(int &row, int col, Charset *cset,
	unsigned long &changes)
{
	QModelIndex	ndx = index(row, col);

	if (row < 0)
		row = vect.indexOf(cset);
	if (col >= 0) {
		changes++;
		emit dataChanged(ndx, ndx);
	}
}


Charset *
CharsetModel::charset(const QModelIndex &ndx)const
{

	return (charset(ndx.row()));
}


Charset *
CharsetModel::charset(int row)const
{
	Charset	*ret;

	if ((row < 0) || (row >= vect.size()))
		ret = NULL;
	else
		ret = vect.at(row);

	return (ret);
}


QVariant
CharsetModel::displayData(const Charset *cset, int col)const
{
	QVariant	ret;

	switch (col) {
		case LAB_CHARSET:
			ret = cset->charset();
			break;
		case LAB_COLLATE:
			ret = cset->collate();
			break;
		case LAB_MAXLEN:
			ret = cset->maxLen();
			break;
		case LAB_DESCRIPTION:
			ret = cset->description();
			break;
	}

	return (ret);
}


QVariant
CharsetModel::backgroundData(const Charset *cset)const
{
	QVariant	ret;

	if (cset->flags() & SELECTED)
		ret = BRUSH_SELECTED;

	return (ret);
}


bool
CharsetModel::isEmpty()const
{

	return (vect.empty());
}

/* EOF */
