/*-
 * Copyright (c) 2013 by SilverSoft.Net
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: collation_model.cpp,v 1.3 2013/08/28 12:18:14 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "collation_model.h"
#include "sql_connection.h"
#include "sql_query.h"
#include "sql_error.h"
#include "plg_gui.h"


#define	LAB_COLLATION		0
#define	LAB_CHARSET		1
#define	LAB_ID			2
#define	LAB_DEFAULT		3
#define	LAB_COMPILED		4
#define	LAB_SORTLEN		5
#define	LAB_TOTAL		(LAB_SORTLEN + 1)


CollationModel::CollationModel(SqlConnection *conn, QObject *parent)
	:QAbstractTableModel(parent)
{

	query = new SqlQuery(conn);
	query->prepare(QString("select COLLATION_NAME, CHARACTER_SET_NAME, "
		"ID, IS_DEFAULT, IS_COMPILED, SORTLEN from %1")
		.arg(Core::informationSchema("COLLATIONS")));

	query->connect(this, SLOT(reloadData()));
}

CollationModel::~CollationModel()
{
	CollationVector::iterator	it;
	Collation			*coll;

	for (it = vect.begin(); it != vect.end(); ++it) {
		coll = *it;
		delete coll;
	}

	delete query;
}


QVariant
CollationModel::data(const QModelIndex &ndx, int role)const
{
	QVariant	ret;
	Collation	*coll = collation(ndx);

	if (coll != NULL)
		switch (role) {
			case Qt::DisplayRole:
				ret = displayData(coll, ndx.column());
				break;
			case Qt::BackgroundRole:
				ret = backgroundData(coll);
				break;
		}

	return (ret);
}


QVariant
CollationModel::headerData(int section, Qt::Orientation ort, int role)const
{
	QVariant	ret;

	if ((ort == Qt::Horizontal) && (role == Qt::DisplayRole))
		switch (section) {
			case LAB_COLLATION:
				ret = tr("Collation");
				break;
			case LAB_CHARSET:
				ret = tr("Charset");
				break;
			case LAB_ID:
				ret = tr("Id");
				break;
			case LAB_DEFAULT:
				ret = tr("is default?");
				break;
			case LAB_COMPILED:
				ret = tr("is compiled?");
				break;
			case LAB_SORTLEN:
				ret = tr("Sort length");
				break;
		}

	return (ret);
}


int
CollationModel::columnCount(const QModelIndex&)const
{

	return (LAB_TOTAL);
}


int
CollationModel::rowCount(const QModelIndex&)const
{

	return (vect.size());
}


void
CollationModel::markAll()
{
	CollationVector::iterator	it;

	for (it = vect.begin(); it != vect.end(); ++it)
		(*it)->setMarked(TRUE);
}


void
CollationModel::deleteMarked(unsigned long &changes)
{
	CollationVector::iterator	it;
	Collation			*coll;
	int				pos;

	for (pos = 0, it = vect.begin(); it != vect.end(); ) {
		coll = *it;
		if (coll->isMarked()) {
			beginRemoveRows(QModelIndex(), pos, pos);
			it = vect.erase(it);
			endRemoveRows();
			map.remove(coll->collation());
			delete coll;
			changes++;
		} else {
			++pos;
			++it;
		}
	}
}


void
CollationModel::reloadData()
{
	Collation	*coll, tmp;
	int		pos;
	unsigned long	changes = 0;

	markAll();
	while (query->next()) {
		copyData(tmp, query);
		coll = map.value(tmp.collation(), NULL);
		if (coll == NULL) {
			/* create a new collation */
			coll = new Collation(tmp);
			pos = findPosition(coll->collation());
			beginInsertRows(QModelIndex(), pos, pos);
			vect.insert(pos, coll);
			endInsertRows();
			map.insert(coll->collation(), coll);
			changes++;
		} else
			updateData(coll, tmp, changes);
	}
	query->free();
	deleteMarked(changes);
	if (changes > 0)
		emit modelChanged();
}


void
CollationModel::timeout()
{

	query->exec();
}


void
CollationModel::clearSelection()
{
	CollationVector::iterator	it;

	for (it = vect.begin(); it != vect.end(); ++it)
		(*it)->setFlags(SELECTED, FALSE);
	emit dataChanged(index(0, LAB_COLLATION),
		index(vect.size(), LAB_SORTLEN));
}


void
CollationModel::selectCollation(const QString &cset, QModelIndex &ndx)
{
	CollationVector::iterator	it;
	Collation			*coll;
	int				was, now, pos;
	QModelIndex			nl, nr;

	for (pos = 0, it = vect.begin(); it != vect.end(); ++it, ++pos) {
		coll = *it;
		was = coll->flags();
		coll->setFlags(SELECTED, coll->charset() == cset);
		now = coll->flags();
		if (now != was) {
			nl = index(pos, LAB_COLLATION);
			nr = index(pos, LAB_SORTLEN);
			if ((now & SELECTED) && !ndx.isValid())
				ndx = nl;
			emit dataChanged(nl, nr);
		}
	}
}


int
CollationModel::findPosition(const QString &name)
{
	CollationVector::const_iterator	it;
	int				ret = vect.size(), i;

	for (i = 0, it = vect.constBegin(); it != vect.constEnd(); ++it, ++i) {
		if ((*it)->collation() > name) {
			ret = i;
			break;
		}
	}

	return (ret);
}


void
CollationModel::copyData(Collation &coll, SqlQuery *q)
{

	coll.setCollation(q->value(0).toString());
	coll.setCharset(q->value(1).toString());
	coll.setId(q->value(2).toLongLong());
	coll.setDefault(q->value(3).toString());
	coll.setCompiled(q->value(4).toString());
	coll.setSortLen(q->value(5).toLongLong());
}


void
CollationModel::updateData(Collation *ptr, const Collation &coll,
	unsigned long &changes)
{
	int	pos = -1;

	ptr->setMarked(FALSE);

	if (ptr->collation() != coll.collation()) {
		ptr->setCollation(coll.collation());
		changed(pos, LAB_COLLATION, ptr, changes);
	}

	if (ptr->charset() != coll.charset()) {
		ptr->setCharset(coll.charset());
		changed(pos, LAB_CHARSET, ptr, changes);
	}

	if (ptr->id() != coll.id()) {
		ptr->setId(coll.id());
		changed(pos, LAB_ID, ptr, changes);
	}

	if (ptr->isDefault() != coll.isDefault()) {
		ptr->setDefault(coll.isDefault());
		changed(pos, LAB_DEFAULT, ptr, changes);
	}

	if (ptr->isCompiled() != coll.isCompiled()) {
		ptr->setCompiled(coll.isCompiled());
		changed(pos, LAB_COMPILED, ptr, changes);
	}

	if (ptr->sortLen() != coll.sortLen()) {
		ptr->setSortLen(coll.sortLen());
		changed(pos, LAB_SORTLEN, ptr, changes);
	}
}


void
CollationModel::changed(int &row, int col, Collation *coll,
	unsigned long &changes)
{
	QModelIndex	ndx = index(row, col);

	if (row < 0)
		row = vect.indexOf(coll);
	if (col >= 0) {
		changes++;
		emit dataChanged(ndx, ndx);
	}
}


Collation *
CollationModel::collation(const QModelIndex &ndx)const
{

	return (collation(ndx.row()));
}


Collation *
CollationModel::collation(int row)const
{
	Collation	*ret;

	if ((row < 0) || (row >= vect.size()))
		ret = NULL;
	else
		ret = vect.at(row);

	return (ret);
}


QVariant
CollationModel::displayData(const Collation *coll, int col)const
{
	QVariant	ret;

	switch (col) {
		case LAB_COLLATION:
			ret = coll->collation();
			break;
		case LAB_CHARSET:
			ret = coll->charset();
			break;
		case LAB_ID:
			ret = coll->id();
			break;
		case LAB_DEFAULT:
			ret = coll->isDefault();
			break;
		case LAB_COMPILED:
			ret = coll->isCompiled();
			break;
		case LAB_SORTLEN:
			ret = coll->sortLen();
			break;
	}

	return (ret);
}


QVariant
CollationModel::backgroundData(const Collation *coll)const
{
	QVariant	ret;

	if (coll->flags() & SELECTED)
		ret = BRUSH_SELECTED;

	return (ret);
}


bool
CollationModel::isEmpty()const
{

	return (vect.empty());
}

/* EOF */
