/*-
 * Copyright (c) 2013 by SilverSoft.Net
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: contributor_model.cpp,v 1.2 2013/08/28 12:18:14 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "mysql_plugin_model.h"
#include "sql_connection.h"
#include "sql_query.h"
#include "sql_error.h"

#define	LAB_PLUGIN		0
#define	LAB_VERSION		1
#define	LAB_STATUS		2
#define	LAB_TYPE		3
#define	LAB_DESCRIPTION		4
#define	LAB_TOTAL		(LAB_DESCRIPTION + 1)


MySQLPluginModel::MySQLPluginModel(SqlConnection *conn, QObject *parent)
	:QAbstractTableModel(parent)
{

	query = new SqlQuery(conn);
	query->prepare(QString("select PLUGIN_NAME, PLUGIN_VERSION, "
		"PLUGIN_STATUS, PLUGIN_TYPE, PLUGIN_TYPE_VERSION, "
		"PLUGIN_LIBRARY, PLUGIN_LIBRARY_VERSION, PLUGIN_AUTHOR, "
		"PLUGIN_DESCRIPTION, PLUGIN_LICENSE, LOAD_OPTION, "
		"PLUGIN_MATURITY, PLUGIN_AUTH_VERSION from %1")
		.arg(Core::informationSchema("PLUGINS")));

	query->connect(this, SLOT(reloadData()));
}

MySQLPluginModel::~MySQLPluginModel()
{
	PluginVector::iterator	it;
	Plugin			*plg;

	for (it = vect.begin(); it != vect.end(); ++it) {
		plg = *it;
		delete plg;
	}

	delete query;
}


QVariant
MySQLPluginModel::data(const QModelIndex &ndx, int role)const
{
	QVariant	ret;
	Plugin		*plg = plugin(ndx);

	if (plg != NULL)
		switch (role) {
			case Qt::DisplayRole:
				ret = displayData(plg, ndx.column());
				break;
/*			case Qt::BackgroundRole:
				ret = backgroundData(eng);
				break;*/
		}

	return (ret);
}


QVariant
MySQLPluginModel::headerData(int section, Qt::Orientation ort, int role)const
{
	QVariant	ret;

	if ((ort == Qt::Horizontal) && (role == Qt::DisplayRole))
		switch (section) {
			case LAB_PLUGIN:
				ret = tr("Plugin");
				break;
			case LAB_VERSION:
				ret = tr("Version");
				break;
			case LAB_STATUS:
				ret = tr("Status");
				break;
			case LAB_TYPE:
				ret = tr("Type");
				break;
			case LAB_DESCRIPTION:
				ret = tr("Description");
				break;
		}

	return (ret);
}


int
MySQLPluginModel::columnCount(const QModelIndex&)const
{

	return (LAB_TOTAL);
}


int
MySQLPluginModel::rowCount(const QModelIndex&)const
{

	return (vect.size());
}


void
MySQLPluginModel::markAll()
{
	PluginVector::iterator	it;

	for (it = vect.begin(); it != vect.end(); ++it)
		(*it)->setMarked(TRUE);
}


void
MySQLPluginModel::deleteMarked(unsigned long &changes)
{
	PluginVector::iterator	it;
	Plugin			*plg;
	int			pos;

	for (pos = 0, it = vect.begin(); it != vect.end(); ) {
		plg = *it;
		if (plg->isMarked()) {
			beginRemoveRows(QModelIndex(), pos, pos);
			it = vect.erase(it);
			endRemoveRows();
			map.remove(plg->pluginName());
			delete plg;
			changes++;
		} else {
			++pos;
			++it;
		}
	}
}


void
MySQLPluginModel::reloadData()
{
	Plugin		*plg, tmp;
	int		pos;
	unsigned long	changes = 0;

	markAll();
	while (query->next()) {
		copyData(tmp, query);
		plg = map.value(tmp.pluginName(), NULL);
		if (plg == NULL) {
			/* create a new plugin */
			plg = new Plugin(tmp);
			pos = findPosition(plg->pluginName());
			beginInsertRows(QModelIndex(), pos, pos);
			vect.insert(pos, plg);
			endInsertRows();
			map.insert(plg->pluginName(), plg);
			changes++;
		} else
			updateData(plg, tmp, changes);
	}
	query->free();
	deleteMarked(changes);
	if (changes > 0)
		emit modelChanged();
}


void
MySQLPluginModel::timeout()
{

	query->exec();
}


void
MySQLPluginModel::loadPlugin(const QString &plg, const QString &lib,
	QWidget *parent)
{
	SqlError	err;
	QString		q_load("install plugin %1 soname '%2'");

	err = query->connection()->exec(q_load.arg(plg).arg(lib));
	if (err != 0)
		err.show(parent);
	else {
		timeout();
		QMessageBox::information(parent, tr("Plugin loaded"),
			tr("A new plugin %1 has been installed from "
			"library %2").arg(plg).arg(lib));
	}
}


void
MySQLPluginModel::unloadPlugin(const QString &plg, QWidget *parent)
{
	SqlError	err;
	QString		q_unload("uninstall plugin %1");

	err = query->connection()->exec(q_unload.arg(plg));
	if (err != 0)
		err.show(parent);
	else {
		timeout();
		QMessageBox::information(parent, tr("Plugin unloaded"),
			tr("Plugin %1 has been uninstalled").arg(plg));
	}
}


int
MySQLPluginModel::findPosition(const QString &name)
{
	PluginVector::const_iterator	it;
	int				ret = vect.size(), i;

	for (i = 0, it = vect.constBegin(); it != vect.constEnd(); ++it, ++i) {
		if ((*it)->pluginName() > name) {
			ret = i;
			break;
		}
	}

	return (ret);
}


void
MySQLPluginModel::copyData(Plugin &plg, SqlQuery *q)
{

	plg.setPluginName(q->value(0).toString());
	plg.setPluginVersion(q->value(1).toString());
	plg.setPluginStatus(q->value(2).toString());
	plg.setPluginType(q->value(3).toString());
	plg.setPluginTypeVersion(q->value(4).toString());
	plg.setPluginLibrary(q->value(5).toString());
	plg.setPluginLibraryVersion(q->value(6).toString());
	plg.setPluginAuthor(q->value(7).toString());
	plg.setPluginDescription(q->value(8).toString());
	plg.setPluginLicense(q->value(9).toString());
	plg.setLoadOption(q->value(10).toString());
	plg.setPluginMaturity(q->value(11).toString());
	plg.setPluginAuthVersion(q->value(12).toString());
}


void
MySQLPluginModel::updateData(Plugin *ptr, const Plugin &plg,
	unsigned long &changes)
{
	int	pos = -1;

	ptr->setMarked(FALSE);

	if (ptr->pluginName() != plg.pluginName()) {
		ptr->setPluginName(plg.pluginName());
		changed(pos, LAB_PLUGIN, ptr, changes);
	}

	if (ptr->pluginVersion() != plg.pluginVersion()) {
		ptr->setPluginVersion(plg.pluginVersion());
		changed(pos, LAB_VERSION, ptr, changes);
	}

	if (ptr->pluginStatus() != plg.pluginStatus()) {
		ptr->setPluginStatus(plg.pluginStatus());
		changed(pos, LAB_STATUS, ptr, changes);
	}

	if (ptr->pluginType() != plg.pluginType()) {
		ptr->setPluginType(plg.pluginType());
		changed(pos, LAB_TYPE, ptr, changes);
	}

	if (ptr->pluginDescription() != plg.pluginDescription()) {
		ptr->setPluginDescription(plg.pluginDescription());
		changed(pos, LAB_DESCRIPTION, ptr, changes);
	}

	ptr->setPluginTypeVersion(plg.pluginTypeVersion());
	ptr->setPluginLibrary(plg.pluginLibrary());
	ptr->setPluginLibraryVersion(plg.pluginLibraryVersion());
	ptr->setPluginAuthor(plg.pluginAuthor());
	ptr->setPluginLicense(plg.pluginLicense());
	ptr->setLoadOption(plg.loadOption());
	ptr->setPluginMaturity(plg.pluginMaturity());
	ptr->setPluginAuthVersion(plg.pluginAuthVersion());
}


void
MySQLPluginModel::changed(int &row, int col, Plugin *plg,
	unsigned long &changes)
{
	QModelIndex	ndx = index(row, col);

	if (row < 0)
		row = vect.indexOf(plg);
	if (col >= 0) {
		changes++;
		emit dataChanged(ndx, ndx);
	}
}


Plugin *
MySQLPluginModel::plugin(const QModelIndex &ndx)const
{

	return (plugin(ndx.row()));
}


Plugin *
MySQLPluginModel::plugin(int row)const
{
	Plugin	*ret;

	if ((row < 0) || (row >= vect.size()))
		ret = NULL;
	else
		ret = vect.at(row);

	return (ret);
}


QVariant
MySQLPluginModel::displayData(const Plugin *plg, int col)const
{
	QVariant	ret;

	switch (col) {
		case LAB_PLUGIN:
			ret = plg->pluginName();
			break;
		case LAB_VERSION:
			ret = plg->pluginVersion();
			break;
		case LAB_STATUS:
			ret = plg->pluginStatus();
			break;
		case LAB_TYPE:
			ret = plg->pluginType();
			break;
		case LAB_DESCRIPTION:
			ret = plg->pluginDescription();
			break;
	}

	return (ret);
}

/*
QVariant
EngineModel::backgroundData(const Engine *eng)const
{
	QVariant	ret;
	QString		sup(eng->support());

	if (sup.compare("default", Qt::CaseInsensitive) == 0)
		ret = BRUSH_DEFAULT;
	else if (sup.compare("no", Qt::CaseInsensitive) == 0)
		ret = BRUSH_NOT_SUPPORTED;

	return (ret);
}
*/

bool
MySQLPluginModel::isEmpty()const
{

	return (vect.empty());
}

/* EOF */
