/*-
 * Copyright (c) 2013 by SilverSoft.Net
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: engine_model.cpp,v 1.3 2013/08/28 12:18:14 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "engine_model.h"
#include "sql_connection.h"
#include "sql_query.h"
#include "sql_error.h"

#define	LAB_ENGINE		0
#define	LAB_SUPPORT		1
#define	LAB_TRANSACTIONS	2
#define	LAB_XA			3
#define	LAB_SAVEPOINTS		4
#define	LAB_COMMENT		5
#define	LAB_TOTAL	(LAB_COMMENT + 1)


EngineModel::EngineModel(SqlConnection *conn, QObject *parent)
	:QAbstractTableModel(parent)
{

	query = new SqlQuery(conn);
	query->prepare(QString("select ENGINE, SUPPORT, TRANSACTIONS, XA, "
		"SAVEPOINTS, COMMENT from %1")
		.arg(Core::informationSchema("ENGINES")));

	query->connect(this, SLOT(reloadData()));
}

EngineModel::~EngineModel()
{
	EngineVector::iterator	it;
	Engine			*engine;

	for (it = vect.begin(); it != vect.end(); ++it) {
		engine = *it;
		delete engine;
	}

	delete query;
}


QVariant
EngineModel::data(const QModelIndex &ndx, int role)const
{
	QVariant	ret;
	Engine		*eng = engine(ndx);

	if (eng != NULL)
		switch (role) {
			case Qt::DisplayRole:
				ret = displayData(eng, ndx.column());
				break;
			case Qt::BackgroundRole:
				ret = backgroundData(eng);
				break;
		}

	return (ret);
}


QVariant
EngineModel::headerData(int section, Qt::Orientation ort, int role)const
{
	QVariant	ret;

	if ((ort == Qt::Horizontal) && (role == Qt::DisplayRole))
		switch (section) {
			case LAB_ENGINE:
				ret = tr("Engine");
				break;
			case LAB_SUPPORT:
				ret = tr("Support");
				break;
			case LAB_TRANSACTIONS:
				ret = tr("Transactions");
				break;
			case LAB_XA:
				ret = tr("XA");
				break;
			case LAB_SAVEPOINTS:
				ret = tr("Savepoints");
				break;
			case LAB_COMMENT:
				ret = tr("Comment");
				break;
		}

	return (ret);
}


int
EngineModel::columnCount(const QModelIndex&)const
{

	return (LAB_TOTAL);
}


int
EngineModel::rowCount(const QModelIndex&)const
{

	return (vect.size());
}


void
EngineModel::markAll()
{
	EngineVector::iterator	it;

	for (it = vect.begin(); it != vect.end(); ++it)
		(*it)->setMarked(TRUE);
}


void
EngineModel::deleteMarked(unsigned long &changes)
{
	EngineVector::iterator	it;
	Engine			*eng;
	int			pos;

	for (pos = 0, it = vect.begin(); it != vect.end(); ) {
		eng = *it;
		if (eng->isMarked()) {
			beginRemoveRows(QModelIndex(), pos, pos);
			it = vect.erase(it);
			endRemoveRows();
			map.remove(eng->engine());
			delete eng;
			changes++;
		} else {
			++pos;
			++it;
		}
	}
}


void
EngineModel::reloadData()
{
	Engine		*eng, tmp;
	int		pos;
	unsigned long	changes = 0;

	markAll();
	while (query->next()) {
		copyData(tmp, query);
		eng = map.value(tmp.engine(), NULL);
		if (eng == NULL) {
			/* create a new engine */
			eng = new Engine(tmp);
			pos = findPosition(eng->engine());
			beginInsertRows(QModelIndex(), pos, pos);
			vect.insert(pos, eng);
			endInsertRows();
			map.insert(eng->engine(), eng);
			changes++;
		} else
			updateData(eng, tmp, changes);
	}
	query->free();
	deleteMarked(changes);
	if (changes > 0)
		emit modelChanged();
}


void
EngineModel::timeout()
{

	query->exec();
}


void
EngineModel::setDefault(const QString &eng, QWidget *parent)
{
	SqlError	err;
	QString		q_set("set global default_storage_engine = '%1', "
				"global storage_engine = '%1', "
				"session default_storage_engine = '%1', "
				"session storage_engine = '%1'");

	err = query->connection()->exec(q_set.arg(eng));
	if (err != 0)
		err.show(parent);
	else {
		timeout();
		QMessageBox::information(parent, tr("Storage engine"),
			tr("%1 was set as the default storage engine")
			.arg(eng));
	}
}


int
EngineModel::findPosition(const QString &name)
{
	EngineVector::const_iterator	it;
	int				ret = vect.size(), i;

	for (i = 0, it = vect.constBegin(); it != vect.constEnd(); ++it, ++i) {
		if ((*it)->engine() > name) {
			ret = i;
			break;
		}
	}

	return (ret);
}


void
EngineModel::copyData(Engine &eng, SqlQuery *q)
{

	eng.setEngine(q->value(0).toString());
	eng.setSupport(q->value(1).toString());
	eng.setTransactions(q->value(2).toString());
	eng.setXa(q->value(3).toString());
	eng.setSavepoints(q->value(4).toString());
	eng.setComment(q->value(5).toString());
}


void
EngineModel::updateData(Engine *ptr, const Engine &eng,
	unsigned long &changes)
{
	int	pos = -1;

	ptr->setMarked(FALSE);

	if (ptr->engine() != eng.engine()) {
		ptr->setEngine(eng.engine());
		changed(pos, LAB_ENGINE, ptr, changes);
	}

	if (ptr->support() != eng.support()) {
		ptr->setSupport(eng.support());
		changed(pos, LAB_SUPPORT, ptr, changes);
	}

	if (ptr->transactions() != eng.transactions()) {
		ptr->setTransactions(eng.transactions());
		changed(pos, LAB_TRANSACTIONS, ptr, changes);
	}

	if (ptr->xa() != eng.xa()) {
		ptr->setXa(eng.xa());
		changed(pos, LAB_XA, ptr, changes);
	}

	if (ptr->savepoints() != eng.savepoints()) {
		ptr->setSavepoints(eng.savepoints());
		changed(pos, LAB_SAVEPOINTS, ptr, changes);
	}

	if (ptr->comment() != eng.comment()) {
		ptr->setComment(eng.comment());
		changed(pos, LAB_COMMENT, ptr, changes);
	}
}


void
EngineModel::changed(int &row, int col, Engine *eng, unsigned long &changes)
{
	QModelIndex	ndx = index(row, col);

	if (row < 0)
		row = vect.indexOf(eng);
	if (col >= 0) {
		changes++;
		emit dataChanged(ndx, ndx);
	}
}


Engine *
EngineModel::engine(const QModelIndex &ndx)const
{

	return (engine(ndx.row()));
}


Engine *
EngineModel::engine(int row)const
{
	Engine	*ret;

	if ((row < 0) || (row >= vect.size()))
		ret = NULL;
	else
		ret = vect.at(row);

	return (ret);
}


QVariant
EngineModel::displayData(const Engine *eng, int col)const
{
	QVariant	ret;

	switch (col) {
		case LAB_ENGINE:
			ret = eng->engine();
			break;
		case LAB_SUPPORT:
			ret = eng->support();
			break;
		case LAB_TRANSACTIONS:
			ret = eng->transactions();
			break;
		case LAB_XA:
			ret = eng->xa();
			break;
		case LAB_SAVEPOINTS:
			ret = eng->savepoints();
			break;
		case LAB_COMMENT:
			ret = eng->comment();
			break;
	}

	return (ret);
}


QVariant
EngineModel::backgroundData(const Engine *eng)const
{
	QVariant	ret;
	QString		sup(eng->support());

	if (sup.compare("default", Qt::CaseInsensitive) == 0)
		ret = BRUSH_DEFAULT;
	else if (sup.compare("no", Qt::CaseInsensitive) == 0)
		ret = BRUSH_NOT_SUPPORTED;

	return (ret);
}


bool
EngineModel::isEmpty()const
{

	return (vect.empty());
}

/* EOF */
