/*-
 * Copyright (c) 2013 by SilverSoft.Net
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: engines.cpp,v 1.4 2013/10/09 14:18:13 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "engines.h"
#include "engine_model.h"


Engines::Engines(ServerInfo *info, QWidget *parent)
	:QWidget(parent), SqlService(info, "engines")
{
	QItemSelectionModel	*smodel;

	model = new EngineModel(connection(), this);
	connect(model, SIGNAL(modelChanged()), this, SLOT(selectItem()));

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	view = new QTableView(this);
	connect(view, SIGNAL(doubleClicked(const QModelIndex&)),
		this, SLOT(setDefault(const QModelIndex&)));
	view->setModel(model);
	TABLE_VIEW_INIT(view);
	smodel = view->selectionModel();
	connect(smodel,
		SIGNAL(currentChanged(const QModelIndex&, const QModelIndex&)),
		this,
		SLOT(testEngine(const QModelIndex&, const QModelIndex&)));
	layout->addWidget(view, 1);

	ctl = new QHBoxLayout();
	layout->addLayout(ctl);

	legend = new EngineLegend(this);
	ctl->addWidget(legend, 0, Qt::AlignLeft);

	def = new QPushButton(QIcon(":/icons/engine.png"),
		tr("set as default"), this);
	connect(def, SIGNAL(clicked()), this, SLOT(setDefault()));
	ctl->addWidget(def, 0, Qt::AlignRight);

	selectItem();
}

Engines::~Engines()
{

	delete def;
	delete legend;
	delete ctl;
	delete view;
	delete layout;
	delete model;
}


void
Engines::timeout()
{

	model->timeout();
}


QString
Engines::tabName()
{

	return (tr("Engines"));
}


void
Engines::destroyIt()
{

	delete this;
}


void
Engines::deleteNow()
{

	destroyIt();
}


void
Engines::selectItem()
{
	QItemSelectionModel	*sel = view->selectionModel();

	if ((sel != NULL) && !sel->hasSelection())
		view->setCurrentIndex(model->index(0, 0));

	def->setEnabled(!model->isEmpty());
}


void
Engines::setDefault()
{

	setDefault(view->currentIndex());
}


void
Engines::setDefault(const QModelIndex &ndx)
{
	Engine	*eng = model->engine(ndx);

	if (eng == NULL)
		Core::noData(this);
	else if (eng->support().compare("NO", Qt::CaseInsensitive) == 0)
		QMessageBox::critical(this, tr("Engine error"),
			tr("The storage engine %1 is not supported by the "
			"server and cannot be set as the default engine")
			.arg(eng->engine()));
	else if (eng->support().compare("DEFAULT", Qt::CaseInsensitive) == 0)
		QMessageBox::information(this, tr("Default engine"),
			tr("The storage engine %1 is already set as "
			"the default engine").arg(eng->engine()));
	else
		model->setDefault(eng->engine(), this);
}


void
Engines::testEngine(const QModelIndex &ndx, const QModelIndex&)
{
	Engine	*eng = model->engine(ndx);
	bool	ena = FALSE;

	if (eng != NULL)
		ena = (eng->support().compare("YES",
			Qt::CaseInsensitive) == 0);

	def->setEnabled(ena);
}

//-----------------------------------------------------------------------------

EngineLegend::EngineLegend(QWidget *parent)
	:QGroupBox(parent)
{

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	l_def = Core::addLegendItem(tr("Default engine"), BRUSH_DEFAULT, this);
	layout->addWidget(l_def);

	l_ns = Core::addLegendItem(tr("Not supported engine"),
		BRUSH_NOT_SUPPORTED, this);
	layout->addWidget(l_ns);
}

EngineLegend::~EngineLegend()
{

	delete l_ns;
	delete l_def;
	delete layout;
}

/* EOF */
