/*-
 * Copyright (c) 2013 by SilverSoft.Net
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: priv_database.cpp,v 1.7 2013/09/19 12:40:39 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "priv_database.h"
#include "priv_database_dialog.h"

#define	MY_LEVEL	GrantInfo::G_DATABASE


PrivDatabase::PrivDatabase(SqlConnection *conn, QWidget *parent)
	:PrivItem(conn, tr("Database (GRANT ON DB.*)"), parent), crc(0)
{

}

PrivDatabase::~PrivDatabase()
{

	destroyData();
}


void
PrivDatabase::timeout(const SqlUser &user)
{
	quint64	tmp = checksum(MY_LEVEL);
	bool	changed = (internalUser() != user);

	if ((crc != tmp) || changed) {
		crc = tmp;
		setInternalUser(user);
		if (changed)
			destroyData();
		readPrivileges(user);
	}
}


void
PrivDatabase::commit(QStringList &queries)const
{
	PrivDatabaseVector::const_iterator	it;

	for (it = vect.constBegin(); it != vect.constEnd(); ++it)
		(*it)->commit(queries);
}


void
PrivDatabase::clear()
{

	setInternalUser(SqlUser());
	destroyData();
}


bool
PrivDatabase::isChanged()const
{
	PrivDatabaseVector::const_iterator	it;
	bool					ret = FALSE;

	for (it = vect.constBegin(); it != vect.constEnd(); ++it) {
		ret |= (*it)->isChanged();
		if (ret != FALSE)
			break;
	}

	return (ret);
}


void
PrivDatabase::destroyData()
{
	PrivDatabaseVector::iterator	it;
	PrivDatabaseItem		*ptr;

	for (it = vect.begin(); it != vect.end(); ++it) {
		ptr = *it;
		removeWidget(ptr);
		delete ptr;
	}
	vect.clear();
	dmap.clear();
}


void
PrivDatabase::buttonAction()
{
	PrivDatabaseDialog	*dlg;
	PrivDatabaseItem	*item;

	dlg = new PrivDatabaseDialog(this);
	if (dlg->exec()) {
		QString	level(QString("%1.*").arg(dlg->database()));

		item = dmap.value(level, NULL);
		if (item == NULL)
			item = createItem(level);
		item->setPrivileges(dlg->privileges());
		item->timeout(internalUser());
	}
	delete dlg;
}


void
PrivDatabase::readPrivileges(const SqlUser &user)
{
	GrantInfo			info;
	SqlPrivilegesMap		map;
	SqlPrivilegesMap::iterator	it;
	QString				level;
	SqlPrivilegesFlags		flags;
	PrivDatabaseItem		*item;

	info.type = MY_LEVEL;
	info.host = user.host();
	info.user = user.user();
	map = privileges(info);

	markAll();

	for (it = map.begin(); it != map.end(); ++it) {
		level = it.key();
		flags = it.value();
		item = dmap.value(level, NULL);
		if (item == NULL) {
			item = createItem(level);
			item->setDefault(flags);
			item->timeout(user);
		} else {
			item->setMarked(FALSE);
			item->setDefault(flags.flags());
		}
	}

	removeMarked();
}


void
PrivDatabase::markAll()
{
	PrivDatabaseVector::iterator	it;

	for (it = vect.begin(); it != vect.end(); ++it)
		(*it)->setMarked(TRUE);
}


void
PrivDatabase::removeMarked()
{
	PrivDatabaseVector::iterator	it;
	PrivDatabaseItem		*item;
	QString				level;
	int				ch = 0;

	for (it = vect.begin(); it != vect.end(); ) {
		item = *it;
		if (item->isMarked()) {
			dmap.remove(item->level());
			it = vect.erase(it);
			removeWidget(item);
			delete item;
			++ch;
		} else
			++it;
	}

	if (ch > 0)
		emit changed();
}


PrivDatabaseItem *
PrivDatabase::createItem(const QString &db)
{
	PrivDatabaseItem	*ret;

	ret = new PrivDatabaseItem(db, this);
	connect(ret, SIGNAL(changed()), SIGNAL(changed()));
	vect.append(ret);
	dmap[db] = ret;
	addWidget(ret);

	return (ret);
}

//-----------------------------------------------------------------------------

PrivDatabaseItem::PrivDatabaseItem(const QString &item, QWidget *parent)
	:PrivItem(NULL, item, parent), BasicData()
{

	priv = new Privileges(this);
	connect(priv, SIGNAL(changed()), SIGNAL(changed()));
	priv->setAllowed(PRIV_DB_MASK);
	priv->setDefault(PRIV_USAGE);
	addWidget(priv);
	buttonAction();
}

PrivDatabaseItem::~PrivDatabaseItem()
{

	delete priv;
}


void
PrivDatabaseItem::setDefault(sql_privileges_t flags)
{
	sql_privileges_t	tmp = priv->privileges();
	bool			changed = (tmp != flags);

	if (!changed && priv->hasChanged())
		priv->setDefault(flags);
	else if (!priv->hasChanged()) {
		/* it has not changed by the user */
		if (changed)
			priv->setDefault(flags);
	} else if (changed && question(this)) {
		/* set as default */
		priv->setDefault(flags);
	}
}


void
PrivDatabaseItem::setPrivileges(sql_privileges_t flags)
{

	priv->setPrivileges(flags);
}


void
PrivDatabaseItem::timeout(const SqlUser &user)
{

	setInternalUser(user);
}


void
PrivDatabaseItem::commit(QStringList &queries)const
{

	if (priv->hasChanged()) {
		GrantInfo	info;

		info.type = MY_LEVEL;
		info.user = internalUser().user();
		info.host = internalUser().host();
		info.db = level();
		GrantInfo::commit(queries, info, priv->defaultPrivileges(),
			priv->privileges());
	}
}


void
PrivDatabaseItem::clear()
{

	/* nothing to do here */
}


void
PrivDatabaseItem::buttonAction()
{

	if (priv->isVisible()) {
		priv->hide();
		setButtonType(PrivHeader::BT_PLUS);
	} else {
		priv->show();
		setButtonType(PrivHeader::BT_MINUS);
	}
}


bool
PrivDatabaseItem::isChanged()const
{

	return (priv->hasChanged());
}

/* EOF */
