/*-
 * Copyright (c) 2013 by SilverSoft.Net
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: priv_table.cpp,v 1.2 2013/09/19 12:40:39 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "priv_table.h"
#include "priv_table_dialog.h"

#define	MY_LEVEL	GrantInfo::G_TABLE


PrivTable::PrivTable(SqlConnection *conn, QWidget *parent)
	:PrivItem(conn, tr("Table (GRANT ON DB.Table)"), parent),
	crc(0)
{

}

PrivTable::~PrivTable()
{

	destroyData();
}


void
PrivTable::timeout(const SqlUser &user)
{
	quint64	tmp = checksum(MY_LEVEL);
	bool	changed = (internalUser() != user);

	if ((crc != tmp) || changed) {
		crc = tmp;
		setInternalUser(user);
		if (changed)
			destroyData();
		readPrivileges(user);
	}
}


void
PrivTable::commit(QStringList &queries)const
{
	PrivTableVector::const_iterator	it;

	for (it = vect.constBegin(); it != vect.constEnd(); ++it)
		(*it)->commit(queries);
}


void
PrivTable::clear()
{

	setInternalUser(SqlUser());
	destroyData();
}


bool
PrivTable::isChanged()const
{
	PrivTableVector::const_iterator	it;
	bool				ret = FALSE;

	for (it = vect.constBegin(); it != vect.constEnd(); ++it) {
		ret |= (*it)->isChanged();
		if (ret != FALSE)
			break;
	}

	return (ret);
}


void
PrivTable::destroyData()
{
	PrivTableVector::iterator	it;
	PrivTableItem			*ptr;

	for (it = vect.begin(); it != vect.end(); ++it) {
		ptr = *it;
		removeWidget(ptr);
		delete ptr;
	}
	vect.clear();
	dmap.clear();
}


void
PrivTable::buttonAction()
{
	PrivTableDialog	*dlg;
	PrivTableItem	*item;

	dlg = new PrivTableDialog(this);
	if (dlg->exec()) {
		QString	level(QString("%1.%2").arg(dlg->database())
				.arg(dlg->table()));

		item = dmap.value(level, NULL);
		if (item == NULL)
			item = createItem(level);
		item->setPrivileges(dlg->privileges());
		item->timeout(internalUser());
	}
	delete dlg;
}


void
PrivTable::readPrivileges(const SqlUser &user)
{
	GrantInfo			info;
	SqlPrivilegesMap		map;
	SqlPrivilegesMap::iterator	it;
	QString				level;
	SqlPrivilegesFlags		flags;
	PrivTableItem			*item;

	info.type = MY_LEVEL;
	info.host = user.host();
	info.user = user.user();
	map = privileges(info);

	markAll();

	for (it = map.begin(); it != map.end(); ++it) {
		level = it.key();
		flags = it.value();
		item = dmap.value(level, NULL);
		if (item == NULL) {
			item = createItem(level);
			item->setDefault(flags);
			item->timeout(user);
		} else {
			item->setMarked(FALSE);
			item->setDefault(flags.flags());
		}
	}

	removeMarked();
}


void
PrivTable::markAll()
{
	PrivTableVector::iterator	it;

	for (it = vect.begin(); it != vect.end(); ++it)
		(*it)->setMarked(TRUE);
}


void
PrivTable::removeMarked()
{
	PrivTableVector::iterator	it;
	PrivTableItem			*item;
	QString				level;
	int				ch = 0;

	for (it = vect.begin(); it != vect.end(); ) {
		item = *it;
		if (item->isMarked()) {
			dmap.remove(item->level());
			it = vect.erase(it);
			removeWidget(item);
			delete item;
			++ch;
		} else
			++it;
	}

	if (ch > 0)
		emit changed();
}


PrivTableItem *
PrivTable::createItem(const QString &db)
{
	PrivTableItem	*ret;

	ret = new PrivTableItem(db, this);
	connect(ret, SIGNAL(changed()), SIGNAL(changed()));
	vect.append(ret);
	dmap[db] = ret;
	addWidget(ret);

	return (ret);
}

//-----------------------------------------------------------------------------

PrivTableItem::PrivTableItem(const QString &item, QWidget *parent)
	:PrivItem(NULL, item, parent), BasicData()
{

	priv = new Privileges(this);
	connect(priv, SIGNAL(changed()), SIGNAL(changed()));
	priv->setAllowed(PRIV_TABLE_MASK);
	priv->setDefault(PRIV_USAGE);
	addWidget(priv);
	buttonAction();
}

PrivTableItem::~PrivTableItem()
{

	delete priv;
}


void
PrivTableItem::setDefault(sql_privileges_t flags)
{
	sql_privileges_t	tmp = priv->privileges();
	bool			changed = (tmp != flags);

	if (!changed && priv->hasChanged())
		priv->setDefault(flags);
	else if (!priv->hasChanged()) {
		/* it has not changed by the user */
		if (changed)
			priv->setDefault(flags);
	} else if (changed && question(this)) {
		/* set as default */
		priv->setDefault(flags);
	}
}


void
PrivTableItem::setPrivileges(sql_privileges_t flags)
{

	priv->setPrivileges(flags);
}


void
PrivTableItem::timeout(const SqlUser &user)
{

	setInternalUser(user);
}


void
PrivTableItem::commit(QStringList &queries)const
{

	if (priv->hasChanged()) {
		GrantInfo	info;
		QStringList	list(level().split('.',
					QString::SkipEmptyParts));

		info.type = MY_LEVEL;
		info.user = internalUser().user();
		info.host = internalUser().host();
		info.db = list[0];
		info.table = list[1];
		GrantInfo::commit(queries, info, priv->defaultPrivileges(),
			priv->privileges());
	}
}


void
PrivTableItem::clear()
{

	/* nothing to do here */
}


void
PrivTableItem::buttonAction()
{

	if (priv->isVisible()) {
		priv->hide();
		setButtonType(PrivHeader::BT_PLUS);
	} else {
		priv->show();
		setButtonType(PrivHeader::BT_MINUS);
	}
}


bool
PrivTableItem::isChanged()const
{

	return (priv->hasChanged());
}

/* EOF */
