/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: privileges.cpp,v 1.7 2013/09/19 12:40:39 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "privileges.h"

#define	ADD_GRID(box, name, tip)	do {				\
	(box) = new QCheckBox(name, this);				\
	(box)->setFocusPolicy(Qt::NoFocus);				\
	(box)->setToolTip(tip);						\
	(box)->setEnabled(FALSE);					\
	grid->addWidget(box, row, col++);				\
	connect(box, SIGNAL(stateChanged(int)),				\
		this, SLOT(privChanged(int)));				\
} while (0)

#define	SET_PRIVILEGE(priv, on)	do {					\
	if ((priv)->isEnabled())					\
		(priv)->setChecked(on);					\
} while (0)

#define	IF_SET(box, priv, flag)						\
	SET_PRIVILEGE(box, (priv).isSet(flag))

#define	IF_ADD(target, box, flag)	do {				\
	if ((box)->isEnabled() && (box)->isChecked())			\
		(target) += (flag);					\
	else								\
		(target) -= (flag);					\
} while (0)

#define	CHANGE_ALL(on)	do {						\
	SET_PRIVILEGE(p_sel, on);					\
	SET_PRIVILEGE(p_ins, on);					\
	SET_PRIVILEGE(p_upd, on);					\
	SET_PRIVILEGE(p_del, on);					\
	SET_PRIVILEGE(p_crt, on);					\
	SET_PRIVILEGE(p_drp, on);					\
	SET_PRIVILEGE(p_rel, on);					\
	SET_PRIVILEGE(p_shd, on);					\
	SET_PRIVILEGE(p_pro, on);					\
	SET_PRIVILEGE(p_fil, on);					\
	SET_PRIVILEGE(p_ref, on);					\
	SET_PRIVILEGE(p_ndx, on);					\
	SET_PRIVILEGE(p_alt, on);					\
	SET_PRIVILEGE(p_sdb, on);					\
	SET_PRIVILEGE(p_sup, on);					\
	SET_PRIVILEGE(p_ctt, on);					\
	SET_PRIVILEGE(p_ltb, on);					\
	SET_PRIVILEGE(p_exe, on);					\
	SET_PRIVILEGE(p_rps, on);					\
	SET_PRIVILEGE(p_rpc, on);					\
	SET_PRIVILEGE(p_crv, on);					\
	SET_PRIVILEGE(p_shv, on);					\
	SET_PRIVILEGE(p_crr, on);					\
	SET_PRIVILEGE(p_alr, on);					\
	SET_PRIVILEGE(p_cru, on);					\
	SET_PRIVILEGE(p_evt, on);					\
	SET_PRIVILEGE(p_trg, on);					\
	SET_PRIVILEGE(p_cts, on);					\
} while (0)


Privileges::Privileges(QWidget *parent)
	:QGroupBox(parent), p_allowed(PRIV_MASK), p_default(PRIV_USAGE)
{

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	grid = new QGridLayout();
	layout->addLayout(grid);

	dlg = new QDialogButtonBox(Qt::Horizontal, this);
	dlg->setEnabled(FALSE);
	layout->addWidget(dlg, 0, Qt::AlignBottom);

	setAll = new QPushButton(QIcon(":/icons/add.png"),
		tr("Grant all"), this);
	setAll->setToolTip(tr("Push this button to grant all privileges"));
	setAll->setFocusPolicy(Qt::NoFocus);
	connect(setAll, SIGNAL(clicked()), this, SLOT(grantAll()));
	dlg->addButton(setAll, QDialogButtonBox::ActionRole);

	unsetAll = new QPushButton(QIcon(":/icons/delete.png"),
		tr("Revoke all"), this);
	unsetAll->setToolTip(tr("Push this button to revoke all privileges"));
	unsetAll->setFocusPolicy(Qt::NoFocus);
	connect(unsetAll, SIGNAL(clicked()), this, SLOT(revokeAll()));
	dlg->addButton(unsetAll, QDialogButtonBox::ActionRole);

	revert = new QPushButton(tr("Revert"), this);
	revert->setToolTip(tr("Push this button to revert the privileges"));
	revert->setFocusPolicy(Qt::NoFocus);
	connect(revert, SIGNAL(clicked()), this, SLOT(revertPrivileges()));
	dlg->addButton(revert, QDialogButtonBox::ActionRole);

	initRows();
}

Privileges::~Privileges()
{

	delete p_cts;
	delete p_trg;
	delete p_evt;
	delete p_cru;
	delete p_alr;
	delete p_crr;
	delete p_shv;
	delete p_crv;
	delete p_rpc;
	delete p_rps;
	delete p_exe;
	delete p_ltb;
	delete p_ctt;
	delete p_sup;
	delete p_sdb;
	delete p_alt;
	delete p_ndx;
	delete p_ref;
	delete p_grt;
	delete p_fil;
	delete p_pro;
	delete p_shd;
	delete p_rel;
	delete p_drp;
	delete p_crt;
	delete p_del;
	delete p_upd;
	delete p_ins;
	delete p_sel;
	delete revert;
	delete unsetAll;
	delete setAll;
	delete dlg;
	delete grid;
	delete layout;
}


void
Privileges::initRows()
{
	int	row = 0;

	initRow1(row);
	initRow2(row);
	initRow3(row);
	initRow4(row);
	initRow5(row);
	initRow6(row);
	initRow7(row);
	initRow8(row);

	privChanged(0);
}


void
Privileges::initRow1(int &row)
{
	int	col = 0;

	ADD_GRID(p_sel, tr("Select"), tr("The <u><b>SELECT</b></u> privilege "
		"enables you to select rows from tables in a database. "
		"<u><b>SELECT</b></u> statements require the "
		"<u><b>SELECT</b></u> privilege only if they actually "
		"retrieve rows from a table. Some <u><b>SELECT</b></u> "
		"statements do not access tables and can be executed without "
		"permission for any database. For example, you can use "
		"<u><b>SELECT</b></u> as a simple calculator to evaluate "
		"expressions that make no reference to tables:"
		"<pre>"
		"SELECT 1+1;\n"
		"SELECT PI()*2;\n"
		"</pre>"
		"The <u><b>SELECT</b></u> privilege is also needed for other "
		"statements that read column values. For example, "
		"<u><b>SELECT</b></u> is needed for columns referenced on the "
		"right hand side of <b>col_name=expr</b> assignment in "
		"<u><b>UPDATE</b></u> statements or for columns named in the "
		"<b>WHERE</b> clause of <u><b>DELETE</b></u> or "
		"<u><b>UPDATE</b></u> statements"));
	ADD_GRID(p_ins, tr("Insert"), tr("The <u><b>INSERT</b></u> privilege "
		"enables rows to be inserted into tables in a database. "
		"<u><b>INSERT</b></u> is also required for the "
		"<u><b>ANALYZE TABLE</b></u>, <u><b>OPTIMIZE TABLE</b></u>, "
		"and <u><b>REPAIR TABLE</b></u> table-maintenance statements"));
	ADD_GRID(p_upd, tr("Update"), tr("The <u><b>UPDATE</b></u> privilege "
		"enables rows to be updated in tables in a database"));
	ADD_GRID(p_del, tr("Delete"), tr("The <u><b>DELETE</b></u> privilege "
		"enables rows to be deleted from tables in a database"));
	++row;
}


void
Privileges::initRow2(int &row)
{
	int	col = 0;

	ADD_GRID(p_crt, tr("Create"), tr("The <u><b>CREATE</b></u> privilege "
		"enables creation of new databases and tables"));
	ADD_GRID(p_drp, tr("Drop"), tr("The <u><b>DROP</b></u> privilege "
		"enables you to drop (remove) existing databases, tables, and "
		"views. The <u><b>DROP</b></u> privilege is required in order "
		"to use the statement <u><b>ALTER TABLE ... DROP "
		"PARTITION</b></u> on a partitioned table. "
		"The <u><b>DROP</b></u> privilege is also required for "
		"<u><b>TRUNCATE TABLE</b></u>. <i>If you grant the "
		"<u><b>DROP</b></u> privilege for the mysql database to a "
		"user, that user can drop the database in which the MySQL "
		"access privileges are stored</i>"));
	ADD_GRID(p_rel, tr("Reload"), tr("The <u><b>RELOAD</b></u> privilege "
		"enables use of the <u><b>FLUSH</b></u> statement. It also "
		"enables <u>mysqladmin</u> commands that are equivalent to "
		"<u><b>FLUSH</b></u> operations: <b>flush-hosts</b>, "
		"<b>flush-logs</b>, <b>flush-privileges</b>, "
		"<b>flush-status</b>, <b>flush-tables</b>, "
		"<b>flush-threads</b>, <b>refresh</b>, and <b>reload</b>"));
	ADD_GRID(p_shd, tr("Shutdown"), tr("The <u><b>SHUTDOWN</b></u> "
		"privilege enables use of the "
		"<u><b>mysqladmin shutdown</b></u> command. There is no "
		"corresponding SQL statement"));
	++row;
}


void
Privileges::initRow3(int &row)
{
	int	col = 0;

	ADD_GRID(p_pro, tr("Process"), tr("The <u><b>PROCESS</b></u> "
		"privilege pertains to display of information about the "
		"threads executing within the server (that is, information "
		"about the statements being executed by sessions). The "
		"privilege enables use of <u><b>SHOW PROCESSLIST</b></u> or "
		"<u>mysqladmin processlist</u> to see threads belonging to "
		"other accounts; you can always see your own threads"));
	ADD_GRID(p_fil, tr("File"), tr("The <u><b>FILE</b></u> privilege "
		"gives you permission to read and write files on the server "
		"host using the <u><b>LOAD DATA INFILE</b></u> and "
		"<u><b>SELECT ... INTO OUTFILE</b></u> statements and the "
		"<u><b>LOAD_FILE()</b></u> function. A user who has the "
		"<u><b>FILE</b></u> privilege can read any file on the server "
		"host that is either world-readable or readable by the MySQL "
		"server. (This implies the user can read any file in any "
		"database directory, because the server can access any of "
		"those files.) The <u><b>FILE</b></u> privilege also enables "
		"the user to create new files in any directory where the "
		"MySQL server has write access. As a security measure, the "
		"server will not overwrite existing files"));
	ADD_GRID(p_grt, tr("Grant"), tr("The <u><b>GRANT OPTION</b></u> "
		"privilege enables you to give to other users or remove from "
		"other users those privileges that you yourself possess"));
	ADD_GRID(p_ref, tr("References"), tr("The <u><b>REFERENCES</b></u> "
		"privilege currently is unused"));
	++row;
}


void
Privileges::initRow4(int &row)
{
	int	col = 0;

	ADD_GRID(p_ndx, tr("Index"), tr("The <u><b>INDEX</b></u> privilege "
		"enables you to create or drop (remove) indexes. "
		"<u><b>INDEX</b></u> applies to existing tables. If you have "
		"the <u><b>CREATE</b></u> privilege for a table, you can "
		"include index definitions in the <u><b>CREATE TABLE</b></u> "
		"statement"));
	ADD_GRID(p_alt, tr("Alter"), tr("The <u><b>ALTER</b></u> privilege "
		"enables use of <u><b>ALTER TABLE</b></u> to change the "
		"structure of tables. <u><b>ALTER TABLE</b></u> also requires "
		"the <u><b>CREATE</b></u> and <u><b>INSERT</b></u> "
		"privileges. Renaming a table requires <u><b>ALTER</b></u> "
		"and <u><b>DROP</b></u> on the old table, <u><b>ALTER</b></u>,"
		" <u><b>CREATE</b></u>, and <u><b>INSERT</b></u> on the new "
		"table"));
	ADD_GRID(p_sdb, tr("Show databases"), tr("The "
		"<u><b>SHOW DATABASES</b></u> privilege enables the account "
		"to see database names by issuing the <b>SHOW DATABASE</b> "
		"statement. Accounts that do not have this privilege see only "
		"databases for which they have some privileges, and cannot "
		"use the statement at all if the server was started with the "
		"<u><b>--skip-show-database</b></u> option. Note that "
		"<i>any</i> global privilege is a privilege for the "
		"database"));
	ADD_GRID(p_sup, tr("Super"), tr("The <u><b>SUPER</b></u> privilege "
		"enables an account to use <u><b>CHANGE MASTER TO</b></u>, "
		"<u><b>KILL</b></u> or <u><b>mysqladmin kill</b></u> to kill "
		"threads belonging to other accounts (you can always kill "
		"your own threads), <u><b>PURGE BINARY LOGS</b></u>, "
		"configuration changes using <u><b>SET GLOBAL</b></u> to "
		"modify global system variables, the <b>mysqladmin debug</b> "
		"command, enabling or disabling logging, performing updates "
		"even if the <u><b>read_only</b></u> system variable is "
		"enabled, starting and stopping replication on slave servers, "
		"specification of any account in the <b>DEFINER</b> attribute "
		"of stored programs and views, and enables you to connect "
		"(once) even if the connection limit controlled by the "
		"<u><b>max_connections</b></u> system variable is reached.<p>"
		"To create or alter stored functions if binary logging is "
		"enabled, you may also need the <u><b>SUPER</b></u> "
		"privilege"));
	++row;
}


void
Privileges::initRow5(int &row)
{
	int	col = 0;

	ADD_GRID(p_ctt, tr("Create tmp table"), tr("The "
		"<u><b>CREATE TMP TABLES</b></u> privilege enables the "
		"creation of temporary tables using the "
		"<u><b>CREATE TEMPORARY TABLE</b></u> statement"));
	ADD_GRID(p_ltb, tr("Lock tables"), tr("The <u><b>LOCK TABLES</b></u> "
		"privilege enables the use of explicit "
		"<u><b>LOCK TABLES</b></u> statements to lock tables for "
		"which you have the <u><b>SELECT</b></u> privilege. This "
		"includes the use of write locks, which prevents other "
		"sessions from reading the locked table"));
	ADD_GRID(p_exe, tr("Execute"), tr("The <u><b>EXECUTE</b></u> "
		"privilege is required to execute stored routines "
		"(procedures and functions)"));
	ADD_GRID(p_rps, tr("Replication slave"), tr("The "
		"<u><b>REPLICATION SLAVE</b></u> privilege should be granted "
		"to accounts that are used by slave servers to connect to the "
		"current server as their master. Without this privilege, the "
		"slave cannot request updates that have been made to "
		"databases on the master server"));
	++row;
}


void
Privileges::initRow6(int &row)
{
	int	col = 0;

	ADD_GRID(p_rpc, tr("Replication client"), tr("The "
		"<u><b>REPLICATION CLIENT</b></u> privilege enables the use "
		"of <u><b>SHOW MASTER STATUS</b></u> and "
		"<u><b>SHOW SLAVE STATUS</b></u>"));
	ADD_GRID(p_crv, tr("Create view"), tr("The <u><b>CREATE VIEW</b></u> "
		"privilege enables use of <u><b>CREATE VIEW</b></u>"));
	ADD_GRID(p_shv, tr("Show view"), tr("The <u><b>SHOW VIEW</b></u> "
		"privilege enables use of <u><b>SHOW CREATE VIEW</b></u>"));
	ADD_GRID(p_crr, tr("Create routine"), tr("The "
		"<u><b>CREATE ROUTINE</b></u> privilege is needed to create "
		"stored routines (procedures and functions)"));
	++row;
}


void
Privileges::initRow7(int &row)
{
	int	col = 0;

	ADD_GRID(p_alr, tr("Alter routine"), tr("The "
		"<u><b>ALTER ROUTINE</b></u> privilege is needed to alter or "
		"drop stored routines (procedures and functions)"));
	ADD_GRID(p_cru, tr("Create user"), tr("The <u><b>CREATE USER</b></u> "
		"privilege enables use of <u><b>CREATE USER</b></u>, "
		"<u><b>DROP USER</b></u>, <u><b>RENAME USER</b></u>, and "
		"<u><b>REVOKE ALL PRIVILEGES</b></u>"));
	ADD_GRID(p_evt, tr("Event"), tr("The <u><b>EVENT</b></u> privilege is "
		"required to create, alter, drop, or see events for the Event "
		"Scheduler"));
	ADD_GRID(p_trg, tr("Trigger"), tr("The <u><b>TRIGGER</b></u> "
		"privilege enables trigger operations. You must have this "
		"privilege for a table to create, drop, or execute triggers "
		"for that table"));
	++row;
}


void
Privileges::initRow8(int &row)
{
	int	col = 0;

	ADD_GRID(p_cts, tr("Create tablespace"), tr("The "
		"<u><b>CREATE TABLESPACE</b></u> privilege is needed to "
		"create, alter, or drop tablespaces and log file groups"));
	++row;
}


void
Privileges::setData(const SqlPrivilegesFlags &priv)
{

	IF_SET(p_sel, priv, PRIV_SELECT);
	IF_SET(p_ins, priv, PRIV_INSERT);
	IF_SET(p_upd, priv, PRIV_UPDATE);
	IF_SET(p_del, priv, PRIV_DELETE);
	IF_SET(p_crt, priv, PRIV_CREATE);
	IF_SET(p_drp, priv, PRIV_DROP);
	IF_SET(p_rel, priv, PRIV_RELOAD);
	IF_SET(p_shd, priv, PRIV_SHUTDOWN);
	IF_SET(p_pro, priv, PRIV_PROCESS);
	IF_SET(p_fil, priv, PRIV_FILE);
	IF_SET(p_grt, priv, PRIV_GRANT);
	IF_SET(p_ref, priv, PRIV_REFERENCES);
	IF_SET(p_ndx, priv, PRIV_INDEX);
	IF_SET(p_alt, priv, PRIV_ALTER);
	IF_SET(p_sdb, priv, PRIV_SHOW_DB);
	IF_SET(p_sup, priv, PRIV_SUPER);
	IF_SET(p_ctt, priv, PRIV_CREATE_TMP_TABLE);
	IF_SET(p_ltb, priv, PRIV_LOCK_TABLES);
	IF_SET(p_exe, priv, PRIV_EXECUTE);
	IF_SET(p_rps, priv, PRIV_REPL_SLAVE);
	IF_SET(p_rpc, priv, PRIV_REPL_CLIENT);
	IF_SET(p_crv, priv, PRIV_CREATE_VIEW);
	IF_SET(p_shv, priv, PRIV_SHOW_VIEW);
	IF_SET(p_crr, priv, PRIV_CREATE_ROUTINE);
	IF_SET(p_alr, priv, PRIV_ALTER_ROUTINE);
	IF_SET(p_cru, priv, PRIV_CREATE_USER);
	IF_SET(p_evt, priv, PRIV_EVENT);
	IF_SET(p_trg, priv, PRIV_TRIGGER);
	IF_SET(p_cts, priv, PRIV_CREATE_TABLESPACE);
	privChanged(0);
}


void
Privileges::grantAll()
{

	CHANGE_ALL(TRUE);
	privChanged(0);
}


void
Privileges::revokeAll()
{

	CHANGE_ALL(FALSE);
	SET_PRIVILEGE(p_grt, FALSE);
	privChanged(0);
}


void
Privileges::revertPrivileges()
{

	if (p_default != privileges()) {
		setData(p_default);
		privChanged(0);
	}
}


void
Privileges::privChanged(int)
{
	SqlPrivilegesFlags	curr(privileges());
	sql_privileges_t	priv = curr.flags() & ~PRIV_GRANT,
				mask = (PRIV_MASK & ~PRIV_GRANT);

	revert->setEnabled((p_default != curr));
	unsetAll->setEnabled(curr.flags() != PRIV_USAGE);
	setAll->setEnabled(priv ^ mask);
	emit changed();
}


void
Privileges::setAllowed(sql_privileges_t spf)
{

	if (p_allowed.flags() != spf)
		p_allowed = spf;

	p_sel->setEnabled(p_allowed.isSet(PRIV_SELECT));
	p_ins->setEnabled(p_allowed.isSet(PRIV_INSERT));
	p_upd->setEnabled(p_allowed.isSet(PRIV_UPDATE));
	p_del->setEnabled(p_allowed.isSet(PRIV_DELETE));
	p_crt->setEnabled(p_allowed.isSet(PRIV_CREATE));
	p_drp->setEnabled(p_allowed.isSet(PRIV_DROP));
	p_rel->setEnabled(p_allowed.isSet(PRIV_RELOAD));
	p_shd->setEnabled(p_allowed.isSet(PRIV_SHUTDOWN));
	p_pro->setEnabled(p_allowed.isSet(PRIV_PROCESS));
	p_fil->setEnabled(p_allowed.isSet(PRIV_FILE));
	p_grt->setEnabled(p_allowed.isSet(PRIV_GRANT));
	p_ref->setEnabled(p_allowed.isSet(PRIV_REFERENCES));
	p_ndx->setEnabled(p_allowed.isSet(PRIV_INDEX));
	p_alt->setEnabled(p_allowed.isSet(PRIV_ALTER));
	p_sdb->setEnabled(p_allowed.isSet(PRIV_SHOW_DB));
	p_sup->setEnabled(p_allowed.isSet(PRIV_SUPER));
	p_ctt->setEnabled(p_allowed.isSet(PRIV_CREATE_TMP_TABLE));
	p_ltb->setEnabled(p_allowed.isSet(PRIV_LOCK_TABLES));
	p_exe->setEnabled(p_allowed.isSet(PRIV_EXECUTE));
	p_rps->setEnabled(p_allowed.isSet(PRIV_REPL_SLAVE));
	p_rpc->setEnabled(p_allowed.isSet(PRIV_REPL_CLIENT));
	p_crv->setEnabled(p_allowed.isSet(PRIV_CREATE_VIEW));
	p_shv->setEnabled(p_allowed.isSet(PRIV_SHOW_VIEW));
	p_crr->setEnabled(p_allowed.isSet(PRIV_CREATE_ROUTINE));
	p_alr->setEnabled(p_allowed.isSet(PRIV_ALTER_ROUTINE));
	p_cru->setEnabled(p_allowed.isSet(PRIV_CREATE_USER));
	p_evt->setEnabled(p_allowed.isSet(PRIV_EVENT));
	p_trg->setEnabled(p_allowed.isSet(PRIV_TRIGGER));
	p_cts->setEnabled(p_allowed.isSet(PRIV_CREATE_TABLESPACE));

	dlg->setEnabled(p_allowed.flags() != 0);
}


void
Privileges::setDefault(sql_privileges_t def)
{

	p_default = def;
	setData(p_default);
}


void
Privileges::setPrivileges(sql_privileges_t flags)
{

	setData(flags);
}


SqlPrivilegesFlags
Privileges::privileges()const
{
	SqlPrivilegesFlags	ret(PRIV_USAGE);

	IF_ADD(ret, p_sel, PRIV_SELECT);
	IF_ADD(ret, p_ins, PRIV_INSERT);
	IF_ADD(ret, p_upd, PRIV_UPDATE);
	IF_ADD(ret, p_del, PRIV_DELETE);
	IF_ADD(ret, p_crt, PRIV_CREATE);
	IF_ADD(ret, p_drp, PRIV_DROP);
	IF_ADD(ret, p_rel, PRIV_RELOAD);
	IF_ADD(ret, p_shd, PRIV_SHUTDOWN);
	IF_ADD(ret, p_pro, PRIV_PROCESS);
	IF_ADD(ret, p_fil, PRIV_FILE);
	IF_ADD(ret, p_grt, PRIV_GRANT);
	IF_ADD(ret, p_ref, PRIV_REFERENCES);
	IF_ADD(ret, p_ndx, PRIV_INDEX);
	IF_ADD(ret, p_alt, PRIV_ALTER);
	IF_ADD(ret, p_sdb, PRIV_SHOW_DB);
	IF_ADD(ret, p_sup, PRIV_SUPER);
	IF_ADD(ret, p_ctt, PRIV_CREATE_TMP_TABLE);
	IF_ADD(ret, p_ltb, PRIV_LOCK_TABLES);
	IF_ADD(ret, p_exe, PRIV_EXECUTE);
	IF_ADD(ret, p_rps, PRIV_REPL_SLAVE);
	IF_ADD(ret, p_rpc, PRIV_REPL_CLIENT);
	IF_ADD(ret, p_crv, PRIV_CREATE_VIEW);
	IF_ADD(ret, p_shv, PRIV_SHOW_VIEW);
	IF_ADD(ret, p_crr, PRIV_CREATE_ROUTINE);
	IF_ADD(ret, p_alr, PRIV_ALTER_ROUTINE);
	IF_ADD(ret, p_cru, PRIV_CREATE_USER);
	IF_ADD(ret, p_evt, PRIV_EVENT);
	IF_ADD(ret, p_trg, PRIV_TRIGGER);
	IF_ADD(ret, p_cts, PRIV_CREATE_TABLESPACE);

	return (ret);
}


SqlPrivilegesFlags
Privileges::defaultPrivileges()const
{

	return (p_default);
}


bool
Privileges::hasChanged()const
{

	return (privileges() != defaultPrivileges());
}

/* EOF */
