/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: user.cpp,v 1.3 2013/09/03 10:28:30 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "user.h"
#include "sql_connection.h"
#include "sql_error.h"


User::User()
	:SqlUser(), BasicData()
{

	setSslType(SSL_NONE);
	setCipher(QString::null);
	setX509Issuer(QString::null);
	setX509Subject(QString::null);
	setMaxQuestions(0);
	setMaxUpdates(0);
	setMaxConnections(0);
	setMaxUserConnections(0);
}

User::User(const User &user)
	:SqlUser(user), BasicData()
{

	setUser(user);
}

User::~User()
{
}


void
User::setUser(const User &user)
{

	setSslType(user.sslType());
	setCipher(user.cipher());
	setX509Issuer(user.x509Issuer());
	setX509Subject(user.x509Subject());
	setMaxQuestions(user.maxQuestions());
	setMaxUpdates(user.maxUpdates());
	setMaxConnections(user.maxConnections());
	setMaxUserConnections(user.maxUserConnections());
	setInternalFlags(user.flags());
}


void
User::setUser(const QString &u)
{

	SqlUser::setUser(u);
}


void
User::setSslType(SslType st)
{

	p_ssl = st;
}


void
User::setSslType(const QString &type)
{

	setSslType(str2type(type));
}


void
User::setCipher(const QString &c)
{

	p_cipher = c;
}


void
User::setX509Issuer(const QString &i)
{

	p_x509_issuer = i;
}


void
User::setX509Subject(const QString &s)
{

	p_x509_subject = s;
}


void
User::setMaxQuestions(quint64 mq)
{

	p_mq = mq;
}


void
User::setMaxUpdates(quint64 mu)
{

	p_mu = mu;
}


void
User::setMaxConnections(quint64 mc)
{

	p_mc = mc;
}


void
User::setMaxUserConnections(qint64 muc)
{

	p_muc = muc;
}


void
User::create(SqlConnection *conn, SqlError &err)const
{
	QString	q("CREATE USER " + account(TRUE));

	err = conn->exec(q);
	if (err == 0)
		grant(conn, err);
}


void
User::grant(SqlConnection *conn, SqlError &err, const User &old)const
{
	QString	q("GRANT USAGE ON *.* TO %1 "), ssl;
	bool	mq, mu, mc, muc;

	if (password() != old.password())
		q += QString(" IDENTIFIED BY '%1' ").arg(password());

	if (p_ssl != old.sslType()) {
		switch (p_ssl) {
			case SSL_NONE:
				ssl = "NONE";
				break;
			case SSL_ANY:
				ssl = "SSL";
				break;
			case SSL_X509:
				ssl = "X509";
				break;
			case SSL_SPEC:
				ssl = "SPECIFIED";
				break;
		}
		q += QString("REQUIRE %1 ").arg(ssl);
		if (p_cipher != old.cipher())
			q += QString("CIPHER '%1' ").arg(p_cipher);
		if (p_x509_issuer != old.x509Issuer())
			q += QString("ISSUER '%1' ").arg(p_x509_issuer);
		if (p_x509_subject != old.x509Subject())
			q += QString("SUBJECT '%1' ").arg(p_x509_subject);
	}

	mq = (p_mq != old.maxQuestions());
	mu = (p_mu != old.maxUpdates());
	mc = (p_mc != old.maxConnections());
	muc = (p_muc != old.maxUserConnections());
	if (mq || mu || mc || muc) {
		q += "WITH ";
		if (mq)
			q += QString("MAX_QUERIES_PER_HOUR %1 ").arg(p_mq);
		if (mu)
			q += QString("MAX_UPDATES_PER_HOUR %1 ").arg(p_mu);
		if (mc)
			q += QString("MAX_CONNECTIONS_PER_HOUR %1 ").arg(p_mc);
		if (muc)
			q += QString("MAX_USER_CONNECTIONS %1 ").arg(p_muc);
	}
	err = conn->exec(q.arg(account(TRUE)));
}


User::SslType
User::sslType()const
{

	return (p_ssl);
}


QString
User::cipher()const
{

	return (p_cipher);
}


QString
User::x509Issuer()const
{

	return (p_x509_issuer);
}


QString
User::x509Subject()const
{

	return (p_x509_subject);
}


quint64
User::maxQuestions()const
{

	return (p_mq);
}


quint64
User::maxUpdates()const
{

	return (p_mu);
}


quint64
User::maxConnections()const
{

	return (p_mc);
}


qint64
User::maxUserConnections()const
{

	return (p_muc);
}


QString
User::type2str(SslType t)
{
	QString	ret;

	switch (t) {
		case SSL_NONE:
			ret = QObject::tr("None");
			break;
		case SSL_ANY:
			ret = QObject::tr("Any");
			break;
		case SSL_X509:
			ret = QObject::tr("X509");
			break;
		case SSL_SPEC:
			ret = QObject::tr("Specified");
			break;
	}

	return (ret);
}


User::SslType
User::str2type(const QString &t)
{
	SslType	ret;

	if (t.compare("ANY", Qt::CaseInsensitive) == 0)
		ret = SSL_ANY;
	else if (t.compare("X509", Qt::CaseInsensitive) == 0)
		ret = SSL_X509;
	else if (t.compare("SPECIFIED", Qt::CaseInsensitive) == 0)
		ret = SSL_SPEC;
	else
		ret = SSL_NONE;

	return (ret);
}

/* EOF */
