/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: user_list.cpp,v 1.6 2013/09/10 11:32:08 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "user_list.h"
#include "sql_connection.h"
#include "sql_error.h"
#include "user_model.h"
#include "user_dialog.h"


UserList::UserList(SqlConnection *conn, QWidget *parent)
	:QWidget(parent)
{
	QItemSelectionModel	*smodel;

	model = new UserModel(conn, this);
	connect(model, SIGNAL(modelChanged()), this, SLOT(testUser()));

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	view = new QTableView(this);
	connect(view, SIGNAL(doubleClicked(const QModelIndex&)),
		this, SLOT(editUser(const QModelIndex&)));
	connect(view, SIGNAL(pressed(const QModelIndex&)),
		this, SLOT(changeAccount(const QModelIndex&)));
	view->setModel(model);
	layout->addWidget(view, 1);

	TABLE_VIEW_INIT(view);
	smodel = view->selectionModel();
	connect(smodel,
		SIGNAL(currentChanged(const QModelIndex&, const QModelIndex&)),
		this,
		SLOT(testUser(const QModelIndex&, const QModelIndex&)));
}

UserList::~UserList()
{

	delete view;
	delete layout;
	delete model;
}


void
UserList::timeout()
{

	model->timeout();
}


void
UserList::addUser()
{
	UserDialog	*dlg;
	User		*ptr, usr;
	SqlError	err;
	bool		add = FALSE;

	dlg = new UserDialog(this);
	FATAL(dlg == NULL);
	dlg->setWindowTitle(tr("A new SQL account"));

	if (dlg->exec()) {
		dlg->getData(usr);
		add = TRUE;
	}

	delete dlg;
	if (add) {
		ptr = model->user(usr.account());
		if (ptr == NULL) {
			/* no such account, create it */
			model->createUser(usr, err);
			if (err != 0)
				err.show(this);
		} else if (QMessageBox::question(this, tr("Account exists"),
			tr("The account %1 already exists.\n"
			"Do you want to set password/limits?")
			.arg(usr.account(FALSE)),
			QMessageBox::Yes | QMessageBox::No, QMessageBox::No) ==
			QMessageBox::Yes) {
			model->updateUser(usr, err, User());
			if (err != 0)
				err.show(this);
		}
	}
}


void
UserList::delUser()
{
	User		*usr = model->user(view->currentIndex());
	QString		account, acc;
	SqlError	err;
	Confirm		q, w;

	if (usr == NULL)
		Core::noData(this);
	else {
		account = usr->account(FALSE);
		acc = usr->account(TRUE);
		q.title = QString::null;
		q.text = tr("Do you want to drop account %1?\n").arg(account);
		w.title = QString::null;
		w.text = tr("The user %1 will not be able to connect the "
			"server.\nWould you like to continue?").arg(account);
		if (Core::doubleConfirm(q, w, this)) {
			model->deleteUser(acc, err);
			if (err == 0)
				QMessageBox::information(this, tr("Amen"),
					tr("Account %1 has been dropped")
					.arg(account));
			else
				err.show(this);
		}
	}
}


void
UserList::editUser()
{
	QModelIndex	ndx(view->currentIndex());

	if (ndx.isValid())
		editUser(ndx);
}


void
UserList::editUser(const QModelIndex &ndx)
{
	UserDialog	*dlg;
	User		*ptr = model->user(ndx), usr, save;
	SqlError	err;
	bool		update = FALSE;

	if (ptr != NULL) {
		save = *ptr;
		dlg = new UserDialog(this);
		FATAL(dlg == NULL);
		dlg->setWindowTitle(tr("Edit account"));
		dlg->setData(save);

		if (dlg->exec()) {
			dlg->getData(usr);
			update = TRUE;
		}

		delete dlg;
	}

	if (update == TRUE) {
		if (usr.account() != save.account()) {
			/* changed */
			ptr = model->user(usr.account());
			if (ptr == NULL) {
				/* no such account */
				model->renameUser(save, usr, err);
				if (err != 0)
					err.show(this);
			} else if (QMessageBox::question(this,
				tr("Account exists"),
				tr("The account %1 already exists.\n"
				"Do you want to change password and/or "
				"limits?").arg(usr.account(FALSE)),
				QMessageBox::Yes | QMessageBox::No,
				QMessageBox::No) == QMessageBox::Yes) {
				goto HAXEP;
			}
		} else {
			/* just change the limits */
			HAXEP:
			model->updateUser(usr, err, save);
			if (err != 0)
				err.show(this);
		}
	}
}


void
UserList::testUser(const QModelIndex &ndx, const QModelIndex&)
{

	emit selected(ndx.isValid());
}


void
UserList::testUser()
{
	QItemSelectionModel	*smodel = view->selectionModel();
	QModelIndex		ndx, tmp;

	if ((smodel != NULL) && !smodel->hasSelection())
		ndx = model->index(0, 0);
	else if ((tmp = model->indexOfNew()).isValid())
		ndx = tmp;

	if (ndx.isValid()) {
		view->setCurrentIndex(ndx);
		changeAccount(view->currentIndex());
	}
}


void
UserList::changeAccount(const QModelIndex &ndx)
{
	User	*ptr = model->user(ndx);
	SqlUser	usr;

	if (ptr != NULL) {
		usr = *ptr;
		emit activated(usr);
	}
}

/* EOF */
