/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: user_model.cpp,v 1.7 2013/09/03 10:28:30 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "user_model.h"
#include "sql_connection.h"
#include "sql_query.h"

#define	LAB_ACCOUNT	0	/* user@host */
#define	LAB_PASSWORD	1	/* Yes/No */
#define	LAB_SSL		2	/* ssl_type */
#define	LAB_TOTAL	(LAB_SSL + 1)


UserModel::UserModel(SqlConnection *conn, QObject *parent)
	:QAbstractTableModel(parent)
{

	checksum = 0;
	systable = Core::systemTable("user");
	query = new SqlQuery(conn);
	query->prepare(QString("select User, Host, Password, ssl_type, "
		"ssl_cipher, x509_issuer, x509_subject, max_questions, "
		"max_updates, max_connections, max_user_connections "
		"from %1").arg(systable));
	query->connect(this, SLOT(reloadData()));
}

UserModel::~UserModel()
{
	UserVector::iterator	it;
	User			*ptr;

	for (it = vect.begin(); it != vect.end(); ++it) {
		ptr = *it;
		delete ptr;
	}
}


QVariant
UserModel::data(const QModelIndex &ndx, int role)const
{
	QVariant	ret;
	User		*ptr = user(ndx);

	if (ptr != NULL) {
		switch (role) {
			case Qt::DisplayRole:
				ret = displayData(ptr, ndx.column());
				break;
			case Qt::BackgroundRole:
				ret = backgroundData(ptr);
				break;
		}
	}

	return (ret);
}


QVariant
UserModel::headerData(int section, Qt::Orientation ort, int role)const
{
	QVariant	ret;

	if ((ort == Qt::Horizontal) && (role == Qt::DisplayRole))
		switch (section) {
			case LAB_ACCOUNT:
				ret = tr("Account");
				break;
			case LAB_PASSWORD:
				ret = tr("Password");
				break;
			case LAB_SSL:
				ret = tr("SSL type");
				break;
		}

	return (ret);
}


int
UserModel::columnCount(const QModelIndex&)const
{

	return (LAB_TOTAL);
}


int
UserModel::rowCount(const QModelIndex&)const
{

	return (vect.size());
}


void
UserModel::timeout()
{

	crc = query->connection()->checksum(systable);
	if (crc != checksum)
		query->exec();
}


void
UserModel::createUser(const User &usr, SqlError &err)
{
	SqlConnection	*conn;

	if ((conn = query->connection()) != NULL) {
		usr.create(conn, err);
		if (err == 0)
			facc = usr.account(FALSE);
		timeout();
	}
}


void
UserModel::deleteUser(const QString &acc, SqlError &err)
{
	SqlConnection	*conn;

	if ((conn = query->connection()) != NULL) {
		err = conn->exec(QString("DROP USER %1").arg(acc));
		if (err == 0)
			timeout();
	}
}


void
UserModel::updateUser(const User &usr, SqlError &err, const User &old)
{
	SqlConnection	*conn;

	if ((conn = query->connection()) != NULL) {
		usr.grant(conn, err, old);
		if (err == 0)
			timeout();
	}
}


void
UserModel::renameUser(const User &from, const User &to, SqlError &err)
{
	SqlConnection	*conn;

	if ((conn = query->connection()) != NULL) {
		err = conn->exec(QString("RENAME USER %1 TO %2")
			.arg(from.account(TRUE)).arg(to.account(TRUE)));
		if (err == 0)
			updateUser(to, err, from);
	}
}


void
UserModel::reloadData()
{
	User		*ptr, tmp;
	QString		acc;
	int		pos;
	unsigned long	changes = 0;

	checksum = crc;
	markAll();
	while (query->next()) {
		copyData(tmp, query);
		acc = tmp.account().toLower();
		ptr = map.value(acc, NULL);
		if (ptr == NULL) {
			ptr = new User(tmp);
			pos = findPosition(acc);
			beginInsertRows(QModelIndex(), pos, pos);
			vect.insert(pos, ptr);
			endInsertRows();
			map.insert(acc, ptr);
			changes++;
		} else
			updateData(ptr, tmp, changes);
	}
	query->free();
	deleteMarked(changes);
	if (changes > 0)
		emit modelChanged();
}


void
UserModel::copyData(User &usr, SqlQuery *q)
{

	usr.setUser(q->value(0).toString());
	usr.setHost(q->value(1).toString());
	usr.setPassword(q->value(2).toString());
	usr.setSslType(q->value(3).toString());
	usr.setCipher(q->value(4).toString());
	usr.setX509Issuer(q->value(5).toString());
	usr.setX509Subject(q->value(6).toString());
	usr.setMaxQuestions(q->value(7).toULongLong());
	usr.setMaxUpdates(q->value(8).toULongLong());
	usr.setMaxConnections(q->value(9).toULongLong());
	usr.setMaxUserConnections(q->value(10).toULongLong());
}


void
UserModel::updateData(User *ptr, const User &usr, unsigned long &changes)
{
	int	pos = -1;

	ptr->setMarked(FALSE);

	if (ptr->user() != usr.user()) {
		ptr->setUser(usr.user());
		changed(pos, LAB_ACCOUNT, ptr, changes);
	}

	if (ptr->host() != usr.host()) {
		ptr->setHost(usr.host());
		changed(pos, LAB_ACCOUNT, ptr, changes);
	}

	if (ptr->password() != usr.password()) {
		ptr->setPassword(usr.password());
		changed(pos, LAB_PASSWORD, ptr, changes);
	}

	if (ptr->sslType() != usr.sslType()) {
		ptr->setSslType(usr.sslType());
		changed(pos, LAB_SSL, ptr, changes);
	}
	ptr->setCipher(usr.cipher());
	ptr->setX509Issuer(usr.x509Issuer());
	ptr->setX509Subject(usr.x509Subject());
	ptr->setMaxQuestions(usr.maxQuestions());
	ptr->setMaxUpdates(usr.maxUpdates());
	ptr->setMaxConnections(usr.maxConnections());
	ptr->setMaxUserConnections(usr.maxUserConnections());
}


void
UserModel::changed(int &row, int col, User *usr, unsigned long &changes)
{

	if (row < 0)
		row = vect.indexOf(usr);

	if (col >= 0) {
		QModelIndex	ndx(index(row, col));

		changes++;
		emit dataChanged(ndx, ndx);
	}
}


void
UserModel::markAll()
{
	UserVector::iterator	it;

	for (it = vect.begin(); it != vect.end(); ++it)
		(*it)->setMarked(TRUE);
}


void
UserModel::deleteMarked(unsigned long &changes)
{
	UserVector::iterator	it;
	User			*usr;
	int			pos;

	for (pos = 0, it = vect.begin(); it != vect.end(); ) {
		usr = *it;
		if (usr->isMarked()) {
			beginRemoveRows(QModelIndex(), pos, pos);
			it = vect.erase(it);
			endRemoveRows();
			map.remove(usr->account().toLower());
			delete usr;
			changes++;
		} else {
			++pos;
			++it;
		}
	}
}


QVariant
UserModel::displayData(const User *usr, int col)const
{
	QVariant	ret;

	switch (col) {
		case LAB_ACCOUNT:
			ret = usr->account(FALSE);
			break;
		case LAB_PASSWORD:
			ret = Core::hasPassword(usr->password());
			break;
		case LAB_SSL:
			ret = User::type2str(usr->sslType());
			break;
	}

	return (ret);
}


QVariant
UserModel::backgroundData(const User *u)const
{
	QVariant	ret;
	bool		pwd = !u->password().isEmpty(),
			usr = !u->user().isEmpty(),
			ssl = (u->sslType() != User::SSL_NONE);
	SqlConnection	*conn = query->connection();

	if ((conn != NULL) && (conn->account() == u->account(FALSE)))
		ret = BRUSH_CONNECTED;
	else if (!usr && !pwd && !ssl)
		ret = BRUSH_NO_SECURITY;
	else if ((!pwd && !ssl) || (!pwd && ssl))
		ret = BRUSH_LOW_SECURITY;
	else if (!ssl)
		ret = BRUSH_MID_SECURITY;

	return (ret);
}


int
UserModel::findPosition(const QString &acc)
{
	UserVector::const_iterator	it;
	int				ret = vect.size(), i;

	for (i = 0, it = vect.constBegin(); it != vect.constEnd(); ++it, ++i)
		if ((*it)->account().toLower() > acc) {
			ret = i;
			break;
		}

	return (ret);
}


User *
UserModel::user(const QModelIndex &ndx)const
{

	return (user(ndx.row()));
}


User *
UserModel::user(int row)const
{
	User	*ret;

	if ((row >= 0) && (row < vect.size()))
		ret = vect.at(row);
	else
		ret = NULL;

	return (ret);
}


User *
UserModel::user(const QString &acc)const
{

	return (map.value(acc.toLower(), NULL));
}


QModelIndex
UserModel::indexOfNew()
{
	QModelIndex	ret;
	User		*usr = user(facc);
	int		row = vect.indexOf(usr);

	facc.clear();
	if ((row >= 0) && (row < vect.size()))
		ret = index(row, 0);

	return (ret);
}

/* EOF */
