/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: user_privileges.cpp,v 1.13 2013/09/20 11:35:54 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "user_privileges.h"
#include "priv_global.h"
#include "priv_database.h"
#include "priv_table.h"
#include "priv_column.h"
#include "priv_proc.h"
#include "sql_connection.h"
#include "sql_user.h"
#include "sql_error.h"


UserPrivileges::UserPrivileges(ServerInfo *ptr, QWidget *parent)
	:QWidget(parent), info(ptr)
{

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	area = new QScrollArea(this);
	layout->addWidget(area, 1);
	area->setAlignment(Qt::AlignTop | Qt::AlignHCenter);
	area->setWidgetResizable(TRUE);
	area->setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
	area->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);

	priv = new QWidget(this);
	area->setWidget(priv);

	dlg = new QDialogButtonBox(Qt::Horizontal, this);
	layout->addWidget(dlg, 0, Qt::AlignBottom);

	commit = new QPushButton(QIcon(":/icons/commit.png"),
		tr("Commit"), this);
	connect(commit, SIGNAL(clicked()), this, SLOT(commitPrivileges()));
	commit->setFocusPolicy(Qt::NoFocus);
	commit->setToolTip(tr("Push it to commit changes"));
	dlg->addButton(commit, QDialogButtonBox::ActionRole);

	p_layout = new QVBoxLayout(priv);
	p_layout->setAlignment(Qt::AlignTop);
	MARGINS_ZERO(p_layout);

	p_global = new PrivGlobal(info->connection, priv);
	connect(p_global, SIGNAL(changed()), this, SLOT(testCommit()));
	p_layout->addWidget(p_global);

	p_dbase = new PrivDatabase(info->connection, priv);
	connect(p_dbase, SIGNAL(changed()), this, SLOT(testCommit()));
	p_layout->addWidget(p_dbase);

	p_table = new PrivTable(info->connection, priv);
	connect(p_table, SIGNAL(changed()), this, SLOT(testCommit()));
	p_layout->addWidget(p_table);

	p_column = new PrivColumn(info->connection, priv);
	connect(p_column, SIGNAL(changed()), this, SLOT(testCommit()));
	p_layout->addWidget(p_column);

	p_proc = new PrivProc(info->connection, priv);
	connect(p_proc, SIGNAL(changed()), this, SLOT(testCommit()));
	p_layout->addWidget(p_proc);

	testCommit();
}

UserPrivileges::~UserPrivileges()
{

	delete p_proc;
	delete p_column;
	delete p_table;
	delete p_dbase;
	delete p_global;
	delete p_layout;
	delete commit;
	delete dlg;
	delete priv;
	delete area;
	delete layout;
}


void
UserPrivileges::timeout()
{

	if (!curr_user.isEmpty()) {
		p_global->timeout(curr_user);
		p_dbase->timeout(curr_user);
		p_table->timeout(curr_user);
		p_column->timeout(curr_user);
		p_proc->timeout(curr_user);
	}
}


void
UserPrivileges::privilegesChanged()
{

	setEnabled(info->priv.isSet(PRIV_GRANT));
}


void
UserPrivileges::loadPrivileges(const SqlUser &usr)
{

	if ((curr_user != usr) && isChanged()) {
		/* we have changed privileges and new user, ask the admin */
		if (QMessageBox::warning(this, tr("Privileges changed"),
			tr("You have uncommited privileges for account %1 "
			"that can be lost.\n"
			"Do you want to commit it to the server?")
			.arg(curr_user.account(FALSE)),
			QMessageBox::Yes | QMessageBox::No,
			QMessageBox::Yes) == QMessageBox::Yes)
			commitPrivileges();
		else
			clear();
	}
	curr_user = usr;
	timeout();
}


void
UserPrivileges::testCommit()
{

	commit->setEnabled(isChanged());
}


void
UserPrivileges::commitPrivileges()
{
	QStringList	queries;
	SqlError	err;

	p_global->commit(queries);
	p_dbase->commit(queries);
	p_table->commit(queries);
	p_column->commit(queries);
	p_proc->commit(queries);

	err = info->connection->exec(queries);
	if (err != 0)
		err.show(this);
	timeout();
}


bool
UserPrivileges::isChanged()const
{
	bool	ret = p_global->isChanged();

	if (ret != TRUE)
		ret |= p_dbase->isChanged();
	if (ret != TRUE)
		ret |= p_table->isChanged();
	if (ret != TRUE)
		ret |= p_column->isChanged();
	if (ret != TRUE)
		ret |= p_proc->isChanged();

	return (ret);
}


void
UserPrivileges::clear()
{

	p_global->clear();
	p_dbase->clear();
	p_table->clear();
	p_column->clear();
	p_proc->clear();
}

/* EOF */
