/*-
 * Copyright (c) 2013 by SilverSoft.Net
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: plugins.cpp,v 1.4 2013/10/09 14:18:14 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "plugins.h"
#include "mysql_plugin_model.h"

#define	BUTTON_ACCEPT	QDialogButtonBox::Ok
#define	BUTTONS		(BUTTON_ACCEPT | QDialogButtonBox::Cancel)


Plugins::Plugins(ServerInfo *info_ptr, QWidget *parent)
	:QWidget(parent), SqlService(info_ptr, "plugins")
{
	QItemSelectionModel	*smodel;

	model = new MySQLPluginModel(connection(), this);
	connect(model, SIGNAL(modelChanged()), this, SLOT(selectItem()));

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	view = new QTableView(this);
	connect(view, SIGNAL(doubleClicked(const QModelIndex&)),
		this, SLOT(pluginInfo(const QModelIndex&)));
	view->setModel(model);
	TABLE_VIEW_INIT(view);
	smodel = view->selectionModel();
	connect(smodel,
		SIGNAL(currentChanged(const QModelIndex&, const QModelIndex&)),
		this,
		SLOT(testPlugin(const QModelIndex&, const QModelIndex&)));
	layout->addWidget(view, 1);

	dlg = new QDialogButtonBox(Qt::Horizontal, this);
	layout->addWidget(dlg, 0, Qt::AlignBottom);

	info = new QPushButton(QIcon(":/icons/information.png"),
		tr("Information"), this);
	connect(info, SIGNAL(clicked()), this, SLOT(pluginInfo()));
	dlg->addButton(info, QDialogButtonBox::ActionRole);

	load = new QPushButton(QIcon(":/icons/plugin-add.png"),
		tr("Load plugin"), this);
	connect(load, SIGNAL(clicked()), this, SLOT(loadPlugin()));
	dlg->addButton(load, QDialogButtonBox::ActionRole);

	unload = new QPushButton(QIcon(":/icons/plugin-delete.png"),
		tr("Unload plugin"), this);
	connect(unload, SIGNAL(clicked()), this, SLOT(unloadPlugin()));
	dlg->addButton(unload, QDialogButtonBox::ActionRole);

	selectItem();
}

Plugins::~Plugins()
{

	delete unload;
	delete load;
	delete info;
	delete dlg;
	delete view;
	delete layout;
	delete model;
}


void
Plugins::timeout()
{

	model->timeout();
}


QString
Plugins::tabName()
{

	return (tr("Plugins"));
}


void
Plugins::destroyIt()
{

	delete this;
}


void
Plugins::deleteNow()
{

	destroyIt();
}


void
Plugins::selectItem()
{
	QItemSelectionModel	*sel = view->selectionModel();
	bool			ena = !model->isEmpty();

	if ((sel != NULL) && !sel->hasSelection())
		view->setCurrentIndex(model->index(0, 0));

	info->setEnabled(ena);
	testPlugin(view->currentIndex(), QModelIndex());
}


void
Plugins::loadPlugin()
{
	PluginDialog	*pdlg;
	QString		pname(QString::null), plib(QString::null);

	pdlg = new PluginDialog(this);
	FATAL(pdlg == NULL);

	if (pdlg->exec()) {
		pname = pdlg->pluginName();
		plib = pdlg->pluginLibrary();
	}

	delete pdlg;

	if (!pname.isEmpty() && ! plib.isEmpty()) {
		/* try to load a new plugin */
		model->loadPlugin(pname, plib, this);
		testPlugin(view->currentIndex(), QModelIndex());
	}
}


void
Plugins::unloadPlugin()
{

	unloadPlugin(view->currentIndex());
}


void
Plugins::unloadPlugin(const QModelIndex &ndx)
{
	Plugin	*plg = model->plugin(ndx);
	QString	pName;
	Confirm	q, w;

	if (plg == NULL)
		Core::noData(this);
	else {
		pName = plg->pluginName();
		q.title = QString::null;
		q.text = tr("Do you want to uninstall plugin %1?").arg(pName);
		w.title = QString::null;
		w.text = tr("Some functions will be unvaialable.\n"
			"Are you sure I should uninstall plugin %1?")
			.arg(pName);
		if (Core::doubleConfirm(q, w, this)) {
			model->unloadPlugin(pName, this);
			testPlugin(view->currentIndex(), QModelIndex());
		}
	}
}


void
Plugins::pluginInfo()
{

	pluginInfo(view->currentIndex());
}


void
Plugins::pluginInfo(const QModelIndex &ndx)
{
	PluginInfoDialog	*pinfo;
	Plugin			*plg = model->plugin(ndx);

	if (plg == NULL)
		Core::noData(this);
	else {
		pinfo = new PluginInfoDialog(this);
		FATAL(pinfo == NULL);
		pinfo->setPluginInfo(plg);
		pinfo->exec();
		delete pinfo;
	}

}


void
Plugins::testPlugin(const QModelIndex &ndx, const QModelIndex&)
{
	Plugin	*plg = model->plugin(ndx);
	bool	ena = FALSE;

	if (plg != NULL)
		ena = !plg->pluginLibrary().isEmpty();

	unload->setEnabled(ena);
}

//-----------------------------------------------------------------------------

PluginDialog::PluginDialog(QWidget *parent)
	:QDialog(parent)
{

	setWindowTitle(tr("Plugin to load"));

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	form = new QFormLayout();
	layout->addLayout(form, 1);

	nameEdit = new QLineEdit(this);
	nameEdit->setMaxLength(PATH_MAX);
	connect(nameEdit, SIGNAL(textChanged(const QString&)),
		this, SLOT(canAccept(const QString&)));
	form->addRow(tr("Plugin name"), nameEdit);

	pathEdit = new QLineEdit(this);
	pathEdit->setMaxLength(PATH_MAX);
	connect(pathEdit, SIGNAL(textChanged(const QString&)),
		this, SLOT(canAccept(const QString&)));
	form->addRow(tr("Path and library name to load"), pathEdit);

	dlg = new QDialogButtonBox(BUTTONS, Qt::Horizontal, this);
	connect(dlg, SIGNAL(accepted()), this, SLOT(accept()));
	connect(dlg, SIGNAL(rejected()), this, SLOT(reject()));
	layout->addWidget(dlg, 0, Qt::AlignBottom);

	canAccept(QString::null);
}

PluginDialog::~PluginDialog()
{

	delete dlg;
	delete pathEdit;
	delete nameEdit;
	delete form;
	delete layout;
}


void
PluginDialog::canAccept(const QString&)
{
	QPushButton	*btn;
	bool		ena = !pluginName().isEmpty();

	if (ena)
		ena = !pluginLibrary().isEmpty();

	btn = dlg->button(BUTTON_ACCEPT);
	if (btn != NULL)
		btn->setEnabled(ena);
}


QString
PluginDialog::pluginName()const
{

	return (nameEdit->text());
}


QString
PluginDialog::pluginLibrary()const
{

	return (pathEdit->text());
}

//-----------------------------------------------------------------------------

PluginInfo::PluginInfo(QWidget *parent)
	:QWidget(parent)
{

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	form = new QFormLayout();
	layout->addLayout(form);

	lPluginName = new QLabel(this);
	form->addRow(tr("Plugin name"), lPluginName);

	lPluginVersion = new QLabel(this);
	form->addRow(tr("Plugin version"), lPluginVersion);

	lPluginStatus = new QLabel(this);
	form->addRow(tr("Plugin status"), lPluginStatus);

	lPluginType = new QLabel(this);
	form->addRow(tr("Plugin type"), lPluginType);

	lPluginTypeVersion = new QLabel(this);
	form->addRow(tr("Plugin type version"), lPluginTypeVersion);

	lPluginLibrary = new QLabel(this);
	form->addRow(tr("Plugin library"), lPluginLibrary);

	lPluginLibraryVersion = new QLabel(this);
	form->addRow(tr("Plugin library version"), lPluginLibraryVersion);

	lPluginAuthor = new QLabel(this);
	form->addRow(tr("Plugin author"), lPluginAuthor);

	lPluginAuthVersion = new QLabel(this);
	form->addRow(tr("Plugin author version"), lPluginAuthVersion);

	lPluginLicense = new QLabel(this);
	form->addRow(tr("Plugin license"), lPluginLicense);

	lLoadOption = new QLabel(this);
	form->addRow(tr("Load option"), lLoadOption);

	lPluginMaturity = new QLabel(this);
	form->addRow(tr("Plugin maturity"), lPluginMaturity);

	bDesc = new QGroupBox(tr("Description of the plugin"), this);
	bDesc->setAlignment(Qt::AlignHCenter);
	layout->addWidget(bDesc, 1);

	lBox = new QVBoxLayout(bDesc);
	MARGINS_ZERO(lBox);

	lDescription = new QLabel(bDesc);
	lDescription->setAlignment(Qt::AlignLeft | Qt::AlignTop);
	lDescription->setWordWrap(TRUE);
	lBox->addWidget(lDescription, 1);
}

PluginInfo::~PluginInfo()
{

	delete lDescription;
	delete lBox;
	delete bDesc;
	delete lPluginMaturity;
	delete lLoadOption;
	delete lPluginLicense;
	delete lPluginAuthVersion;
	delete lPluginAuthor;
	delete lPluginLibraryVersion;
	delete lPluginLibrary;
	delete lPluginTypeVersion;
	delete lPluginType;
	delete lPluginStatus;
	delete lPluginVersion;
	delete lPluginName;
	delete form;
	delete layout;
}


void
PluginInfo::setPluginInfo(const Plugin *plg)
{

	lPluginName->setText(plg->pluginName());
	lPluginVersion->setText(plg->pluginVersion());
	lPluginStatus->setText(plg->pluginStatus());
	lPluginType->setText(plg->pluginType());
	lPluginTypeVersion->setText(plg->pluginTypeVersion());
	lPluginLibrary->setText(plg->pluginLibrary());
	lPluginLibraryVersion->setText(plg->pluginLibraryVersion());
	lPluginAuthor->setText(plg->pluginAuthor());
	lPluginAuthVersion->setText(plg->pluginAuthVersion());
	lPluginLicense->setText(plg->pluginLicense());
	lLoadOption->setText(plg->loadOption());
	lPluginMaturity->setText(plg->pluginMaturity());
	lDescription->setText(plg->pluginDescription());
}

//-----------------------------------------------------------------------------

PluginInfoDialog::PluginInfoDialog(QWidget *parent)
	:QDialog(parent)
{

	setWindowTitle(tr("Plugin information"));

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	info = new PluginInfo(this);
	layout->addWidget(info, 1);

	dlg = new QDialogButtonBox(QDialogButtonBox::Ok, Qt::Horizontal, this);
	connect(dlg, SIGNAL(accepted()), this, SLOT(accept()));
	connect(dlg, SIGNAL(rejected()), this, SLOT(reject()));
	layout->addWidget(dlg, 0, Qt::AlignBottom);
}

PluginInfoDialog::~PluginInfoDialog()
{

	delete dlg;
	delete info;
	delete layout;
}


void
PluginInfoDialog::setPluginInfo(const Plugin *plg)
{

	info->setPluginInfo(plg);
}

/* EOF */
