/*-
 * Copyright (c) 2013 by SilverSoft.Net
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: process_model.cpp,v 1.7 2013/08/28 12:18:14 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "process_model.h"
#include "sql_connection.h"
#include "sql_query.h"
#include "sql_error.h"

static const char
	*F_ID = "ID",
	*F_USER = "USER",
	*F_HOST = "HOST",
	*F_DB = "DB",
	*F_COMMAND = "COMMAND",
	*F_TIME = "TIME",
	*F_TIME_MS = "TIME_MS",
	*F_STAGE = "STAGE",
	*F_MAX_STAGE = "MAX_STAGE",
	*F_PROGRESS = "PROGRESS",
	*F_MEMORY_USED = "MEMORY_USED",
	*F_EXAMINED_ROWS = "EXAMINED_ROWS",
	*F_STATE = "STATE",
	*F_INFO = "INFO";


ProcessModel::ProcessModel(SqlConnection *conn, QObject *parent)
	:QAbstractTableModel(parent), smt(FALSE)
{

	query = new SqlQuery(conn);
#if	defined(FEATURES)
	qtf = new SqlQuery(conn);
#endif	/* FEATURES */

	addField(F_ID, tr("Id"));
	addField(F_USER, tr("User"));
	addField(F_HOST, tr("Host"));
	addField(F_DB, tr("Db"));
	addField(F_COMMAND, tr("Command"));
	addField(F_TIME, tr("Time"));
	addField(F_STATE, tr("State"));
	addField(F_TIME_MS, tr("Time (ms)"), FALSE);
	addField(F_STAGE, tr("Stage"), FALSE);
	addField(F_MAX_STAGE, tr("Max stage"), FALSE);
	addField(F_PROGRESS, tr("Progress"), FALSE);
	addField(F_MEMORY_USED, tr("Used memory"), FALSE);
	addField(F_EXAMINED_ROWS, tr("Rows examined"), FALSE);
	addField(F_INFO, tr("Info"));

	testFeatures(conn);

	query->connect(this, SLOT(reloadData()));
}

ProcessModel::~ProcessModel()
{
	FieldMap::iterator	it;
	ProcessField		*fld;

	for (it = f_map.begin(); it != f_map.end(); ++it) {
		fld = *it;
		delete fld;
	}

	destroyVector();
#if	defined(FEATURES)
	delete qtf;
#endif	 /* FEATURES */
	delete query;
}


QVariant
ProcessModel::data(const QModelIndex &ndx, int role)const
{
	QVariant	ret;
	Process		*proc = process(ndx);

	if (proc != NULL)
		switch (role) {
			case Qt::DisplayRole:
				ret = displayData(proc, ndx.column());
				break;
			case Qt::BackgroundRole:
				ret = backgroundData(proc);
				break;
		}

	return (ret);
}


QVariant
ProcessModel::headerData(int section, Qt::Orientation ort, int role)const
{
	QVariant	ret;

	if ((ort == Qt::Horizontal) && (role == Qt::DisplayRole) &&
		(section < f_vect.size()))
		ret = f_vect.at(section)->description();

	return (ret);
}


int
ProcessModel::columnCount(const QModelIndex&)const
{

	return (f_vect.size());
}


int
ProcessModel::rowCount(const QModelIndex&)const
{

	return (vect.size());
}


void
ProcessModel::addField(const QString &name, const QString &descr, bool mark)
{
	ProcessField	*fld;

	fld = new ProcessField(name, descr);
	fld->setMarked(mark);
	if (fld->isMarked()) {
		fld->setPosition(f_vect.size());
		f_vect<<fld;
	}
	f_map.insert(name, fld);
	
}


void
ProcessModel::testFeatures(SqlConnection *)
{
	FieldVector::const_iterator	it;
	QStringList	list;

	for (it = f_vect.constBegin(); it != f_vect.constEnd(); ++it)
		list<<(*it)->name();

	/* FIXME: test server version and add some columns to the names list */

	query->prepare(QString("select %1 from %2").arg(list.join(", "))
		.arg(Core::informationSchema("PROCESSLIST")));
}


void
ProcessModel::markAll()
{
	ProcessVector::iterator	it;

	for (it = vect.begin(); it != vect.end(); ++it)
		(*it)->setMarked(TRUE);
}


void
ProcessModel::deleteMarked(unsigned long &changes)
{
	ProcessVector::iterator	it;
	Process			*proc;
	int			pos;

	for (pos = 0, it = vect.begin(); it != vect.end(); ) {
		proc = *it;
		if (proc->isMarked()) {
			beginRemoveRows(QModelIndex(), pos, pos);
			it = vect.erase(it);
			endRemoveRows();
			map.remove(proc->id());
			delete proc;
			changes++;
		} else {
			++pos;
			++it;
		}
	}
}


void
ProcessModel::reloadData()
{
	Process		*proc, tmp;
	int		pos;
	unsigned long	id, changes = 0;

	markAll();
	id = myThreadId();
	while (query->next()) {
		copyData(tmp, query);
		if (!smt && (id == tmp.id())) {
			/* skip my thread */
			continue;
		}
		proc = map.value(tmp.id(), NULL);
		if (proc == NULL) {
			/* create a new process */
			proc = new Process(tmp);
			pos = findPosition(proc->id());
			beginInsertRows(QModelIndex(), pos, pos);
			vect.insert(pos, proc);
			endInsertRows();
			map.insert(proc->id(), proc);
			changes++;
		} else
			updateData(proc, tmp, changes);
	}
	query->free();
	deleteMarked(changes);
	if (changes > 0)
		emit modelChanged();
}


void
ProcessModel::timeout()
{

	query->exec();
}


void
ProcessModel::kill(const QModelIndex &ndx, QWidget *parent)
{
	Process		*proc = process(ndx);
	unsigned long	id, tmp;
	SqlError	err;

	if (proc == NULL) {
		Core::noData(parent);
		id = tmp = 0;
	} else {
		id = myThreadId();
		tmp = proc->id();
	}
	if ((id == 0) && (tmp == 0))
		/* nothing to do here, just skip this situation */;
	else if (id == tmp)
		QMessageBox::warning(parent, tr("Kamikaze mode"),
			tr("Do you want to kill my thread %1?\n"
			"Really, do you?\n"
			"I think, no, I can't allow you to do it...\n"
			"If you still want to do it anyway, "
			"learn mysqladmin ;-)").arg(id));
	else if ((err = query->connection()->kill(tmp)) != 0)
		err.show(parent);
	else
		QMessageBox::information(parent, tr("Killed"),
			tr("Thread id %1 killed, amen ;-)").arg(tmp));
}


int
ProcessModel::findPosition(unsigned long id)
{
	ProcessVector::const_iterator	it;
	int				ret = vect.size(), i;

	for (i = 0, it = vect.constBegin(); it != vect.constEnd(); ++it, ++i) {
		if ((*it)->id() > id) {
			ret = i;
			break;
		}
	}

	return (ret);
}


void
ProcessModel::copyData(Process &proc, SqlQuery *q)
{

	proc.setId(q->value(F_ID).toLongLong());
	proc.setUser(q->value(F_USER).toString());
	proc.setHost(q->value(F_HOST).toString());
	proc.setDb(q->value(F_DB).toString());
	proc.setCommand(q->value(F_COMMAND).toString());
	proc.setTime(q->value(F_TIME).toInt());
	proc.setState(q->value(F_STATE).toString());
	proc.setInfo(q->value(F_INFO).toString());
	proc.setTimeMs(q->value(F_TIME_MS).toDouble());
	proc.setStage(q->value(F_STAGE).toInt());
	proc.setMaxStage(q->value(F_MAX_STAGE).toInt());
	proc.setProgress(q->value(F_PROGRESS).toDouble());
	proc.setMemoryUsed(q->value(F_MEMORY_USED).toInt());
	proc.setExaminedRows(q->value(F_EXAMINED_ROWS).toInt());
}


void
ProcessModel::updateData(Process *ptr, const Process &proc,
	unsigned long &changes)
{
	int	pos = -1;

	ptr->setMarked(FALSE);

	if (ptr->user() != proc.user()) {
		ptr->setUser(proc.user());
		changed(pos, fieldPosition(F_USER), ptr, changes);
	}

	if (ptr->host() != proc.host()) {
		ptr->setHost(proc.host());
		changed(pos, fieldPosition(F_HOST), ptr, changes);
	}

	if (ptr->db() != proc.db()) {
		ptr->setDb(proc.db());
		changed(pos, fieldPosition(F_DB), ptr, changes);
	}

	if (ptr->command() != proc.command()) {
		ptr->setCommand(proc.command());
		changed(pos, fieldPosition(F_COMMAND), ptr, changes);
	}

	if (ptr->time() != proc.time()) {
		ptr->setTime(proc.time());
		changed(pos, fieldPosition(F_TIME), ptr, changes);
	}

	if (ptr->state() != proc.state()) {
		ptr->setState(proc.state());
		changed(pos, fieldPosition(F_STATE), ptr, changes);
	}

	if (ptr->info() != proc.info()) {
		ptr->setInfo(proc.info());
		changed(pos, fieldPosition(F_INFO), ptr, changes);
	}
}


void
ProcessModel::changed(int &row, int col, Process *proc, unsigned long &changes)
{
	QModelIndex	ndx = index(row, col);

	if (row < 0)
		row = vect.indexOf(proc);
	if (col >= 0) {
		changes++;
		emit dataChanged(ndx, ndx);
	}
}


Process *
ProcessModel::process(const QModelIndex &ndx)const
{

	return (process(ndx.row()));
}


Process *
ProcessModel::process(int row)const
{
	Process	*ret;

	if ((row < 0) || (row >= vect.size()))
		ret = NULL;
	else
		ret = vect.at(row);

	return (ret);
}


QVariant
ProcessModel::displayData(const Process *proc, int col)const
{
	QVariant		ret;
	ProcessField		*fld;
	Qt::CaseSensitivity	sens = Qt::CaseInsensitive;

	if ((col >= 0) && (col < f_vect.size()) &&
		((fld = f_vect.at(col)) != NULL)) {
		QString	name(fld->name());

		if (name.compare(F_ID, sens) == 0)
			ret = (qulonglong)proc->id();
		else if (name.compare(F_USER, sens) == 0)
			ret = proc->user();
		else if (name.compare(F_HOST, sens) == 0)
			ret = proc->host();
		else if (name.compare(F_DB, sens) == 0)
			ret = proc->db();
		else if (name.compare(F_COMMAND, sens) == 0)
			ret = proc->command();
		else if (name.compare(F_TIME, sens) == 0)
			ret = proc->time();
		else if (name.compare(F_TIME_MS, sens) == 0)
			ret = proc->timeMs();
		else if (name.compare(F_STAGE, sens) == 0)
			ret = proc->stage();
		else if (name.compare(F_MAX_STAGE, sens) == 0)
			ret = proc->maxStage();
		else if (name.compare(F_PROGRESS, sens) == 0)
			ret = proc->progress();
		else if (name.compare(F_MEMORY_USED, sens) == 0)
			ret = proc->memoryUsed();
		else if (name.compare(F_EXAMINED_ROWS, sens) == 0)
			ret = proc->examinedRows();
		else if (name.compare(F_STATE, sens) == 0)
			ret = proc->state();
		else if (name.compare(F_INFO, sens) == 0)
			ret = proc->info();
	}

	return (ret);
}


QVariant
ProcessModel::backgroundData(const Process *proc)const
{
	QVariant	ret;
	unsigned long	id = myThreadId();

	if ((id != 0) && (id == proc->id()))
		ret = BRUSH_MY_ID;

	return (ret);
}


void
ProcessModel::switchMyThread(int state)
{

	smt = (state == Qt::Unchecked);
	timeout();
}


bool
ProcessModel::isEmpty()const
{
	int	size = vect.size(), trigger = 1;

	if (smt)
		trigger++;

	return (size < trigger);
}


unsigned long
ProcessModel::myThreadId()const
{

	return (query->connection()->threadId());
}


int
ProcessModel::fieldPosition(const char *name)
{
	int		ret;
	ProcessField	*fld = f_map.value(name, NULL);

	if (fld != NULL)
		ret = fld->position();
	else
		ret = -1;

	return (ret);
}


void
ProcessModel::destroyVector()
{
	ProcessVector::iterator	it;
	Process			*proc;

	for (it = vect.begin(); it != vect.end(); ++it) {
		proc = *it;
		delete proc;
	}
}

/* EOF */
