/*-
 * Copyright (c) 2013 by SilverSoft.Net
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: integer_editor.cpp,v 1.2 2013/09/30 12:10:24 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "integer_editor.h"
#include "tunable_integer.h"


IntegerEditor::IntegerEditor(QWidget *parent)
	:AbstractEditor(parent), saved_value(0)
{
	int	row = 0;

	desc = new QLabel(this);
	desc->setAlignment(Qt::AlignLeft | Qt::AlignTop);
	desc->setWordWrap(TRUE);
	layout->addWidget(desc, 1);

	addDefaultValue();

	grid = new QGridLayout();
	MARGINS_ZERO(grid);
	layout->addLayout(grid);

	lmin = new QLabel(this);
	grid->addWidget(lmin, row, 0);
	grid->setAlignment(lmin, Qt::AlignLeft);

	lmax = new QLabel(this);
	grid->addWidget(lmax, row++, 1);
	grid->setAlignment(lmax, Qt::AlignRight);

	slider = new QSlider(Qt::Horizontal, this);
	connect(slider, SIGNAL(valueChanged(int)),
		this, SLOT(updateValue(int)));
	slider->setTickPosition(QSlider::NoTicks);
	slider->setRange(SLIDER_MIN, SLIDER_MAX);
	grid->addWidget(slider, row, 0, 1, 2);
	grid->setColumnStretch(1, 1);

	edit = new QLineEdit(this);
	edit->setMinimumWidth(edit->fontMetrics().lineSpacing() * 12);
	connect(edit, SIGNAL(textChanged(const QString&)),
		this, SLOT(updateValue(const QString&)));
	grid->addWidget(edit, row, 3);
	grid->setColumnStretch(3, 0);

	validator = new QRegExpValidator(this);

	addDialog();
}

IntegerEditor::~IntegerEditor()
{

	delete validator;
	delete edit;
	delete slider;
	delete lmax;
	delete lmin;
	delete grid;
	delete desc;
}


void
IntegerEditor::setTemplate(const TunableVariable *var)
{
	const TunableInteger	*tint;
	qint64			val;
	QRegExp			rx;

	FATAL(var->type() != TunableVariable::T_INT);
	tint = static_cast<const TunableInteger *>(var);
	desc->setText(var->description());

	val = tint->value();
	setDefaultValue(QString::number(val));
	min_value = tint->minValue();
	max_value = tint->maxValue();
	lmin->setText(QString::number(min_value));
	lmax->setText(QString::number(max_value));

	rx.setCaseSensitivity(Qt::CaseSensitive);
	rx.setPatternSyntax(QRegExp::RegExp);
	if ((min_value < 0) && (max_value < 0))
		rx.setPattern("\\-\\d+");
	else if ((min_value == 0) && (max_value == 0))
		rx.setPattern("0");
	else if ((min_value >= 0) && (max_value >= 0))
		rx.setPattern("\\+?\\d+");
	else
		rx.setPattern("[\\-|\\+]?\\d+");
	validator->setRegExp(rx);
	edit->setValidator(validator);

	setValue(QString::number(val));
	saved = FALSE;
}


void
IntegerEditor::setValue(const QString &val)
{
	int	pos;
	qint64	curr = fitValue(val.toLongLong());

	pos = static_cast<int>(SCALE(min_value, curr, max_value,
		slider->minimum(), slider->maximum()));
	slider->setValue(pos);
	curr_value = curr;
	if (saved != TRUE) {
		saved = TRUE;
		saved_value = curr;
	}
	edit->setText(QString::number(curr_value));
}


void
IntegerEditor::updateValue(int val)
{

	if (slider->hasFocus() && !edit->hasFocus()) {
		curr_value = static_cast<qint64>(SCALE(slider->minimum(), val,
			slider->maximum(), min_value, max_value));
		edit->setText(QString::number(curr_value));
		setSaveEnabled(saved_value != curr_value);
	}
}


void
IntegerEditor::updateValue(const QString &val)
{
	qint64	tmp = val.toLongLong(),
		curr = fitValue(tmp);
	int	pos;

	if (curr == tmp) {
		if (!slider->hasFocus()) {
			pos = static_cast<int>(SCALE(min_value, curr,
				max_value, slider->minimum(),
				slider->maximum()));
			slider->setValue(pos);
		}
		curr_value = curr;
		setSaveEnabled(saved_value != curr_value);
	} else
		edit->setText(QString::number(curr_value));
}


qint64
IntegerEditor::fitValue(qint64 val)
{

	if (val < min_value)
		val = min_value;
	else if (val > max_value)
		val = max_value;

	return (val);
}


void
IntegerEditor::setDefault()
{

	edit->setText(defaultValue());
}


void
IntegerEditor::commit(SqlConnection *conn, const QString &name)
{

	doCommit(conn, name, QString::number(curr_value));
}

/* EOF */
