/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: real_editor.cpp,v 1.1 2013/10/07 17:00:13 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "real_editor.h"
#include "tunable_real.h"

#define	NUMBER(num)	QString::number(num, 'g', prec)
/*
 * For "smart" programmers:
 * we cannot compare two floating pont variables "as is"
 * beacause they will never equivalent (why? read some books!)
 * so, here we can speak about some error between it
 * in this case we choosed min_err as 1/(10^precision)
 * ^ here is not the XOR!
 * so, now good luck in the exploration of the code ;-)
 */
#define	EQU(v1, v2)	(fabsl(v1 - v2) < min_err)


RealEditor::RealEditor(QWidget *parent)
	:AbstractEditor(parent), saved_value(0.0), min_err(1.0)
{
	int	row = 0;

	desc = new QLabel(this);
	desc->setAlignment(Qt::AlignLeft | Qt::AlignTop);
	desc->setWordWrap(TRUE);
	layout->addWidget(desc, 1);

	addDefaultValue();

	grid = new QGridLayout();
	MARGINS_ZERO(grid);
	layout->addLayout(grid);

	lmin = new QLabel(this);
	grid->addWidget(lmin, row, 0);
	grid->setAlignment(lmin, Qt::AlignLeft);

	lmax = new QLabel(this);
	grid->addWidget(lmax, row++, 1);
	grid->setAlignment(lmax, Qt::AlignRight);

	slider = new QSlider(Qt::Horizontal, this);
	connect(slider, SIGNAL(valueChanged(int)),
		this, SLOT(updateValue(int)));
	slider->setTickPosition(QSlider::NoTicks);
	slider->setRange(SLIDER_MIN, SLIDER_MAX);
	grid->addWidget(slider, row, 0, 1, 2);
	grid->setColumnStretch(1, 1);

	edit = new QLineEdit(this);
	edit->setMinimumWidth(edit->fontMetrics().lineSpacing() * 12);
	connect(edit, SIGNAL(textChanged(const QString&)),
		this, SLOT(updateValue(const QString&)));
	grid->addWidget(edit, row, 3);
	grid->setColumnStretch(3, 0);

	validator = new QRegExpValidator(this);

	addDialog();
}

RealEditor::~RealEditor()
{

	delete validator;
	delete edit;
	delete slider;
	delete lmax;
	delete lmin;
	delete grid;
	delete desc;
}


void
RealEditor::setTemplate(const TunableVariable *var)
{
	const TunableReal	*treal;
	qreal			val;
	QRegExp			rx;

	FATAL(var->type() != TunableVariable::T_REAL);
	treal = static_cast<const TunableReal *>(var);
	desc->setText(var->description());

	val = treal->value();
	prec = treal->precision();
	min_err = 1.0 / pow(10.0, -prec);
	setDefaultValue(QString::number(val));
	min_value = treal->minValue();
	max_value = treal->maxValue();
	lmin->setText(NUMBER(min_value));
	lmax->setText(NUMBER(max_value));

	rx.setCaseSensitivity(Qt::CaseSensitive);
	rx.setPatternSyntax(QRegExp::RegExp);
	if ((min_value < 0.0) && (max_value < 0.0))
		rx.setPattern("\\-\\d+(\\.\\d+)?");
	else if ((min_value == 0.0) && (max_value == 0.0))
		rx.setPattern("0(\\.0)?");
	else if ((min_value >= 0.0) && (max_value >= 0.0))
		rx.setPattern("\\+?\\d+(\\.\\d+)?");
	else
		rx.setPattern("[\\-|\\+]?\\d+(\\.\\d+)?");
	validator->setRegExp(rx);
	edit->setValidator(validator);

	setValue(NUMBER(val));
	saved = FALSE;
}


void
RealEditor::setValue(const QString &val)
{
	int	pos;
	qreal	curr = fitValue(val.toDouble());

	pos = static_cast<int>(SCALE(min_value, curr, max_value,
		slider->minimum(), slider->maximum()));
	slider->setValue(pos);
	curr_value = curr;
	if (saved != TRUE) {
		saved = TRUE;
		saved_value = curr;
	}
	edit->setText(NUMBER(curr_value));
}


void
RealEditor::updateValue(int val)
{

	if (slider->hasFocus() && !edit->hasFocus()) {
		curr_value = static_cast<qreal>(SCALE(slider->minimum(), val,
			slider->maximum(), min_value, max_value));
		edit->setText(NUMBER(curr_value));
		setSaveEnabled(!EQU(saved_value, curr_value));
	}
}


void
RealEditor::updateValue(const QString &val)
{
	qreal	tmp = val.toDouble(),
		curr = fitValue(tmp);
	int	pos;

	if (EQU(curr, tmp)) {
		if (!slider->hasFocus()) {
			pos = static_cast<int>(SCALE(min_value, curr,
				max_value, slider->minimum(),
				slider->maximum()));
			slider->setValue(pos);
		}
		curr_value = curr;
		setSaveEnabled(!EQU(saved_value, curr_value));
	} else
		edit->setText(QString::number(curr_value));
}


qreal
RealEditor::fitValue(qreal val)
{

	if (val < min_value)
		val = min_value;
	else if (val > max_value)
		val = max_value;

	return (val);
}


void
RealEditor::setDefault()
{

	edit->setText(defaultValue());
}


void
RealEditor::commit(SqlConnection *conn, const QString &name)
{

	doCommit(conn, name, NUMBER(curr_value));
}

/* EOF */
