/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: tuning.cpp,v 1.9 2013/10/09 14:18:14 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "tuning.h"
#include "tuning_model.h"
#include "abstract_editor.h"
#include "abstract_variable.h"
#include "tunable_variable.h"
#include "informer.h"


Tuning::Tuning(ServerInfo *info, QWidget *parent)
	:QWidget(parent), SqlService(info, "tuning")
{

	model = new TuningModel(connection(), this);

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	view = new QTableView(this);
	connect(view, SIGNAL(doubleClicked(const QModelIndex&)),
		this, SLOT(showInfo(const QModelIndex&)));
	connect(view, SIGNAL(clicked(const QModelIndex&)),
		this, SLOT(testInfo(const QModelIndex&)));
	view->setModel(model);
	TABLE_VIEW_INIT(view);
	layout->addWidget(view, 1);

	hbox = new QHBoxLayout();
	MARGINS_ZERO(hbox);
	layout->addLayout(hbox);

	filter = new TuningFilter(this);
	connect(model, SIGNAL(found(int)), filter, SLOT(setFound(int)));
	connect(filter, SIGNAL(filterChanged(const QString&)),
		model, SLOT(setFilter(const QString&)));
	hbox->addWidget(filter, 0, Qt::AlignLeft);

	legend = new TuningLegend(this);
	hbox->addWidget(legend);

	ctl = new QDialogButtonBox(Qt::Vertical, this);
	ctl->setCenterButtons(TRUE);
	hbox->addWidget(ctl);

	cmd_line = new QPushButton(tr("Command line"), this);
	cmd_line->setToolTip(tr("Click this button to show command line to "
		"start the server with all changes"));
	cmd_line->setEnabled(FALSE);
	ctl->addButton(cmd_line, QDialogButtonBox::ActionRole);

	cfg_file = new QPushButton(tr("mysqld section"), this);
	cfg_file->setToolTip(tr("Click this button to show command line to "
		"show the changes as mysqld section in cnf-file"));
	cfg_file->setEnabled(FALSE);
	ctl->addButton(cfg_file, QDialogButtonBox::ActionRole);

	dlg = new QDialogButtonBox(Qt::Vertical, this);
	dlg->setCenterButtons(TRUE);
	hbox->addWidget(dlg);

	v_info = new QPushButton(QIcon(":/icons/info.png"),
		tr("Variable information"), this);
	connect(v_info, SIGNAL(clicked()), this, SLOT(showInfo()));
	dlg->addButton(v_info, QDialogButtonBox::ActionRole);

	v_edit = new QPushButton(QIcon(":/icons/edit.png"),
		tr("Modify"), this);
	connect(v_edit, SIGNAL(clicked()), this, SLOT(editCurrent()));
	dlg->addButton(v_edit, QDialogButtonBox::ActionRole);

	testInfo(QModelIndex());
}

Tuning::~Tuning()
{

	delete v_edit;
	delete v_info;
	delete dlg;
	delete cfg_file;
	delete cmd_line;
	delete ctl;
	delete legend;
	delete filter;
	delete hbox;
	delete view;
	delete layout;
	delete model;
}


void
Tuning::timeout()
{

	model->timeout();
}


void
Tuning::privilegesChanged()
{

	/* nothing to do here (at this moment?) */
}


QString
Tuning::tabName()
{

	return (tr("Tunable"));
}


void
Tuning::destroyIt()
{

	delete this;
}


void
Tuning::deleteNow()
{

	destroyIt();
}


void
Tuning::showInfo()
{

	showInfo(view->currentIndex());
}


void
Tuning::showInfo(const QModelIndex &ndx)
{
	Tunable			*tune = model->tunable(ndx);
	AbstractVariable	*var;
	MessageInformer		*inf;

	if (tune == NULL)
		Core::noData(this);
	else if ((var = tune->parent()) == NULL)
		unknownVariable(tune->name());
	else {
		inf = new MessageInformer(tune, this);
		connect(inf, SIGNAL(edit()), this, SLOT(editCurrent()));
		inf->exec();
		delete inf;
	}
}


void
Tuning::testInfo(const QModelIndex &ndx)
{
	Tunable			*tune = model->tunable(ndx);
	AbstractVariable	*var;
	bool			ena = (tune != NULL);

	v_info->setEnabled(ena);
	if (ena) {
		var = tune->parent();
		if (var != NULL)
			ena = (var->type() == AbstractVariable::ReadWrite);
		else
			ena = FALSE;
	}
	v_edit->setEnabled(ena);
}


void
Tuning::editCurrent()
{
	Tunable			*tune = model->tunable(view->currentIndex());
	AbstractEditor		*editor;
	bool			reload = FALSE;
	union {
		AbstractVariable	*var;
		TunableVariable		*tvar;
	} udata;

	if (tune == NULL)
		Core::noData(this);
	else if ((udata.var = tune->parent()) == NULL)
		unknownVariable(tune->name());
	else if (udata.var->type() != AbstractVariable::ReadWrite)
		QMessageBox::critical(this, tr("Variable error"),
			tr("Cannot modify read-only variable"));
	else  if ((editor = udata.tvar->createEditor(this)) == NULL)
		QMessageBox::critical(this, tr("no editor"),
			tr("No editor defined for this variable"));
	else {
		editor->setWindowTitle(tune->prettyName());
		editor->setTemplate(udata.tvar);
		editor->setValue(tune->value());
		if (editor->exec()) {
			editor->commit(connection(), tune->name());
			reload = TRUE;
		}
		delete editor;
	}

	if (reload)
		timeout();
}


void
Tuning::unknownVariable(const QString &name)
{

	QMessageBox::warning(this, Core::text(Core::UnknownVariable),
		tr("Oops, you have a new (or too old) version of "
		"server that has this unknown variable (%1). You would "
		"send the name of it and the server version, you are "
		"running, to us and maybe we will fix it.\n"
		"Yes, maybe! ;-)").arg(name));
}

//-----------------------------------------------------------------------------

TuningLegend::TuningLegend(QWidget *parent)
	:QGroupBox(tr("Tunable legend"), parent)
{

	layout = new QVBoxLayout(this);
	MARGINS_ZERO(layout);

	unk = Core::addLegendItem(Core::text(Core::UnknownVariable),
		BRUSH_UNKNOWN, this);
	layout->addWidget(unk);

	rw = Core::addLegendItem(tr("Tunable variable (can be modified)"),
		BRUSH_READWRITE, this);
	layout->addWidget(rw);

	ro = Core::addLegendItem(tr("Status or read-only variable "
		"(cannot be modified)"),
		BRUSH_READONLY, this);
	layout->addWidget(ro);
}

TuningLegend::~TuningLegend()
{

	delete ro;
	delete rw;
	delete unk;
	delete layout;
}

//-----------------------------------------------------------------------------

TuningFilter::TuningFilter(QWidget *parent)
	:QGroupBox(parent)
{

	layout = new QFormLayout(this);
	MARGINS_ZERO(layout);

	edit = new QLineEdit(this);
	connect(edit, SIGNAL(textChanged(const QString&)),
		SIGNAL(filterChanged(const QString&)));
	edit->setMinimumWidth(edit->fontMetrics().lineSpacing() * 10);
	layout->addRow(tr("Filter variables"), edit);

	found = new QLCDNumber(12, this);
	found->setMode(QLCDNumber::Dec);
	found->setSegmentStyle(QLCDNumber::Flat);
	layout->addRow(tr("Number of variables"), found);
	setFound(0);
}

TuningFilter::~TuningFilter()
{

	delete found;
	delete edit;
	delete layout;
}


QString
TuningFilter::filter()const
{

	return (edit->text());
}


void
TuningFilter::setFound(int fnd)
{

	found->display(fnd);
}

/* EOF */
