/*-
 * Copyright (c) 2013 by SilverSoft.Net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * $Id: tuning_model.cpp,v 1.15 2013/10/09 14:18:14 denis Exp $
 */

/*-
 * Author: Denis Kozadaev (denis@silversoft.net)
 *
 * Please add file description here
 */

#include "tuning_model.h"
#include "sql_connection.h"
#include "sql_query.h"
#include "sql_error.h"
#include "abstract_variable.h"

#define	LAB_NAME	0
#define	LAB_VALUE	1
#define	LAB_TOTAL	(LAB_VALUE + 1)


TuningModel::TuningModel(SqlConnection *conn, QObject *parent)
	:QAbstractTableModel(parent), filter(QString::null)
{
	QString	query("select VARIABLE_NAME, VARIABLE_VALUE from %1 "
			"order by VARIABLE_NAME");

	q_status = new SqlQuery(conn);
	q_status->prepare(query.arg(Core::informationSchema("GLOBAL_STATUS")));

	q_status->connect(this, SLOT(reloadStatus()));

	q_variable = new SqlQuery(conn);
	q_variable->prepare(
		query.arg(Core::informationSchema("GLOBAL_VARIABLES")));
	q_variable->connect(this, SLOT(reloadVariables()));
}

TuningModel::~TuningModel()
{
	TunableVector::iterator	it;
	Tunable			*tune;

	for (it = vect.begin(); it != vect.end(); ++it) {
		tune = *it;
		delete tune;
	}

	delete q_variable;
	delete q_status;
}


QVariant
TuningModel::data(const QModelIndex &ndx, int role)const
{
	QVariant	ret;
	Tunable		*tune = tunable(ndx);

	if (tune != NULL)
		switch (role) {
			case Qt::DisplayRole:
				ret = displayData(tune, ndx.column());
				break;
			case Qt::BackgroundRole:
				ret = backgroundData(tune);
				break;
		}

	return (ret);
}


QVariant
TuningModel::headerData(int section, Qt::Orientation ort, int role)const
{
	QVariant	ret;

	if ((ort == Qt::Horizontal) && (role == Qt::DisplayRole))
		switch (section) {
			case LAB_NAME:
				ret = tr("Name");
				break;
			case LAB_VALUE:
				ret = tr("Value");
				break;
		}

	return (ret);
}


int
TuningModel::columnCount(const QModelIndex&)const
{

	return (LAB_TOTAL);
}


int
TuningModel::rowCount(const QModelIndex&)const
{

	return (vect.size());
}


void
TuningModel::loadQuery(SqlQuery *q)
{
	Tunable		*tune, tmp;
	int		pos;

	while (q->next()) {
		copyData(tmp, q);
		if (!filter.isEmpty() && (tmp.name().indexOf(filter, 0,
			Qt::CaseInsensitive) != 0))
			continue;
		tune = map.value(tmp.name().toLower(), NULL);
		if (tune == NULL) {
			/* create a new tunable status */
			tune = new Tunable(tmp.name());
			tune->setValue(tmp.value());
			pos = findPosition(tune->name().toLower());
			beginInsertRows(QModelIndex(), pos, pos);
			vect.insert(pos, tune);
			endInsertRows();
			map.insert(tune->name().toLower(), tune);
			changes++;
		} else
			updateData(tune, tmp);
	}
	q->free();
}


void
TuningModel::markAll()
{
	TunableVector::iterator	it;

	for (it = vect.begin(); it != vect.end(); ++it)
		(*it)->setMarked(TRUE);
	changes = 0;
}


void
TuningModel::deleteMarked()
{
	TunableVector::iterator	it;
	Tunable			*tune;
	int			pos;

	for (pos = 0, it = vect.begin(); it != vect.end(); ) {
		tune = *it;
		if (tune->isMarked()) {
			beginRemoveRows(QModelIndex(), pos, pos);
			it = vect.erase(it);
			endRemoveRows();
			map.remove(tune->name().toLower());
			delete tune;
			changes++;
		} else {
			++pos;
			++it;
		}
	}
}


void
TuningModel::reloadStatus()
{

	loadQuery(q_status);
}


void
TuningModel::reloadVariables()
{

	loadQuery(q_variable);

	deleteMarked();
	if (changes > 0) {
		emit modelChanged();
		emit found(vect.size());
	}
}


void
TuningModel::timeout()
{

	markAll();
	q_status->exec();
	q_variable->exec();
}


int
TuningModel::findPosition(const QString &name)
{
	TunableVector::const_iterator	it;
	int				ret = vect.size(), i;

	for (i = 0, it = vect.constBegin(); it != vect.constEnd(); ++it, ++i) {
		if ((*it)->name().toLower() > name) {
			ret = i;
			break;
		}
	}

	return (ret);
}


void
TuningModel::copyData(Tunable &tune, SqlQuery *q)
{

	tune.setName(q->value(0).toString());
	tune.setValue(q->value(1).toString());
}


void
TuningModel::updateData(Tunable *ptr, const Tunable &tune)
{
	int	pos = -1;

	ptr->setMarked(FALSE);

	if (ptr->value() != tune.value()) {
		ptr->setValue(tune.value());
		changed(pos, LAB_VALUE, ptr);
	}
}


void
TuningModel::changed(int &row, int col, Tunable *tune)
{
	QModelIndex	ndx = index(row, col);

	if (row < 0)
		row = vect.indexOf(tune);
	if (col >= 0) {
		changes++;
		emit dataChanged(ndx, ndx);
	}
}


Tunable *
TuningModel::tunable(const QModelIndex &ndx)const
{

	return (tunable(ndx.row()));
}


Tunable *
TuningModel::tunable(int row)const
{
	Tunable	*ret;

	if ((row < 0) || (row >= vect.size()))
		ret = NULL;
	else
		ret = vect.at(row);

	return (ret);
}


QVariant
TuningModel::displayData(const Tunable *tune, int col)const
{
	QVariant	ret;

	switch (col) {
		case LAB_NAME:
			ret = tune->prettyName();
			break;
		case LAB_VALUE:
			ret = tune->value();
			break;
	}

	return (ret);
}


QVariant
TuningModel::backgroundData(const Tunable *tune)const
{
	QVariant		ret;
	AbstractVariable	*var = tune->parent();

	if (var == NULL)
		ret = BRUSH_UNKNOWN;
	else switch (var->type()) {
		case AbstractVariable::ReadOnly:
			ret = BRUSH_READONLY;
			break;
		case AbstractVariable::ReadWrite:
			ret = BRUSH_READWRITE;
			break;
	}

	return (ret);
}


void
TuningModel::setFilter(const QString &f)
{

	filter = f;
	timeout();
}

/* EOF */
