/*
 * Copyright (c) 2009 by Denis Kozadaev (aka denk@RusNet)
 * All rights reserved
 *
 * $Id: cisco.cpp,v 0.1 2009/05/20 14:18:23 denis Exp $
 *
 * Author: Denis Kozadaev (denis@tambov.ru)
 * Description:
 *
 * See also: style(9), Qt manual pages
 *
 * Hacked by:
 */

#include "cisco.h"
#include "cisco_int.h"

#define	TIMER_INTERVAL	1000

static const QString
	eol("\r\n"),
	sh_int_cmd("sh int"),
	sh_bgp_cmd("sh ip bgp su");

static const Q_UINT16
	telnet = TELNET_PORT;

static const char
	*pwd = "Password: ",
	prompt1 = '>',
	prompt2 = '#';

extern QColor	alert;

Cisco::Cisco(QWidget *parent)
	:QScrollView(parent),
	normal(paletteBackgroundColor())
{

	cmd = SH_INT;	/*   */
	logged = FALSE;	/*  */
	sh_bgp = FALSE;	/*   BGP */
	bgp = NULL;

	setVScrollBarMode(QScrollView::Auto);
	setHScrollBarMode(QScrollView::AlwaysOff);

	iflist.setAutoDelete(TRUE);

	sock = new QSocket(this);
	timer = new QTimer(this);
	reconnect = new QTimer(this);

	menu = new QPopupMenu(this);
	menu->setCheckable(TRUE);

	connect(sock, SIGNAL(connected()), this, SLOT(ciscoEnable()));
	connect(sock, SIGNAL(readyRead()), this, SLOT(readData()));
	connect(sock, SIGNAL(connectionClosed()), this, SLOT(ciscoDown()));
	connect(sock, SIGNAL(error(int)), this, SLOT(ciscoError(int)));
	connect(timer, SIGNAL(timeout()), this, SLOT(ciscoCommand()));
	connect(reconnect, SIGNAL(timeout()), this, SLOT(ciscoReconnect()));
	connect(menu, SIGNAL(activated(int)), this, SLOT(chInterface(int)));

	setEnabled(FALSE);
}

Cisco::~Cisco()
{

	quitCommand();
	if (bgp != NULL)
		delete bgp;
	delete menu;
	delete reconnect;
	delete timer;
	delete sock;
}


void
Cisco::resizeEvent(QResizeEvent *e)
{

	QScrollView::resizeEvent(e);
	resizeChildren();
}


/*     */
void
Cisco::mousePressEvent(QMouseEvent *e)
{

	if (e->button() == Qt::RightButton) {
		/*    */
		menu->exec(e->globalPos());
	}
}


void
Cisco::setCisco(cisco_t &info)
{

	hst = info.host;
	usr = info.login;
	pass = info.password;
	sh_bgp = info.bgp;
	ciscoReconnect();
}


void
Cisco::quitCommand()
{

	sendCommand("quit");
}


void
Cisco::sendCommand(const QString &buf)
{
	QString	tmp(buf + eol);

	if (sock->state() == QSocket::Connected) {
		const char	*ptr = tmp.ascii();
		size_t		len = tmp.length();
		ssize_t		l;

		while (len > 0) {
			l = sock->writeBlock(ptr, len);
			if (l >= 0) {
				sock->flush();
				ptr += l;
				len -= l;
			} else if (l < 0)
				abort();
		}
	}
}


/*  ӣ,         */
void
Cisco::readAll()
{
	char	*buf, *ptr, *tmp;
	size_t	size;
	ssize_t	len, l;

	size = sock->bytesAvailable();
	if (size > 0) {
		buf = new char[size + 1];
		len = sock->readBlock(buf, size);
		for (ptr = buf; len > 0; ) {
			tmp = (char *)memchr(ptr, '\n', len);
			if (tmp != NULL) {
				*tmp = '\0';
				tmp++;
				l = tmp - ptr;
				addChunk(ptr, l);
				if (!tail.isEmpty()) {
					if (!isPrompt() &&
						(tail.find(pwd) < 0) &&
						!isCommand())
						input += tail;
					tail.truncate(0);
				}
			} else {
				ptr[len] = '\0';
				addChunk(ptr, len);
				break;
			}
			ptr = tmp;
			len -= l;
		}

		delete []buf;
	}
}


/*  ݣ      */
void
Cisco::addChunk(char *ptr, size_t len)
{
	char	*tmp;

	tmp = (char *)memchr(ptr, '\r', len);
	if (tmp != NULL)
		*tmp = '\0';

	tmp = strrchr(ptr, '\b');
	if (tmp != NULL)
		ptr = tmp + 1;

	if (((unsigned char)*ptr != 0xFF) && (*ptr != '\0'))
		tail += ptr;
}


/*       */
void
Cisco::readData()
{

	/*       */
	readAll();
	if (isMore()) {
		/*  ݣ    */
		tail.truncate(0);
		sendCommand(QString::null);
	} else if(isLogin()) {
		/*   */
		if (!usr.isEmpty())
			sendCommand(usr);
		else
			abort();	/*    */
	} else if(isPassword()) {
		if (!pass.isEmpty()) {
			sendCommand(pass);
			input.clear();
		} else
			abort();	/*    */
	} else if (isPrompt()) {
		if (logged) {
			/*   */
			if (!input.isEmpty())
				parseData();
		} else {
			/*    */
			logged = TRUE;
			tail.truncate(0);
			input.clear();
			timer->start(0, TRUE);
		}
	}
}


/*        */
void
Cisco::ciscoDown()
{

	setEnabled(FALSE);
	sock->close();
	reconnect->start(60000, TRUE);
}


/*     */
void
Cisco::ciscoError(int)
{

	setPaletteBackgroundColor(alert);
	emit showMe((QWidget *)this);
	emit statusDown();
	ciscoDown();
}


/*         "" */
void
Cisco::ciscoEnable()
{

	setEnabled(TRUE);
	setPaletteBackgroundColor(normal);
	emit showMe((QWidget *)this);
	emit statusUp();
}


/*
 *      
 *    
 */
void
Cisco::ciscoReconnect()
{

	cmd = SH_INT;
	sock->connectToHost(hst, telnet);
}


/*     cmd */
void
Cisco::ciscoCommand()
{

	switch (cmd) {
		case SH_INT:
			/*    */
			sendCommand(sh_int_cmd);
			break;
		case SH_BGP:
			/*  BGP */
			sendCommand(sh_bgp_cmd);
			break;
		default:
			/* NOOP */
			break;
	}
}


/*  TRUE,  ""     */
bool
Cisco::isPrompt()
{
	bool	ret = (tail.find(prompt1) > 0);

	if (ret != TRUE)
		ret = (tail.find(prompt2) > 0);

	return (ret);
}


/*  TRUE,       */
bool
Cisco::isMore()
{

	return (hasTail(" --More-- "));
}


/*  TRUE,       */
bool
Cisco::isLogin()
{

	return (tail == "login:");
}


/*  TRUE,      */
bool
Cisco::isPassword()
{

	return (tail == pwd);
}


/*  TRUE,      */
bool
Cisco::isCommand()
{
	bool	ret = (tail == sh_int_cmd);

	if (ret != TRUE)
		ret = (tail == sh_bgp_cmd);

	return (ret);
}


/*  TRUE,      */
bool
Cisco::hasTail(const char *str)
{

	return (tail.find(str, FALSE) == 0);
}


/*      */
Cisco::Command
Cisco::nextCommand(Cisco::Command cmd)
{

	if (cmd == SH_INT) {
		cmd = SH_BGP;
		if (sh_bgp != TRUE)
			cmd = nextCommand(cmd);
	} else if (cmd == SH_BGP)
		cmd = NOOP;
	else
		cmd = SH_INT;

	return (cmd);
}


/*     */
void
Cisco::parseInterfaces(QStringList &lst)
{
	QString				name, tmp;
	CiscoInt::InterfaceStatus	istat;
	CiscoInt::ProtocolStatus	pstat;
	CiscoInt			*ptr = NULL;

	for (QStringList::Iterator it = lst.begin(); it != lst.end(); ++it) {
		tmp = *it;

		if (tmp[0].isSpace()) {
			/*    */
			if (ptr != NULL) {
				QString	str(tmp.stripWhiteSpace());

				ptr->parseLine(str);
			}
		} else {
			/*   */
			CiscoInt::parseInterface(tmp, name, istat, pstat);
			if (ptr != NULL) {
				ptr->showInfo();
				menu->setItemChecked(ptr->menuId(),
					ptr->isShown());
			}
			ptr = findIface(name);
			if (ptr != NULL) {
				ptr->setStatus(istat);
				ptr->setStatus(pstat);
			}
		}
	}

	if (ptr != NULL) {
		ptr->showInfo();
		menu->setItemChecked(ptr->menuId(), ptr->isShown());
		resizeChildren();
	}
}


/*        */
CiscoInt *
Cisco::findIface(const QString &name)
{
	int		pos;
	CiscoInt	*ret;

	if (ifmap.contains(name)) {
		/*   */
		pos = ifmap[name];
		ret = iflist.at(pos);
	} else
		ret = NULL;
	if (ret == NULL) {
		int	id;

		/*   ݣ  */
		ret = new CiscoInt(this);
		id = menu->insertItem(name);
		ret->setMenuId(id);
		if (list.find(name) != list.end()) {
			addChild(ret);
			ret->show();
			menu->setItemChecked(id, TRUE);
		} else
			ret->showInterface(FALSE);
		ret->setTitle(name);
		iflist.append(ret);
		pos = iflist.find(ret);
		ifmap[name] = pos;
		resizeChildren();
	}

	return (ret);
}


/*   BGP */
void
Cisco::parseBGP(QStringList &lst)
{


	if (bgp == NULL) {
		/*  ,  ݣ   */
		bgp = new CiscoBGP(this);
		connect(bgp, SIGNAL(bgpUp()), this, SLOT(bgpUp()));
		connect(bgp, SIGNAL(bgpDown()), this, SLOT(bgpDown()));
		addChild(bgp);
		bgp->show();
		resizeChildren();
	}

	for (QStringList::Iterator it = lst.begin(); it != lst.end(); ++it) {
		QString	str((*it).stripWhiteSpace());

		bgp->parseLine(str);
	}
}


/*     */
void
Cisco::parseData()
{

	switch (cmd) {
		case SH_INT:
			parseInterfaces(input);
			break;
		case SH_BGP:
			parseBGP(input);
			break;
		default:
			cerr<<"  "<<endl;
	}

	input.clear();
	tail.truncate(0);
	cmd = nextCommand(cmd);
	if (cmd == NOOP) {
		cmd = nextCommand(cmd);
		timer->start(5000, TRUE);
	} else {
		timer->start(0, TRUE);
	}
}


/*       */
void
Cisco::resizeChildren()
{
	CiscoInt	*ptr;
	int		x, wdt, hgt;

	x = 0;
	wdt = visibleWidth() - x;
	hgt = 0;

	if (bgp != NULL) {
		moveChild(bgp, x, hgt);
		bgp->resize(wdt, bgp->height());
		hgt += bgp->height();
	}

	for (ptr = iflist.first(); ptr != NULL; ptr = iflist.next())
		if (ptr->isShown()) {
			moveChild(ptr, x, hgt);
			ptr->resize(wdt, ptr->height());
			hgt += ptr->height();
		} else
			removeChild(ptr);

	resizeContents(wdt, hgt);
}


/*    */
void
Cisco::saveSettings(QSettings *cfg, int ndx, const QString &cisco)
{
	QString	str(hst + ':' + usr + ':' + pass + ':');
	QString	name(CISCO + QString::number(ndx));
	int	i, cnt;

	if (sh_bgp)
		str += '1';
	else
		str += '0';

	cfg->writeEntry(CISCO_KEY"/" + name, str);

	str.truncate(0);
	cnt = menu->count();
	/*    */
	for (i = 0; i < cnt; i++) {
		int	id = menu->idAt(i);

		if (menu->isItemChecked(id)) {
			/*      */
			if (!str.isEmpty())
				str += ',';
			str += menu->text(id);
		}
	}
	cfg->writeEntry(CISCO_KEY"/" + cisco, str);
}


/*    */
void
Cisco::loadSettings(const QString &str)
{

	list = QStringList::split(',', str, FALSE);
}


/*  BGP- */
void
Cisco::bgpUp()
{

	emit showMe((QWidget *)this);
	emit statusUp();
}


/* BGP-  */
void
Cisco::bgpDown()
{

	emit showMe((QWidget *)this);
	emit statusDown();
	center(childX(bgp), childY(bgp));
}


/*     */
void
Cisco::chInterface(int id)
{
	QString		name = menu->text(id);
	bool		ena = menu->isItemChecked(id);
	int		pos = ifmap[name];
	CiscoInt	*iface = iflist.at(pos);

	if (iface != NULL) {
		ena ^= TRUE;
		menu->setItemChecked(id, ena);
		iface->showInterface(ena);
		resizeChildren();
	}
}


/*     */
void
Cisco::readCisco(QString &out)
{
	QString		data;
	QTextOStream	str(&data);
	CiscoInt	*ptr;

	str<<"<table border=\"0\" width=\"100%\" cellpadding=\"2\" "
	<<"cellspacing=\"1\" bgcolor=\"#000000\">"<<endl
	<<"<tr><th bgcolor=\"#0090FF\"><font color=\"#FFFFFF\">"
	<<hst<<"</font></th></tr>"<<endl
	<<"<tr><td bgcolor=\"#FFFFFF\" width=\"100%\">"<<endl;

	/*    */
	if (bgp != NULL) {
		QString	data;

		bgp->readData(data, hst);
		str<<data;
	}

	for (ptr = iflist.first(); ptr != NULL; ptr = iflist.next())
		if (ptr->isShown()) {
			QString	data;

			ptr->readData(data, hst);
			str<<data;
		}

	str<<"</td></tr>"<<endl
	<<"</table><br>"<<endl;
	out += data;
}


/*     */
void
Cisco::readImage(QByteArray &img, const QStringList &lst)
{
	QString	str(lst[0].upper());
	QChar	c = str[0];

	if (str == HTTP_BGP) {
		/*   BGP */
		if (bgp != NULL)
			bgp->readImage(img, lst[1]);
		else {
			QPixmap		tmp(32, 32);
			QPainter	*p;

			/* BGP ,    */
			tmp.fill(Qt::black);
			p = new QPainter(&tmp);
			p->setPen(QPen(Qt::red, 3));
			p->drawLine(0, 0, tmp.width(), tmp.height());
			p->drawLine(width(), 0, 0, height());
			delete p;

			QBuffer	buf(img);

			buf.open(IO_WriteOnly);
			tmp.save(&buf, "PNG");
			buf.close();
		}
	} else if ((c == 'I') || (c == 'O')) {
		int		pos = ifmap[lst[1]];
		CiscoInt	*iface = iflist.at(pos);

		/*   */
		if (iface != NULL)
			iface->readImage(img, c);
	}
}

/* EOF */
