/*
 * Copyright (c) 2009 by Denis Kozadaev (aka denk@RusNet)
 * All rights reserved
 *
 * $Id: cisco_bgp.cpp,v 0.1 2009/05/21 19:26:10 denis Exp $
 *
 * Author: Denis Kozadaev (denis@tambov.ru)
 * Description:
 *
 * See also: style(9), Qt manual pages
 *
 * Hacked by:
 */

#include "cisco_bgp.h"
#include "apply.xpm"
#include "cancel.xpm"

extern QColor	alert;

#define	PIXCOL	5

CiscoBGP::CiscoBGP(QWidget *parent)
	:QGroupBox(parent),
	normal(paletteBackgroundColor())
{

	old = NULL;	/*     */

	setAlignment(Qt::AlignHCenter);

	/*   */
	net = new MemLine(tr("Network entries"), this);

	/*   */
	path = new MemLine(tr("Path entries"), this);

	/*   BGP */
	bgp = new QListView(this);
	bgp->addColumn(tr("Neighbor"));
	bgp->addColumn(tr("AS"));
	bgp->addColumn(tr("MsgRcvd"));
	bgp->addColumn(tr("MsgSent"));
	bgp->addColumn(tr("Up/Down"));
	bgp->addColumn(tr("State/PfxRcd"));
	bgp->setAllColumnsShowFocus(TRUE);
	bgp->setSelectionMode(QListView::Single);
	bgp->setResizeMode(QListView::AllColumns);
	bgp->setColumnAlignment(0, Qt::AlignCenter);
	bgp->setColumnAlignment(1, Qt::AlignCenter);
	bgp->setColumnAlignment(2, Qt::AlignCenter);
	bgp->setColumnAlignment(3, Qt::AlignCenter);
	bgp->setColumnAlignment(4, Qt::AlignCenter);

	resizeBGP();
}

CiscoBGP::~CiscoBGP()
{

	delete bgp;
	delete path;
	delete net;
}


void
CiscoBGP::resizeEvent(QResizeEvent *e)
{
	int	wdt = e->size().width() - EXTRA,
		hgt = e->size().height() - EXTRA,
		fh = fontMetrics().lineSpacing() + EXTRA,
		y = fh;

	QGroupBox::resizeEvent(e);

	/*   */
	net->move(EXTRA, y);
	net->resize(wdt - net->x(), fh);
	y += fh;

	/*   */
	path->move(EXTRA, y);
	path->resize(wdt - path->x(), fh);
	y += fh;

	/*   BGP */
	bgp->move(EXTRA, y);
	bgp->resize(wdt - bgp->x(), hgt - bgp->y());
}


void
CiscoBGP::parseLine(QString &str)
{

	if (str.find("BGP router identifier") == 0) {
		QStringList	lst(QStringList::split(' ', str));

		/*  BGP- */
		setTitle("BGP: " + lst[3].remove(',') + " AS " + lst[7]);
	} else if (str.find("network entries using") > 0) {
		QStringList	lst(QStringList::split(' ', str));

		/*   */
		net->setData(lst[0], lst[4]);
	} else if (str.find("path entries using") > 0) {
		QStringList	lst(QStringList::split(' ', str));

		/*   */
		path->setData(lst[0], lst[4]);
	} else {
		addBGP(str);
	}

}


/*     BGP- */
void
CiscoBGP::addBGP(QString &str)
{
	QStringList	lst(QStringList::split(' ', str));
	QHostAddress	addr;

	if (addr.setAddress(lst[0])) {
		QListViewItem	*item = bgp->findItem(lst[0], 0);
		int		num = lst[9].toInt();
		char		**ptr = (num > 0)?apply:cancel;
		QPixmap		xpm((const char **)ptr);
		QWMatrix	mtx;
		double		fh = (double)bgp->fontMetrics().lineSpacing();

		if (item == NULL) {
			/*   ݣ   */
			item = new QListViewItem(bgp);
			resizeBGP();
		}
		item->setText(0, lst[0]);
		item->setText(1, lst[2]);
		item->setText(2, lst[3]);
		item->setText(3, lst[4]);
		item->setText(4, lst[8]);
		item->setText(5, lst[9]);
		mtx.scale(fh / xpm.width(), fh / xpm.height());
		item->setPixmap(PIXCOL, xpm.xForm(mtx));

		testStatus(ptr);
	}
}


void
CiscoBGP::resizeBGP()
{
	int	cnt = 2,
		hgt = fontMetrics().lineSpacing() + EXTRA +
		(fontMetrics().lineSpacing() + EXTRA) * cnt +
		bgp->header()->height() + 2 * EXTRA;

	if ((cnt = bgp->childCount()) > 0)
		hgt += (bgp->fontMetrics().lineSpacing() + 2 * EXTRA) * cnt;
	else
		hgt += (bgp->fontMetrics().lineSpacing() + EXTRA) * cnt;

	resize(width(), hgt);
}


/*     ,    */
void
CiscoBGP::testStatus(char **status)
{

	if (old == NULL)
		old = status;	/*    */
	else if (old != status) {
		/*   */
		if (status == apply) {
			/*  */
			setPaletteBackgroundColor(normal);
			emit bgpUp();
		} else if (status == cancel) {
			/*  */
			setPaletteBackgroundColor(alert);
			emit bgpDown();
		}
		old = status;
	}
}


/*  HTML-  ޣ  BGP */
void
CiscoBGP::readData(QString &out, const QString &cisco)
{
	QString		data;
	QTextOStream	str(&data);
	QListViewItem	*i;
	int		cnt;

	str<<"<table border=\"0\" width=\"100%\" bgcolor=\"#000000\" "
		"cellpadding=\"1\" cellspacing=\"1\">"<<endl
	<<"<tr bgcolor=\"#00FF90\"><th colspan=\"6\">"
		<<title()<<"</th></tr>"<<endl
	<<"<tr bgcolor=\"#FFFFFF\" align=\"center\">"
		<<net->text()<<"</tr>"<<endl
	<<"<tr bgcolor=\"#FFFFFF\" align=\"center\">"
		<<path->text()<<"</tr>"<<endl;
	str<<"<tr bgcolor=\"#00FF90\" align=\"center\">"
		<<"<td>Neighbor</td>"
		<<"<td>AS</td>"
		<<"<td>MsgRcvd</td>"
		<<"<td>MsgSent</td>"
		<<"<td>Up/Down</td>"
		<<"<td>State/PfxRcd</td>"
		<<"</tr>"<<endl;
	for (i = bgp->firstChild(); i != NULL; i = i->nextSibling()) {
		str<<"<tr bgcolor=\"#FFFFFF\" align=\"center\">";
		cnt = bgp->columns();
		for (int n = 0; n < cnt; n++) {
			str<<"<td";
			if (n == (cnt - 1)) {
				/*  ,   */
				str<<" align=\"left\"><img src=\""HTTP_IMG
				<<cisco<<","HTTP_BGP","<<i->text(0)
				<<"\">&nbsp;";
			} else
				str<<'>';
			str<<i->text(n)<<"</td>"<<endl;
		}
		str<<"</tr>"<<endl;
	}
	str<<"</table>"<<endl;
	out += data;
}


/*     */
void
CiscoBGP::readImage(QByteArray &img, const QString &ip)
{
	QListViewItem	*item = bgp->findItem(ip, 0);

	if (item != NULL) {
		const char	**ptr;

		if (item->text(PIXCOL).toULongLong() == 0) {
			/* active */
			ptr = (const char **)cancel;
		} else {
			/* up */
			ptr = (const char **)apply;
		}
		QImage	xpm(ptr);
		QBuffer	buf(img);

		buf.open(IO_WriteOnly);
		xpm.save(&buf, "PNG");
		buf.close();
	}
}

//-----------------------------------------------------------------------------

MemLine::MemLine(const QString &desc, QWidget *parent)
	:QWidget(parent)
{

	/*   */
	l_desc = new QLabel(desc + ' ', this);
	l_desc->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

	l_bm = new QLabel(' ' + tr("bytes of memory"), this);
	l_bm->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

	/*     */
	bm = new QLCDNumber(DIGITS, this);
	bm->setSegmentStyle(STYLE);

	l_using = new QLabel(tr("using"), this);
	l_using->setAlignment(Qt::AlignCenter);

	/*   */
	ent = new QLCDNumber(DIGITS, this);
	ent->setSegmentStyle(STYLE);
}

MemLine::~MemLine()
{

	delete ent;
	delete l_using;
	delete bm;
	delete l_bm;
	delete l_desc;
}


void
MemLine::resizeEvent(QResizeEvent *e)
{
	int	wdt = e->size().width(),
		hgt = e->size().height();

	QWidget::resizeEvent(e);

	l_desc->move(0, 0);

	l_bm->resize(fontMetrics().width(l_bm->text()), hgt);
	l_bm->move(wdt - l_bm->width(), 0);

	/*     */
	bm->resize(bm->numDigits() * (hgt * 3 / 7), l_bm->height());
	bm->move(l_bm->x() - bm->width(), l_bm->y());

	l_using->resize(fontMetrics().width(l_using->text()) + hgt,
		l_bm->height());
	l_using->move(bm->x() - l_using->width(), bm->y());

	ent->resize(bm->size());
	ent->move(l_using->x() - ent->width(), l_using->y());

	l_desc->resize(ent->x() - l_desc->x(), ent->height());
}


void
MemLine::setData(const QString &arg, const QString &mem)
{

	ent->display(arg);
	bm->display(mem);
}


QString
MemLine::text()const
{
	QString	ret;

	ret = "<td align=\"right\">" + l_desc->text() +
		"</td><td>" +
		QString::number(ent->intValue()) + "</td><td>" +
		l_using->text() + "</td><td>" +
		QString::number(bm->intValue()) +
		"</td><td colspan=\"2\" align=\"left\">" +
		l_bm->text() + "</td>";

	return (ret);
}

/* EOF */
