/*
 * Copyright (c) 2009 by Denis Kozadaev (aka denk@RusNet)
 * All rights reserved
 *
 * $Id: cisco_int.cpp,v 0.1 2009/05/20 16:58:27 denis Exp $
 *
 * Author: Denis Kozadaev (denis@tambov.ru)
 * Description:
 *
 * See also: style(9), Qt manual pages
 *
 * Hacked by:
 */

#include "cisco_int.h"

static const char
	desc_str[] = "Description: ",
	addr_str[] = "Internet address is ",
	last_str[] = "Last input ",
	irate_str[] = "5 minute input rate ",
	orate_str[] = "5 minute output rate ",
	pkt_inp_str[] = "packets input, ",
	pkt_out_str[] = "packets output, ",
	inp_err_str[] = "input errors",
	out_err_str[] = "output errors";

static const QColor
	color_up(0x75, 0xBC, 0x74),
	color_down(0xDE, 0x5E, 0x5E);

CiscoInt::CiscoInt(QWidget *parent)
	:QGroupBox(parent)
{
	const int	cnt = 3;
	int		hgt;

	flag = TRUE;
	setEnabled(FALSE);
	st_int = INT_DOWN;
	st_pro = PROTO_DOWN;
	clearInfo(input);
	clearInfo(output);

	/*   */
	l_desc = new QLabel(this);
	l_desc->setAlignment(Qt::AlignCenter);

	/*   */
	l_st_int = new QLabel(this);
	l_st_int->setAlignment(Qt::AlignCenter);

	/*   */
	l_st_pro = new QLabel(this);
	l_st_pro->setAlignment(Qt::AlignCenter);

	/*     */
	i_info = new IntInfo(tr("Incoming"), this);

	/*     */
	o_info = new IntInfo(tr("Outgoing"), this);

	hgt = (fontMetrics().lineSpacing() + EXTRA) * cnt + EXTRA +
		i_info->height();
	resize(width(), hgt);
}

CiscoInt::~CiscoInt()
{

	delete o_info;
	delete i_info;
	delete l_st_pro;
	delete l_st_int;
	delete l_desc;
}


void
CiscoInt::resizeEvent(QResizeEvent *e)
{
	int	wdt = e->size().width() - EXTRA,
		fh = fontMetrics().lineSpacing() + EXTRA,
		y = fh;

	QGroupBox::resizeEvent(e);

	/*   */
	l_desc->move(EXTRA, y);
	l_desc->resize(wdt - l_desc->x(), fh);
	y += fh;

	/*   */
	l_st_int->move(EXTRA, y);
	l_st_int->resize(wdt / 2 - l_st_int->x() - EXTRA, fh);

	/*   */
	l_st_pro->move(l_st_int->x() + l_st_int->width() + EXTRA, y);
	l_st_pro->resize(wdt - l_st_pro->x(), fh);
	y += fh;

	/*     */
	i_info->move(EXTRA, y);
	i_info->resize(wdt / 2 - i_info->x(), i_info->height());

	/*     */
	o_info->move(i_info->x() + i_info->width(), y);
	o_info->resize(wdt - o_info->x(), o_info->height());
}


/*          */
void
CiscoInt::parseInterface(QString &str, QString &name,
	InterfaceStatus &is, ProtocolStatus &ps)
{
	QStringList	lst(QStringList::split(' ', str));
	int		pos = 0;
	QString		tmp;

	name = lst[pos++];

	/* interface is ... */
	if (lst[pos] == "is")
		pos++;
	for (;;) {
		tmp = lst[pos++];
		if (tmp.find("up") >= 0)
			is = INT_UP;
		else if (tmp.find("down") >= 0)
			is = INT_DOWN;
		else if (tmp.isEmpty())
			break;
		else
			continue;
		break;
	}

	for (;;) {
		tmp = lst[pos++];
		if (tmp.find("up") >= 0)
			ps = PROTO_UP;
		else if (tmp.find("down") >= 0)
			ps = PROTO_DOWN;
		else if (tmp.isEmpty())
			break;
		else
			continue;
		break;
	}
}


/*    */
void
CiscoInt::setStatus(InterfaceStatus st)
{
	QColor	clr;
	QString	txt;

	st_int = st;
	switch (st_int) {
		case INT_UP:
			txt = "UP";
			clr = color_up;
			break;
		case INT_DOWN:
			txt = "Down";
			clr = color_down;
			break;
		default:
			clr = paletteBackgroundColor();
			txt = QString::null;
	}

	l_st_int->setText(tr("Interface is") + ' ' + txt);
	l_st_int->setPaletteBackgroundColor(clr);
}


void
CiscoInt::setStatus(ProtocolStatus st)
{
	QColor	clr;
	QString	txt;

	st_pro = st;
	switch (st_pro) {
		case PROTO_UP:
			txt = "UP";
			clr = color_up;
			break;
		case PROTO_DOWN:
			txt = "Down";
			clr = color_down;
			break;
		default:
			clr = paletteBackgroundColor();
			txt = QString::null;
	}

	l_st_pro->setText(tr("Line protocol is") + ' ' + txt);
	l_st_pro->setPaletteBackgroundColor(clr);
}


/*     */
void
CiscoInt::parseLine(QString &str)
{

	if (str.find(desc_str) == 0) {
		/*   */
		str.remove(0, sizeof(desc_str) - 1);
		desc = str;
		if (!addr.isEmpty())
			str = desc + ", " + addr;
		l_desc->setText(str);
	} else if (str.find(addr_str) == 0) {
		/* IP-  */
		str.remove(0, sizeof(addr_str) - 1);
		addr = str;
		if (!desc.isEmpty())
			str = desc + ", " + addr;
		l_desc->setText(str);
		setEnabled(!addr.isEmpty());
	} else if (str.find(last_str) == 0) {
		QStringList	lst(QStringList::split(' ', str));
		QString		t1(lst[2].remove(',')),
				t2(lst[4].remove(','));

		/*    */
		setTime(t1, t2);
	} else if (str.find(irate_str) == 0) {
		/*     5  */
		str.remove(0, sizeof(irate_str) - 1);
		setRate(input, str);
	} else if (str.find(orate_str) == 0) {
		/*     5  */
		str.remove(0, sizeof(orate_str) - 1);
		setRate(output, str);
	} else if (str.find(pkt_inp_str) > 0) {
		/*   */
		str.remove(pkt_inp_str);
		setTraffic(input, str);
	} else if (str.find(pkt_out_str) > 0) {
		/*   */
		str.remove(pkt_out_str);
		setTraffic(output, str);
	} else if (str.find(inp_err_str) > 0) {
		int	pos = str.find(' ');

		/*      */
		input.errs = str.left(pos).toULong();
	} else if (str.find(out_err_str) > 0) {
		int	pos = str.find(' ');

		/*      */
		output.errs = str.left(pos).toULong();
	}
}


/*     */
void
CiscoInt::setRate(iface_info_t &info, QString &str)
{
	int	pos = str.find(' ');

	info.rate = str.left(pos).toULong();
	str.remove(0, str.find(", ") + 2);
	pos = str.find(' ');
	info.speed = str.left(pos).toULong();
}


/*  ߣ   */
void
CiscoInt::setTraffic(iface_info_t &info, QString &str)
{
	QStringList	lst(QStringList::split(' ', str));

	info.pkts = lst[0].toULong();
	info.bytes = lst[1].toULong();
}


/*     */
void
CiscoInt::setTime(const QString &inp, const QString &out)
{
	QStringList	lst(QStringList::split(':', inp));

	if (lst.count() == 3)
		input.act = strTime(lst);
	else
		input.ena = FALSE;
	lst = QStringList::split(':', out);
	if (lst.count() == 3)
		output.act = strTime(lst);
	else
		output.ena = FALSE;
}


/*     ң  */
QTime
CiscoInt::strTime(const QStringList &lst)const
{

	return (QTime(lst[0].toInt(), lst[1].toInt(), lst[2].toInt()));
}


/*    */
void
CiscoInt::clearInfo(iface_info_t &info)
{

	info.rate = 0;
	info.speed = 0;
	info.pkts = 0;
	info.bytes = 0;
	info.errs = 0;
	info.ena = FALSE;
}


/*        */
void
CiscoInt::showInfo()
{

	if (flag) {
		if (addr.isEmpty()) {
			hide();
			emit needResize();
		} else if (isHidden()) {
			/*   */
			show();
			emit needResize();
		}
	}
	if (isShown()) {
		i_info->showInfo(input);
		o_info->showInfo(output);
	}
}


void
CiscoInt::showInterface(bool ena)
{

	if (ena)
		show();
	else
		hide();
	flag = ena;
}


void
CiscoInt::readData(QString &out, const QString &cisco)
{
	QString		data;
	QTextOStream	str(&data);

	str<<"<table border=\"0\" width=\"100%\" bgcolor=\"#000000\" "
		"cellpadding=\"1\" cellspacing=\"1\">"<<endl
	<<"<tr bgcolor=\"#00FFFF\"><th colspan=\"2\" align=\"left\">"
		<<title()<<"</th></tr>"<<endl
	<<"<tr bgcolor=\"#FFFFFF\"><td colspan=\"2\" align=\"center\">"
		<<l_desc->text()<<"</td></tr>"<<endl
	<<"<tr bgcolor=\"#75BC74\">"
		<<"<td align=\"center\"><b>"
			<<l_st_int->text()<<"</b></td>"
		<<"<td align=\"center\"><b>"
			<<l_st_pro->text()<<"</b></td></tr>"<<endl
	<<"<tr bgcolor=\"#FFFFFF\">"
		<<"<td>"<<i_info->text(cisco, title())<<"</td>"
		<<"<td>"<<o_info->text(cisco, title())<<"</td>"
		<<"</tr>"<<endl
	<<"</table>"<<endl;

	out += data;
}


/*    */
void
CiscoInt::readImage(QByteArray &img, const QChar &dir)
{

	if (dir == 'I')
		i_info->readImage(img);
	else if (dir == 'O')
		o_info->readImage(img);
}

//-----------------------------------------------------------------------------

IntInfo::IntInfo(const QString &ttl, QWidget *parent)
	:QGroupBox(ttl, parent)
{
	const int	cnt = 7;
	int		hgt;

	setAlignment(Qt::AlignHCenter);

	/*     5  */
	l_rate = new QLabel(tr("Rate bits/sec"), this);
	l_rate->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	rate = new QLCDNumber(DIGITS, this);
	rate->setSegmentStyle(STYLE);

	l_speed = new QLabel(tr("Rate packets/sec"), this);
	l_speed->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	speed = new QLCDNumber(DIGITS, this);
	speed->setSegmentStyle(STYLE);

	/*    */
	l_pkts = new QLabel(tr("Packets"), this);
	l_pkts->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	pkts = new QLCDNumber(DIGITS, this);
	pkts->setSegmentStyle(STYLE);

	/* ߣ   */
	l_traf = new QLabel(tr("Traffic (bytes)"), this);
	l_traf->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	traf = new QLCDNumber(DIGITS, this);
	traf->setSegmentStyle(STYLE);

	/*   */
	l_errs = new QLabel(tr("Errors"), this);
	l_errs->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	errs = new QLCDNumber(DIGITS, this);
	errs->setSegmentStyle(STYLE);

	/*   */
	l_idle = new QLabel(tr("Idle time"), this);
	l_idle->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	idle = new QLCDNumber(DIGITS, this);
	idle->setSegmentStyle(STYLE);

	drawer = new IntDrawer(this);

	hgt = 2 * EXTRA + (fontMetrics().lineSpacing() + EXTRA) * cnt + 99;
	resize(width(), hgt);
}

IntInfo::~IntInfo()
{

	delete drawer;
	delete idle;
	delete l_idle;
	delete errs;
	delete l_errs;
	delete traf;
	delete l_traf;
	delete pkts;
	delete l_pkts;
	delete speed;
	delete l_speed;
	delete rate;
	delete l_rate;
}


void
IntInfo::resizeEvent(QResizeEvent *e)
{
	int	wdt = e->size().width() - EXTRA,
		hgt = e->size().height() - EXTRA,
		fh = fontMetrics().lineSpacing() + EXTRA,
		y = fh, w2 = wdt / 2;

	QGroupBox::resizeEvent(e);

	/*     5  */
	moveLine(y, wdt, w2, fh, l_rate, rate);
	moveLine(y, wdt, w2, fh, l_speed, speed);

	/*    */
	moveLine(y, wdt, w2, fh, l_pkts, pkts);

	/* ߣ   */
	moveLine(y, wdt, w2, fh, l_traf, traf);

	/*   */
	moveLine(y, wdt, w2, fh, l_errs, errs);

	/*   */
	moveLine(y, wdt, w2, fh, l_idle, idle);

	/*  */
	drawer->move(EXTRA, y);
	drawer->resize(wdt - drawer->x(), hgt - drawer->y());
}


void
IntInfo::moveLine(int &y, int wdt, int w2, int fh,
	QLabel *lab, QLCDNumber *lcd)
{

	lab->setGeometry(EXTRA, y, w2 - EXTRA, fh);
	lcd->move(lab->x() + lab->width(), y);
	lcd->resize(wdt - lcd->x(), fh);
	y += fh;
}


/*    */
void
IntInfo::showInfo(const iface_info_t &info)
{

	rate_str = QString::number(info.rate);
	speed_str = QString::number(info.speed);
	idle_str = info.act.toString(TIME_FMT);
	pkts_str = QString::number(info.pkts);
	traf_str = QString::number(info.bytes);
	errs_str = QString::number(info.errs);

	rate->display(rate_str);
	speed->display(speed_str);

	pkts->display(pkts_str);
	traf->display(traf_str);

	errs->display(errs_str);

	idle->display(idle_str);

	drawer->insertRate(info.rate);
}


QString
IntInfo::text(const QString &cisco, const QString &iface)
{
	QString		ret, ttl(title());
	QTextOStream	str(&ret);

	str<<"<table border=\"0\" bgcolor=\"#000000\" cellpadding=\"1\" "
		<<"cellspacing=\"1\" width=\"100%\">"<<endl
	<<"<tr bgcolor=\"#3598BC\"><th colspan=\"2\">"
		<<ttl<<"</th></tr>"<<endl
	<<"<tr bgcolor=\"#FFFFFF\"><td>"<<l_rate->text()<<"</td>"
		<<"<td align=\"center\">"<<rate_str
		<<"</td></tr>"<<endl
	<<"<tr bgcolor=\"#FFFFFF\"><td>"<<l_speed->text()<<"</td>"
		<<"<td align=\"center\">"<<speed_str
		<<"</td></tr>"<<endl
	<<"<tr bgcolor=\"#FFFFFF\"><td>"<<l_pkts->text()<<"</td>"
		<<"<td align=\"center\">"<<pkts_str
		<<"</td></tr>"<<endl
	<<"<tr bgcolor=\"#FFFFFF\"><td>"<<l_traf->text()<<"</td>"
		<<"<td align=\"center\">"<<traf_str
		<<"</td></tr>"<<endl
	<<"<tr bgcolor=\"#FFFFFF\"><td>"<<l_errs->text()<<"</td>"
		<<"<td align=\"center\">"<<errs_str
		<<"</td></tr>"<<endl
	<<"<tr bgcolor=\"#FFFFFF\"><td>"<<l_idle->text()<<"</td>"
		<<"<td align=\"center\">"<<idle_str
		<<"</td></tr>"<<endl
	<<"<tr align=\"center\" bgcolor=\"#FFFFFF\"><td colspan=\"2\"><img src=\""HTTP_IMG
		<<cisco<<","<<ttl[0]<<","<<iface<<"\">"
		<<"</td></tr>"<<endl
	<<"</table>"<<endl;

	return (ret);
}


/*   */
void
IntInfo::readImage(QByteArray &img)
{

	img = drawer->image();
}

//-----------------------------------------------------------------------------

IntDrawer::IntDrawer(QWidget *parent)
	:QWidget(parent, NULL, Qt::WPaintClever | Qt::WNoAutoErase |
		Qt::WRepaintNoErase),
	kbps(tr("Kb/s")),
	fillColor(0x00, 0x00, 0x20),	/*   */
	gridColor(0x40, 0x40, 0x40),	/*   */
	color(0x00, 0x9B, 0xE9),	/*   */
	colorText(Qt::white)		/*   */
{

	min_rate = max_rate = 0;
	old_x = old_y = -1;	/*    */

	xpm = new QPixmap();
	list = new RateList();
	list->setAutoDelete(TRUE);
	setMouseTracking(TRUE);
}

IntDrawer::~IntDrawer()
{

	delete list;
	delete xpm;
}


void
IntDrawer::resizeEvent(QResizeEvent *e)
{

	QWidget::resizeEvent(e);
	xpm->resize(e->size());
	redraw();
}


void
IntDrawer::paintEvent(QPaintEvent *e)
{
	QPainter	*p;
	const QRect	&r = e->rect();

	p = new QPainter(this);
	if (!xpm->isNull())
		p->drawPixmap(r.topLeft(), *xpm, r);
	if ((old_x >= 0) && (old_x < width()) &&
		(old_y >= 0) && (old_y < height())) {
		point_t	*ptr = list->at(old_x);

		if (ptr != NULL) {
			int	align = Qt::AlignLeft | Qt::AlignTop,
				rate = countRate(old_y);
			/*   */
			p->setPen(colorText);
			p->drawText(0, 0, width(), height(), align,
				ptr->time.toString(TIME_FMT) + ", " +
				QString::number(rate) + ' ' + kbps);
		}
	}
	delete p;
}


/*       */
void
IntDrawer::mouseMoveEvent(QMouseEvent *e)
{
	int	x = e->x(),
		y = e->y();

	if ((x != old_x) || (y != old_y)) {
		/*   */
		old_x = x;
		old_y = y;
		update();
	}
}


/*       */
int
IntDrawer::countRate(int y)
{
	int	h = height(),
		m = max_rate - min_rate;

	return (((h - y) * m / h + min_rate) / 1000);
}


/*    XPM */
void
IntDrawer::redraw()
{
	const int	step = 10;
	int		i, e;
	QPainter	*p;

	if (!xpm->isNull()) {
		QPointArray	lst(xpm->width());

		xpm->fill(fillColor);
		padList();
		p = new QPainter(xpm);
		e = xpm->width() - 1;
		p->setPen(gridColor);
		for (i = 0; i < xpm->height(); i += step)
			p->drawLine(0, i, e, i);
		e = xpm->height() - 1;
		for (i = 0; i < xpm->width(); i += step)
			p->drawLine(i, 0, i, e);
		makeArray(lst);
		p->setPen(color);
		p->drawPolyline(lst);
		delete p;

		QBuffer	buf(img);

		buf.open(IO_WriteOnly);
		xpm->save(&buf, "PNG");
		buf.close();
	}
}


/*
 *      
 *   
 */
void
IntDrawer::padList()
{
	point_t	*ptr;

	while ((int)list->count() > xpm->width())
		list->remove((uint)0);

	while ((int)list->count() < xpm->width()) {
		ptr = new point_t();
		ptr->time = QTime::currentTime();
		ptr->rate = 0;
		list->prepend(ptr);
	}

	/*     */
	ptr = list->first();
	if (ptr != NULL)
		min_rate = max_rate = ptr->rate;
	else
		min_rate = max_rate = 0;
	for (; ptr != NULL; ptr = list->next()) {
		if (ptr->rate < min_rate)
			min_rate = ptr->rate;
		if (ptr->rate > max_rate)
			max_rate = ptr->rate;
	}
}


/*        */
void
IntDrawer::makeArray(QPointArray &lst)
{
	point_t	*i;
	int	x, y, m = max_rate - min_rate,
		h1 = xpm->height(), h = h1 - 1;

	for (x = 0, i = list->first(); i != NULL; i = list->next(), x++) {
		if (m == 0)
			y = h;
		else
			y = h - ((i->rate - min_rate) * h1 / m);
		lst.setPoint(x, x, y);
	}
}


/*      */
void
IntDrawer::insertRate(int r)
{
	point_t	*ptr;

	ptr = new point_t();
	ptr->time = QTime::currentTime();
	ptr->rate = r;
	list->append(ptr);
	redraw();
	update();
}

/* EOF */
