/*
 * Copyright (c) 2009 by Denis Kozadaev (aka denk@RusNet)
 * All rights reserved
 *
 * $Id: mainwindow.cpp,v 0.1 2009/05/20 14:09:34 denis Exp $
 *
 * Author: Denis Kozadaev (denis@tambov.ru)
 * Description:
 *
 * See also: style(9), Qt manual pages
 *
 * Hacked by:
 */

#include "mainwindow.h"
#include "cisco.h"
#include "cisco_dialog.h"

#include "neko.xpm"


MainWindow::MainWindow(QSettings *ptr, QWidget *parent)
	:QMainWindow(parent)
{

	setIcon(QPixmap((const char **)neko));
	grad = -1;

	manager = new Manager(this);

	cisco = new QPopupMenu(this);
	cisco->setCheckable(TRUE);
	add_id = cisco->insertItem(tr("Add"), this, SLOT(addCisco()),
		CTRL + Key_N);
	cisco->insertSeparator();
	full_id = cisco->insertItem(tr("FullScreen"), this, SLOT(fullScreen()),
		Key_F11);
	cisco->insertSeparator();
	cisco->insertItem(tr("Quit"), qApp, SLOT(quit()), CTRL + Key_Q);

	option = new QPopupMenu(this);
	option->insertItem(tr("Font"), this, SLOT(changeFont()));
	option->insertItem(tr("Sound device"), this, SLOT(changeDevice()));
	option->insertItem(tr("Mixer device"), this, SLOT(changeMixer()));
	option->insertItem(tr("Sound file"), this, SLOT(chooseFile()));
	option->insertItem(tr("Sound volume"), this, SLOT(soundVolume()));

	menuBar()->insertItem(tr("Cisco"), cisco);
	menuBar()->insertItem(tr("Options"), option);

	setCentralWidget(manager);

	info.sock = new QSocket(this);

	player = new Player(ptr);

	timer = new QTimer(this);

	connect(info.sock, SIGNAL(connected()), this, SLOT(testConnection()));
	connect(info.sock, SIGNAL(connectionClosed()),
		this, SLOT(connectionDone()));
	connect(info.sock, SIGNAL(error(int)), this, SLOT(socketError(int)));
	connect(timer, SIGNAL(timeout()), this, SLOT(mixerControl()));
	connect(manager, SIGNAL(startSound()), this, SLOT(startSound()));
	connect(manager, SIGNAL(stopSound()), this, SLOT(stopSound()));

	loadSettings(ptr);
	timer->start(1000);	/*     */
}

MainWindow::~MainWindow()
{

	saveSettings();
	player->saveSettings(cfg);
	delete timer;
	delete player;
	delete info.sock;
	delete option;
	delete cisco;
	delete manager;
}


/*      */
void
MainWindow::addCisco()
{
	CiscoDialog	*dlg;

	dlg = new CiscoDialog(this);
	if (dlg->exec()) {
		dlg->getData(info);
		info.srv = FALSE;
		info.sock->connectToHost(info.host, TELNET_PORT);
		cisco->setItemEnabled(add_id, FALSE);
		QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
	}
	delete dlg;
}


/*        */
void
MainWindow::testConnection()
{

	info.srv = TRUE;
	info.sock->close();
}


/*       */
void
MainWindow::connectionDone()
{

	QApplication::restoreOverrideCursor();
	if (info.srv == TRUE) {
		/*  ,   ,     */
		manager->insertCisco(info, cfg);
	} else if (info.err == QSocket::ErrConnectionRefused) {
		/* ,    */
		info.sock->close();
		QMessageBox::critical(this, tr("Connection refused"),
			tr("The host %1 does not have telnet service")
			.arg(info.host));
	} else if (info.err == QSocket::ErrHostNotFound) {
		info.sock->close();
		QMessageBox::critical(this, tr("Host not found"),
			tr("Host \"%1\" not found").arg(info.host));
	} else {
		QSocketDevice	*dev = info.sock->socketDevice();
		if (dev != NULL) {
			QSocketDevice::Error	err = dev->error();

			if (err == QSocketDevice::NetworkFailure) {
				info.sock->close();
				QMessageBox::critical(this,
					tr("Network failure"),
					tr("Cannot connect to host %1")
					.arg(info.host));
			}
		}
	}

	cisco->setItemEnabled(add_id, TRUE);
}


/*    ,    */
void
MainWindow::socketError(int err)
{

	info.srv = FALSE;
	info.err = (QSocket::Error)err;
	connectionDone();
}


/*    */
void
MainWindow::loadSettings(QSettings *ptr)
{

	cfg = ptr;

	manager->loadSettings(cfg);
}


/*     */
void
MainWindow::saveSettings()
{

	manager->saveSettings(cfg);
}


/*       */
void
MainWindow::fullScreen()
{

	if (cisco->isItemChecked(full_id)) {
		/*     */
		showNormal();
		cisco->setItemChecked(full_id, FALSE);
	} else {
		/*     */
		showFullScreen();
		cisco->setItemChecked(full_id, TRUE);
	}
}


/*    */
void
MainWindow::changeFont()
{
	bool	ok;
	QFont	fnt(QFontDialog::getFont(&ok, QApplication::font(), this));

	if (ok) {
		/*   */
		QApplication::setFont(fnt, TRUE);
	}
}


/*     */
void
MainWindow::changeDevice()
{
	bool	ok;
	QString	dsp(QInputDialog::getText(tr("Select a sound device"),
			tr("Specify a sound device to play sound"),
			QLineEdit::Normal, player->device(), &ok, this));

	if (ok) {
		QString	err(tr("Sound device error"));
		int	error;

		if (dsp.isEmpty())
			QMessageBox::information(this, err,
				tr("You did not specify a sound device.\n"
				"The sound will not be played"));
		else if ((error = player->testDevice(dsp)) != 0) {
			/*     */
			QMessageBox::critical(this, err,
				tr("Incorrect device name or specified device "
				"is not configured\n"
				"Device: %1\n"
				"Error: %2")
				.arg(dsp).arg(strerror(error)));
		} else {
			/*  ,     */
			player->setDevice(dsp);
		}
	}
}


/*      */
void
MainWindow::changeMixer()
{
	bool	ok;
	QString	dev(QInputDialog::getText(tr("Select a mixer device"),
			tr("Specify a mixer device to control sound volume"),
			QLineEdit::Normal, player->mixer(), &ok, this));

	if (ok) {
		QString	err(tr("Mixer device error"));
		int	error;

		if (dev.isEmpty())
			QMessageBox::information(this, err,
				tr("You did not specify a mixer device.\n"
				"The sound will not be controled"));
		else if ((error = player->testMixer(dev)) != 0) {
			/*     */
			QMessageBox::critical(this, err,
				tr("Incorrect device name or specified device "
				"is not configured\n"
				"Device: %1\n"
				"Error: %2")
				.arg(dev).arg(strerror(error)));
		} else {
			/*  ,     */
			player->setMixer(dev);
		}
	}
}


/*        */
void
MainWindow::mixerControl()
{
	QTime	t(QTime::currentTime());
	int	h = t.hour(), m = t.minute();

	/*        */
	if ((h < 8) || (h > 21) || ((h == 21) && (m > 50))) {
		/*  ,  ,     ;-) */
		player->setMute(TRUE);
	} else if (player->isMute())
		player->setMute(FALSE);

	if (grad == 0) {
		/*  ,    */
		player->moveNormal();
	} else if (grad > 0)
		grad--;
}


/*        */
void
MainWindow::startSound()
{

	if (!player->isMute()) {
		grad = 30;
		player->startSound();
		player->setAlarm();
	}
}


/*        */
void
MainWindow::stopSound()
{

	player->stopSound(grad);
}


/*     */
void
MainWindow::chooseFile()
{
	QString	file(QFileDialog::getOpenFileName(QString::null, "*.ogg", this,
			NULL, tr("Choose a sound file")));

	if (!file.isEmpty()) {
		QFileInfo	info(file);
		QString		err(tr("File error"));

		/*     */
		if (!info.exists()) {
			/*    */
			QMessageBox::critical(this, err,
				tr("File \"%1\" does not exist")
				.arg(file));
		} else if (!info.isReadable()) {
			/*      */
			QMessageBox::critical(this, err,
				tr("You don't have permission to read "
				"\"%1\"").arg(file));
		} else {
			player->setFile(file);
		}
	}
}


/*     */
void
MainWindow::soundVolume()
{
	VolumeDialog	*dlg;

	dlg = new VolumeDialog(this);
	dlg->setVolume(player->normalVolume(), player->alarmVolume());

	if (dlg->exec()) {
		/*    */
		player->setMixerVolume(dlg->normalVolume(),
			dlg->alarmVolume());
	}

	delete dlg;
}

//-----------------------------------------------------------------------------

VolumeDialog::VolumeDialog(QWidget *parent)
	:QDialog(parent)
{
	int	hgt;

	setCaption(tr("Set volume"));

	normal = new Volume(tr("Normal"), this);
	alarm = new Volume(tr("Alarm"), this);

	set = new QPushButton(tr("Set"), this);
	set->setFocusPolicy(QWidget::NoFocus);

	cancel = new QPushButton(tr("Cancel"), this);
	cancel->setFocusPolicy(QWidget::NoFocus);

	connect(normal, SIGNAL(volumeChanged(int)),
		this, SLOT(checkNormal(int)));
	connect(alarm, SIGNAL(volumeChanged(int)),
		this, SLOT(checkAlarm(int)));
	connect(set, SIGNAL(clicked()), this, SLOT(accept()));
	connect(cancel, SIGNAL(clicked()), this, SLOT(reject()));

	hgt = fontMetrics().lineSpacing() * 8;
	resize(hgt * 4 / 3, hgt);
	setMinimumSize(size());
	setMaximumSize(size());
}

VolumeDialog::~VolumeDialog()
{

	delete cancel;
	delete set;
	delete alarm;
	delete normal;
}


void
VolumeDialog::resizeEvent(QResizeEvent *e)
{
	int	wdt = e->size().width(),
		hgt = e->size().height(),
		w2 = wdt / 2,
		w3 = wdt / 3;

	QDialog::resizeEvent(e);

	cancel->resize(w3, cancel->fontMetrics().lineSpacing() * 7 / 5);
	set->resize(cancel->size());
	cancel->move(w2 + EXTRA, hgt - cancel->height());
	set->move(w2 - EXTRA - set->width(), cancel->y());

	normal->resize(w3, set->y() - EXTRA);
	normal->move(w2 - normal->width() - EXTRA, 0);
	alarm->resize(normal->size());
	alarm->move(w2, normal->y());
}


/*     */
void
VolumeDialog::setVolume(int n, int a)
{

	normal->setVolume(n);
	alarm->setVolume(a);
}


/*       */
void
VolumeDialog::checkNormal(int vol)
{

	if (vol > alarm->volume()) {
		/*
		 *      ,
		 *   ;-)
		 */
		alarm->setVolume(vol);
	}
}


/*      */
void
VolumeDialog::checkAlarm(int vol)
{

	if (vol < normal->volume()) {
		/*
		 *      ,
		 *    ;-)
		 */
		normal->setVolume(vol);
	}
}

//-----------------------------------------------------------------------------

Volume::Volume(const QString &ttl, QWidget *parent)
	:QGroupBox(ttl, parent)
{

	setAlignment(Qt::AlignHCenter);

	/*      */
	num = new QLCDNumber(3, this);
	num->setSegmentStyle(STYLE);

	dial = new QDial(this);
	dial->setMinValue(0);
	dial->setMaxValue(100);
	dial->setLineStep(2);
	dial->setPageStep(5);
	dial->setNotchesVisible(TRUE);
	dial->setTracking(TRUE);
	dial->setFocusPolicy(QWidget::NoFocus);

	connect(dial, SIGNAL(valueChanged(int)), num, SLOT(display(int)));
	connect(dial, SIGNAL(valueChanged(int)), SIGNAL(volumeChanged(int)));
}

Volume::~Volume()
{

	delete num;
}


void
Volume::resizeEvent(QResizeEvent *e)
{
	int	wdt = e->size().width() - EXTRA,
		hgt = e->size().height() - EXTRA,
		s = wdt - EXTRA;

	QGroupBox::resizeEvent(e);

	dial->resize(s, s);
	dial->move(EXTRA, hgt - dial->height());
	num->move(EXTRA, fontMetrics().lineSpacing() + EXTRA);
	num->resize(wdt - num->x(), dial->y() - num->y());
}


/*    */
void
Volume::setVolume(int vol)
{

	if (vol < 0)
		vol = 0;
	else if (vol > 100)
		vol = 100;

	dial->setValue(vol);
}

/* EOF */
