/*
 * Copyright (c) 2009 by Denis Kozadaev (aka denk@RusNet)
 * All rights reserved
 *
 * $Id: player.cpp,v 0.1 2009/05/22 20:10:12 denis Exp $
 *
 * Author: Denis Kozadaev (denis@tambov.ru)
 * Description:
 *
 * See also: style(9)
 *
 * Hacked by:
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>

#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "player.h"

Player::Player(QSettings *cfg)
	:QThread()
{

	events = 0;	/*   */
	dsp = cfg->readEntry(DSP_KEY, "/dev/dsp");
	mix = cfg->readEntry(MIXER_KEY, "/dev/mixer");
	alarm = cfg->readNumEntry(ALARM_KEY, 0x5A5A);	/* 90 */
	normal = cfg->readNumEntry(NORMAL_KEY, 0x3C3C);	/* 60 */
	file = cfg->readEntry(FILE_KEY, QString::null);

	/*       ,  ;-) */
	mute = FALSE;

	start();
}

Player::~Player()
{

	mtx.lock();
	events = -1;
	cond.wakeOne();
	mtx.unlock();

	wait();
}


/*        */
void
Player::setDevice(const QString &dev)
{

	mtx.lock();
	dsp = dev;
	mtx.unlock();
}


/*         */
void
Player::setMixer(const QString &dev)
{

	mix = dev;
}


void
Player::setMute(bool mt)
{
	int	dev;

	mute = mt;
	dev = openMixer(mix.ascii());
	if (dev >= 0) {
		if (mute) {
			/*     */
			setVolume(dev, 0);
		} else {
			/*     */
			setVolume(dev, normal);
		}
		closeMixer(dev);
	}
}


/*     */
void
Player::startSound()
{

	mtx.lock();
	events++;	/*      */
	if (events == 1)
		cond.wakeOne();	/*   */
	mtx.unlock();
}


/*    */
void
Player::stopSound(int &grad)
{

	mtx.lock();
	if (events > 0) {
		events--;
		if (events == 0)
			grad = -1;
	}
	mtx.unlock();
}


/*     */
void
Player::setAlarm()
{
	int	dev;

	dev = openMixer(mix.ascii());
	if (dev >= 0) {
		setVolume(dev, alarm);
		closeMixer(dev);
	}
}


/*      */
void
Player::setFile(const QString &f)
{

	mtx.lock();
	file = f;
	mtx.unlock();
}


/*       */
void
Player::moveNormal()
{
	int	dev, vol;
	int	left, right, norm;

	dev = openMixer(mix.ascii());
	if (dev >= 0) {
		if (mixerVolume(dev, vol) == 0) {
			/*  */
			left = vol & 0xFF;
			norm = normal & 0xFF;
			if (left > norm)
				vol--;

			right = vol & 0xFF00;
			norm = normal & 0xFF00;
			if (right > norm)
				vol -= 0x100;

			setVolume(dev, vol);
		}
		closeMixer(dev);
	}
}


/*    */
void
Player::setMixerVolume(int n, int a)
{
	int	tmp, vol, left;
	int	dev;

	/*   */
	tmp = (a & 0xFF);
	alarm = (tmp << 8) | tmp;

	/*    */
	tmp = (n & 0xFF);
	normal = (tmp << 8) | tmp;
	dev = openMixer(mix.ascii());
	if (dev >= 0) {
		if (mixerVolume(dev, vol) == 0) {
			left = vol & 0xFF;
			if (left < tmp)
				setVolume(dev, normal);
		}
		closeMixer(dev);
	}
}


/*    */
void
Player::saveSettings(QSettings *cfg)
{

	mtx.lock();
	cfg->writeEntry(DSP_KEY, dsp);
	cfg->writeEntry(FILE_KEY, file);
	mtx.unlock();
	cfg->writeEntry(MIXER_KEY, mix);
	cfg->writeEntry(ALARM_KEY, alarm);
	cfg->writeEntry(NORMAL_KEY, normal);
}

/*  TRUE,     */
int
Player::testDevice(const QString &dsp)
{
	int	dev, ret;

	mtx.lock();
	dev = openDevice(dsp.ascii());
	mtx.unlock();
	if (dev >= 0) {
		int	channels = 2,		/*   */
			fmt = AFMT_S16_LE,	/*   */
			speed = 44100;		/*   */

		/*     */
		if ((ret = resetDevice(dev)) != 0)
			closeDevice(dev);
		else if ((ret = setChannels(dev, channels)) != 0)
			closeDevice(dev);
		else if ((ret = setFormat(dev, fmt)) != 0)
			closeDevice(dev);
		else if ((ret = setSpeed(dev, speed)) != 0)
			closeDevice(dev);
		else {
			/*  ,  ݣ ? ;-) */
			closeDevice(dev);
			ret = 0;
		}
	} else
		ret = errno;	/* ,    */

	return (ret);
}


/*      */
int
Player::testMixer(const QString &mix)
{
	int	dev, ret;

	dev = openMixer(mix.ascii());
	if (dev >= 0) {
		int	vol;

		if ((ret = mixerVolume(dev, vol)) != 0)
			closeMixer(dev);
		else if ((ret = setVolume(dev, 0)) != 0)
			closeMixer(dev);
		else {
			/*   */
			setVolume(dev, vol);
			closeMixer(dev);
		}
	} else
		ret = errno;	/* ,    */

	return (ret);
}


/*        */
int
Player::openDevice(const char *device)
{
	int	ret;

	ret = open(device, O_WRONLY);

	return (ret);
}


/*      ,    */
int
Player::resetDevice(int dev)
{
	int	ret;

	if (ioctl(dev, SNDCTL_DSP_RESET, 0) < 0)
		ret = errno;
	else
		ret = 0;

	return (ret);
}


/*       */
int
Player::setChannels(int dev, int channels)
{
	int	ret;

	if (ioctl(dev, SNDCTL_DSP_CHANNELS, &channels) < 0)
		ret = errno;
	else
		ret = 0;

	return (ret);
}


/*       */
int
Player::setFormat(int dev, int fmt)
{
	int	ret;

	if (ioctl(dev, SNDCTL_DSP_SETFMT, &fmt) < 0)
		ret = errno;
	else
		ret = 0;

	return (ret);
}


/*        */
int
Player::setSpeed(int dev, int speed)
{
	int	ret;

	if (ioctl(dev, SNDCTL_DSP_SPEED, &speed) < 0)
		ret = errno;
	else
		ret = 0;

	return (ret);
}


/*      */
void
Player::closeDevice(int dev)
{

	resetDevice(dev);
	close(dev);
}


/*        */
int
Player::openMixer(const char *device)
{
	int	ret;

	ret = open(device, O_RDWR);

	return (ret);
}


/*    */
int
Player::mixerVolume(int dev, int &vol)
{
	int	ret;

	if (ioctl(dev, SOUND_MIXER_READ_PCM, &vol) < 0)
		ret = errno;
	else
		ret = 0;

	return (ret);
}


/*    */
int
Player::setVolume(int dev, int vol)
{
	int	ret;

	if (ioctl(dev, SOUND_MIXER_WRITE_PCM, &vol) < 0)
		ret = errno;
	else
		ret = 0;

	return (ret);
}


/*      */
void
Player::closeMixer(int dev)
{

	close(dev);
}


/*    */
void
Player::run()
{
	int	cnt;

	for (cnt = 0; cnt >= 0; ) {
		mtx.lock();
		if (events == 0)
			cond.wait(&mtx);
		cnt = events;
		mtx.unlock();
		if (cnt > 0)
			play(cnt);
	}
}


/*    */
int
Player::count()
{
	int	ret;

	mtx.lock();
	ret = events;
	mtx.unlock();

	return (ret);
}


/*      */
void
Player::play(int cnt)
{
	FILE		*ogg;
	OggVorbis_File	vf;
	int		res, dev;

	mtx.lock();
	ogg = fopen(file.ascii(), "rb");
	mtx.unlock();
	if (ogg != NULL) {
		/*   */

		res = ov_open_callbacks(ogg, &vf, NULL, 0,
			OV_CALLBACKS_NOCLOSE);
		if (res == 0) {
			dev = setupDevice(ov_info(&vf, -1));
			if (dev >= 0) {
				playFile(&vf, dev, cnt);
				closeDevice(dev);
			}
			ov_clear(&vf);
		}

		fclose(ogg);
	}
}


/*          */
int
Player::setupDevice(vorbis_info *vi)
{
	int	dev = -1;

	if (vi != NULL) {
		mtx.lock();
		dev = openDevice(dsp.ascii());
		mtx.unlock();
		if (dev >= 0) {
			/*   */
			setFormat(dev, AFMT_S16_LE);
			setChannels(dev, vi->channels);
			setSpeed(dev, vi->rate);
		}
	}

	return (dev);
}


/*     */
void
Player::playFile(OggVorbis_File *vf, int dev, int cnt)
{
	char	buf[4096];
	long	len = -1;
	ssize_t	res;
	int	bitstream, tmp;

	for (; ((tmp = count()) <= cnt) && (tmp > 0); ) {
		len = ov_read(vf, buf, sizeof(buf), 0, 2, 1, &bitstream);
		if (len <= 0)
			break;
		res = write(dev, buf, len);
		if (res < len)
			break;
	}
	if (len == 0) {
		/*  ,   ޣ  */
		mtx.lock();
		events = 0;
		mtx.unlock();
	}
}

/* EOF */
