/*
 *  Copyright (c) 2004
 *      Progressive Software Engineering Company
 */

#include <mysql/mysqld_error.h>

#include <cstdlib>
#include <cstring>
#include <iterator>

#include "database.h"
#include "common.h"

using std::insert_iterator;

/*
#ifndef CSHOP_SQL_DEBUG
# define CSHOP_SQL_DEBUG
#endif  // CSHOP_SQL_DEBUG
*/

#ifndef CSHOP_SQL_DEBUG
# define QUERY(db, str) \
    mysql_real_query(db, str.c_str(), str.size())
#else
int QUERY(MYSQL *db, const std::string & str)
{
    (void)printf("QUERY: %s\n", str.c_str());

    if (mysql_real_query(db, str.c_str(), str.size()))
    {
        (void)printf("ERROR: %s\n", mysql_error(db));
        return 1;
    }

    return 0;
}
#endif  // CSHOP_DEBUG

DB::DB() : _connected(false)
{
    mysql_init(&_db);
    _db.reconnect = true;   // reconnect automaticaly
}

void DB::close()
{
    mysql_close(&_db);
    _connected = false;
}

void DB::getGoodsGroups(GoodsGroup::Container & cont)
{
    if (!mysql_query(&_db, "SELECT IDENT, NAME FROM GOODS_GROUPS"))
    {
        MYSQL_RES *res = mysql_use_result(&_db);

        if (res)
        {
            MYSQL_ROW row;
            insert_iterator<GoodsGroup::Container> inserter(cont, cont.begin());

            while ((row = mysql_fetch_row(res)))
                if (row[0] && row[1])
                    *inserter++ = GoodsGroup(strtol(row[0], 0, 10), row[1]);

            mysql_free_result(res);
        }
    }
}

int DB::dropGoodsGroup(const int group_id, bool recursive)
{
    std::string query;
    char Str[32];

    INT_TO_STR(Str, group_id);

    if (!recursive)
    {
        // check dependencies ...
        query.assign("SELECT COUNT(DEBTS.DEBTOR) FROM DEBTS, GOODS WHERE "
                     "GOODS.IDENT = DEBTS.GOODS_ID AND GOODS.GRP = ").append(Str);

        if (QUERY(&_db, query))
            return 2;

        MYSQL_RES *res = mysql_use_result(&_db);

        if (res)
        {
            MYSQL_ROW row = mysql_fetch_row(res);
            int dep = 2;

            if (row && row[0])
                dep = strtol(row[0], 0, 10) > 0 ? 1 : 0;

            mysql_free_result(res);

            if (dep)
                return dep;
        }
    }

    if (!mysql_query(&_db, "BEGIN"))
    {
        query.assign("DELETE FROM GOODS_GROUPS WHERE IDENT = ").append(Str);

        if (!QUERY(&_db, query) &&
            !QUERY(&_db, (query.assign("DELETE SELLS.* FROM SELLS, GOODS WHERE SELLS.GOODS_ID = "
                                "GOODS.IDENT AND GOODS.GRP = ").append(Str))) &&
            !QUERY(&_db, (query.assign("DELETE BUYS.* FROM BUYS, GOODS WHERE BUYS.GOODS_ID = "
                                "GOODS.IDENT AND GOODS.GRP = ").append(Str))) &&
            !QUERY(&_db, (query.assign("DELETE DEBTS.* FROM DEBTS, GOODS WHERE DEBTS.GOODS_ID = "
                                "GOODS.IDENT AND GOODS.GRP = ").append(Str))) &&
            !QUERY(&_db, (query.assign("DELETE FROM GOODS WHERE GRP = ").append(Str))) &&
            !mysql_query(&_db, "COMMIT"))
        {
            return 0;
        }

        mysql_query(&_db, "ROLLBACK");
    }

    return 2;
}

int DB::createGoodsGroup(const char *group_name)
{
    std::string query = "INSERT INTO GOODS_GROUPS (NAME) VALUES ('";

    query.append(encode_sql_statement(group_name, strlen(group_name)).get()).append("')");

    if (!QUERY(&_db, query))
        return mysql_insert_id(&_db);

    return -1;
}

void DB::getGoods(const int group_id, Goods::Container & cont)
{
    std::string query = "SELECT IDENT, NAME, COST_PRICE, PRICE, QUANTITY, "
                        "CURDATE() - PRICE_DT FROM GOODS WHERE GRP = ";
    char Str[32];

    INT_TO_STR(Str, group_id);
    query += Str;

    if (!QUERY(&_db, query))
    {
        MYSQL_RES *res = mysql_use_result(&_db);

        if (res)
        {
            MYSQL_ROW row;
            insert_iterator<Goods::Container> inserter(cont, cont.begin());

            while ((row = mysql_fetch_row(res)))
                if (row[0] && row[1] && row[2] && row[3] && row[4])
                    *inserter++ = Goods(strtol(row[0], 0, 10), group_id, strtod(row[2], 0), strtod(row[3], 0),
                                         strtol(row[4], 0, 10), row[1], strtol(row[5], 0, 10) < 3);

            mysql_free_result(res);
        }
    }
}

bool DB::deleteFromTable(const char *table, const char *col, const char *value)
{
    std::string query = "DELETE FROM ";

    query.append(table).append(" WHERE ").append(col).append(" = ").append(value);
    return !QUERY(&_db, query);
}

int DB::dropGoods(const char *str_id, bool recursive)
{
    if (!recursive)
    {
        int ret = isGoodsHasDebtor(str_id);

        if (ret)
            return ret;
    }

    if (!mysql_query(&_db, "BEGIN"))
    {
        if (deleteFromTable("GOODS", "IDENT", str_id) &&
            deleteFromTable("BUYS", "GOODS_ID", str_id) &&
            deleteFromTable("SELLS", "GOODS_ID", str_id) &&
            !mysql_query(&_db, "COMMIT"))
        {
            return 0;
        }

        mysql_query(&_db, "ROLLBACK");
    }

    return 2;
}

bool DB::createGoods(const int group_id, const std::string & goods_id,
                     const std::string & goods_name)
{
    std::string query = "INSERT INTO GOODS VALUES (";

    query.append(goods_id).append(", '").append(
        encode_sql_statement(goods_name.c_str(), goods_name.size()).get()).append("', 0, 0, CURDATE(), 0, ");

    char Str[32];

    INT_TO_STR(Str, group_id);
    query.append(Str).append(")");

    return !QUERY(&_db, query);
}

bool DB::updateGoodsIdentTable(const char *old_id, const char *new_id,
                               const char *table, const char *column)
{
    std::string query = "UPDATE ";

    query.append(table).append(" SET ").append(column).append(" = ").append(new_id).append(" WHERE ").append(
        column).append(" = ").append(old_id);

    return !QUERY(&_db, query);
}

bool DB::updateGoodsIdent(const char *old_id, const char *new_id)
{
    if (!mysql_query(&_db, "BEGIN"))
    {
        if (updateGoodsIdentTable(old_id, new_id, "GOODS", "IDENT") &&
            updateGoodsIdentTable(old_id, new_id, "BUYS", "GOODS_ID") &&
            updateGoodsIdentTable(old_id, new_id, "SELLS", "GOODS_ID") &&
            updateGoodsIdentTable(old_id, new_id, "DEBTS", "GOODS_ID") &&
            !mysql_query(&_db, "COMMIT"))
        {
            return true;
        }

        mysql_query(&_db, "ROLLBACK");
    }

    return false;
}

bool DB::updateGoodsName(const char *gid, const std::string & name)
{
    std::string query = "UPDATE GOODS SET NAME = '";

    query.append(encode_sql_statement(name.c_str(), name.size()).get()).append("' WHERE IDENT = ").append(gid);
    return !QUERY(&_db, query);
}

bool DB::updateGoodsPrice(const char *gid, const char *price)
{
    std::string query = "UPDATE GOODS SET PRICE = ";

    query.append(price).append(", PRICE_DT = CURDATE() WHERE IDENT = ").append(gid);
    return !QUERY(&_db, query);
}

bool DB::buyGoods(const char *gid, std::string & quantity, std::string & price)
{
    if (!mysql_query(&_db, "BEGIN"))
    {
        std::string query = "UPDATE GOODS SET COST_PRICE = (COST_PRICE * QUANTITY + ";

        if (!QUERY(&_db, (query.append(quantity).append(" * ").append(price).append(
                ") / (QUANTITY + ").append(quantity).append("), QUANTITY = QUANTITY + ").append(
                quantity).append(" WHERE IDENT = ").append(gid))) &&
            !QUERY(&_db, (query.assign("INSERT INTO BUYS VALUES (NOW(), ").append(gid).append(
                ", ").append(quantity).append(", ").append(price).append(")"))) &&
            !QUERY(&_db, (query.assign("SELECT COST_PRICE, QUANTITY FROM GOODS WHERE IDENT = ").append(gid))))
        {
            MYSQL_RES *res = mysql_use_result(&_db);

            if (res)
            {
                MYSQL_ROW row = mysql_fetch_row(res);

                if (row)
                {
                    price = row[0];
                    quantity = row[1];
                }

                mysql_free_result(res);
            }

            if (!mysql_query(&_db, "COMMIT"))
                return true;
        }

        mysql_query(&_db, "ROLLBACK");
    }

    return false;
}

bool DB::sellGoods(const char *gid, int & quant, int max_quant)
{
    if (!mysql_query(&_db, "BEGIN"))
    {
        char Str[32];

        INT_TO_STR(Str, quant);

        std::string query = "INSERT INTO SELLS SELECT NOW(), ";

        if (!QUERY(&_db, (query.append(gid).append(", ").append(Str).append(
                    ", PRICE, PRICE - COST_PRICE FROM GOODS WHERE IDENT = ").append(gid))) &&
            !QUERY(&_db, (query.assign("UPDATE GOODS SET QUANTITY = QUANTITY - ").append(Str).append(
                    " WHERE IDENT = ").append(gid))))
        {
            quant = max_quant - quant;

            if (!mysql_query(&_db, "COMMIT"))
                return true;
        }

        mysql_query(&_db, "ROLLBACK");
    }

    return false;
}

bool DB::revertDillersGoods(const char* goods_id, const char* cost_price, const int quant)
{
    if (!mysql_query(&_db, "BEGIN"))
    {
        char Str[16];

        INT_TO_STR(Str, quant);

        std::string query = "UPDATE GOODS SET COST_PRICE = (COST_PRICE * QUANTITY + ";

        query.append(Str).append(" * ").append(cost_price).append(") / (QUANTITY + ").append(
            Str).append("), QUANTITY = QUANTITY + ").append(Str).append(" WHERE IDENT = ").append(goods_id);

        if (!QUERY(&_db, query))
        {
            query.assign("UPDATE DEBTS SET QUANTITY = QUANTITY - ").append(Str);

            if (!QUERY(&_db, query) && !mysql_query(&_db, "DELETE FROM DEBTS WHERE QUANTITY <= 0") &&
                !mysql_query(&_db, "COMMIT"))
            {
                return true;
            }
        }

        mysql_query(&_db, "ROLLBACK");
    }

    return false;
}

bool DB::sellDillersGoods(const char* goods_id, const int quant)
{
    if (!mysql_query(&_db, "BEGIN"))
    {
        std::string query = "INSERT INTO SELLS SELECT NOW(), ";

        char Str[16];
        INT_TO_STR(Str, quant);

        query.append(goods_id).append(", ").append(Str).append(
                ", PRICE, PRICE - COST_PRICE FROM DEBTS WHERE GOODS_ID = ").append(goods_id);

        if (!QUERY(&_db, query))
        {
            query.assign("UPDATE DEBTS SET QUANTITY = QUANTITY - ").append(Str);

            if (!QUERY(&_db, query) && !mysql_query(&_db, "DELETE FROM DEBTS WHERE QUANTITY <= 0") &&
                !mysql_query(&_db, "COMMIT"))
            {
                return true;
            }
        }

        mysql_query(&_db, "ROLLBACK");
    }

    return false;
}

void DB::getSellInfo(sell_account_info_t & sinfo, const int goods_id,
                     const std::string & from_date, const std::string & to_date,
                     const std::string & from_time, const std::string & to_time)
{
    std::string query = "SELECT SELL_DATE, GOODS_ID, QUANTITY, PRICE, PROFIT "
                        "FROM SELLS WHERE SELL_DATE >= '";

    query.append(from_date).append(query.size(), ' ').append(from_time).append(
        "' AND SELL_DATE <= '").append(to_date).append(query.size(), ' ').append(to_time);

    if (goods_id == -1)
        query += '\'';
    else
    {
        char Str[32];

        INT_TO_STR(Str, goods_id);
        query.append("' AND GOODS_ID = ").append(Str);
    }

    query.append(" ORDER BY SELL_DATE");

    if (!QUERY(&_db, query))
    {
        MYSQL_RES *res = mysql_use_result(&_db);

        if (res)
        {
            MYSQL_ROW row;
            insert_iterator<sell_account_info_t> inserter(sinfo, sinfo.begin());

            while ((row = mysql_fetch_row(res)))
                *inserter++ = SellAccountInfo_t(row[0], row[1], row[2], row[3], row[4]);

            mysql_free_result(res);
        }
    }
}

void DB::getBuyInfo(buy_account_info_t & binfo, const int goods_id,
                    const std::string & from_date, const std::string & to_date,
                    const std::string & from_time, const std::string & to_time)
{
    std::string query = "SELECT BUY_DATE, GOODS_ID, QUANTITY, PRICE FROM BUYS WHERE BUY_DATE >= '";

    query.append(from_date).append(query.size(), ' ').append(from_time).append("' AND BUY_DATE <= '").append(
        to_date).append(query.size(), ' ').append(to_time);

    if (goods_id == -1)
        query += '\'';
    else
    {
        char Str[32];

        INT_TO_STR(Str, goods_id);
        query.append("' AND GOODS_ID = ").append(Str);
    }

    query.append(" ORDER BY BUY_DATE");

    if (!QUERY(&_db, query))
    {
        MYSQL_RES *res = mysql_use_result(&_db);

        if (res)
        {
            MYSQL_ROW row;
            insert_iterator<buy_account_info_t> inserter(binfo, binfo.begin());

            while ((row = mysql_fetch_row(res)))
                *inserter++ = BuyAccountInfo_t(row[0], row[1], row[2], row[3]);

            mysql_free_result(res);
        }
    }
}

void DB::getDebtorList(Debtor::Container & cont)
{
    if (!mysql_query(&_db, "SELECT IDENT, NAME FROM DEBTORS"))
    {
        MYSQL_RES *res = mysql_use_result(&_db);

        if (res)
        {
            MYSQL_ROW row;
            insert_iterator<Debtor::Container> inserter(cont, cont.begin());

            while ((row = mysql_fetch_row(res)))
                *inserter++ = Debtor(strtol(row[0], 0, 10), row[1]);

            mysql_free_result(res);
        }
    }
}

bool DB::dropDebtor(const int id)
{
    if (!mysql_query(&_db, "BEGIN"))
    {
        char Str[32];

        INT_TO_STR(Str, id);

        std::string query = "DELETE FROM DEBTORS WHERE IDENT = ";

        query += Str;

        if (!QUERY(&_db, query))
        {
            query.assign("DELETE FROM DEBTS WHERE DEBTOR = ").append(Str);

            if (!QUERY(&_db, query) && !mysql_query(&_db, "COMMIT"))
                return true;
        }

        mysql_query(&_db, "ROLLBACK");
    }

    return false;
}

bool DB::getDebtorInfo(const int id, DebtorInfo & debtor)
{
    char Str[32];

    INT_TO_STR(Str, id);

    std::string query = "SELECT * FROM DEBTORS WHERE IDENT = ";

    query += Str;

    bool ret = false;

    if (!QUERY(&_db, query))
    {
        MYSQL_RES *res = mysql_use_result(&_db);

        if (res)
        {
            MYSQL_ROW row = mysql_fetch_row(res);

            if (row)
            {
                debtor.setId(strtol(row[0], 0, 10));
                debtor.setName(row[1]);
                debtor.setAddress(row[2]);
                debtor.setHomePhone(row[3]);
                debtor.setMobilePhone(row[4]);

                ret = true;
            }

            mysql_free_result(res);
        }
    }

    return ret;
}

bool DB::updateDebtorsTable(const int id, const char *var, const std::string & value)
{
    std::string query = "UPDATE DEBTORS SET ";

    query.append(var).append(" = '").append(encode_sql_statement(value.c_str(), value.size()).get());

    char Str[32];

    INT_TO_STR(Str, id);
    query.append("' WHERE IDENT = ").append(Str);

    return !QUERY(&_db, query);
}

bool DB::giveDebt(const int debtor_id, const char *goods_id, const char *price, const char *cost_price, const int quant)
{
    if (!mysql_query(&_db, "BEGIN"))
    {
        char Str_DebtId[16];

        INT_TO_STR(Str_DebtId, debtor_id);

        std::string query = "INSERT INTO DEBTS VALUES (";

        query.append(Str_DebtId).append(", ").append(goods_id);

        char Str[16];

        INT_TO_STR(Str, quant);

        query.append(", ").append(Str).append(", ").append(price).append(", ").append(cost_price).append(")");

        if (QUERY(&_db, query))
        {
            // insert failed, maybe this goods already in use ? trying to update it ...
            if (mysql_errno(&_db) == ER_DUP_ENTRY)
            {
                // update entry ...
                query.assign("UPDATE DEBTS SET PRICE = (QUANTITY * PRICE + ").append(Str).append(" * ").append(
                    price).append(") / (QUANTITY + ").append(Str).append(
                    "), COST_PRICE = (QUANTITY * COST_PRICE + ").append(Str).append(" * ").append(
                    cost_price).append(") / (QUANTITY + ").append(Str).append("), QUANTITY = QUANTITY + ").append(
                        Str).append(" WHERE GOODS_ID = ").append(goods_id).append(" AND DEBTOR = ").append(Str_DebtId);

                if (!QUERY(&_db, query))
                    goto _update_quant;
            }

            return false;
        }

_update_quant:  query.assign("UPDATE GOODS SET QUANTITY = QUANTITY - ").append(
                                Str).append(" WHERE IDENT = ").append(goods_id);

                if (!QUERY(&_db, query) && !mysql_query(&_db, "COMMIT"))
                    return true;

        mysql_query(&_db, "ROLLBACK");
    }

    return false;
}

void DB::getDebts(const int debtor_id, Debt::Container & cont)
{
    char d_Str[16];

    INT_TO_STR(d_Str, debtor_id);

    std::string query = "SELECT GOODS_ID, QUANTITY, PRICE, COST_PRICE FROM DEBTS WHERE DEBTOR = ";

    query += d_Str;

    if (!QUERY(&_db, query))
    {
        MYSQL_RES *res = mysql_use_result(&_db);

        if (res)
        {
            MYSQL_ROW row;
            insert_iterator<Debt::Container> inserter(cont, cont.begin());

            while ((row = mysql_fetch_row(res)))
                *inserter++ = Debt(strtol(row[0], 0, 10), strtol(row[1], 0, 10), strtof(row[2], 0), strtof(row[3], 0));

            mysql_free_result(res);
        }
    }
}

int DB::getCount(const char *table, const char *count_column,
                 const char *where_column, const char *expr)
{
    int ret = 2;
    std::string query = "SELECT COUNT(";

    query.append(count_column).append(") FROM ").append(table).append(
                " WHERE ").append(where_column).append(" = ").append(expr);

    if (!QUERY(&_db, query))
    {
        MYSQL_RES *res = mysql_use_result(&_db);

        if (res)
        {
            MYSQL_ROW row = mysql_fetch_row(res);

            if (row && row[0])
                ret = strtol(row[0], 0, 10) > 0 ? 1 : 0;

            mysql_free_result(res);
        }
    }

    return ret;
}

int DB::createDebtor(const DebtorInfo & info)
{
    std::string query = "INSERT INTO DEBTORS (NAME, ADDRESS, H_PHONE, M_PHONE) VALUES ('";

    query.append(encode_sql_statement(info.name().c_str(), info.name().size()).get()).append(
            "', '").append(encode_sql_statement(info.address().c_str(), info.address().size()).get()).append(
                "', '").append(encode_sql_statement(info.homePhone().c_str(), info.homePhone().size()).get()).append(
                    "', '").append(encode_sql_statement(info.mobilePhone().c_str(),
                                                                info.mobilePhone().size()).get()).append("')");

    if (!QUERY(&_db, query))
        return mysql_insert_id(&_db);

    if (mysql_errno(&_db) == ER_DUP_ENTRY)
        return -1;

    return -2;
}

