/*
 *  Copyright (c) 2004
 *      Progressive Software Engineering Company
 */

#ifndef _CSHOP_DATABASE_H_
#define _CSHOP_DATABASE_H_

#include <mysql/mysql.h>

#include <vector>
#include <string>
#include <memory>

#include "common.h"

class GoodsGroup
{ public:
    typedef std::vector<GoodsGroup> Container;

    GoodsGroup(const int c_id, const char *c_name) :
                _id(c_id), _name(c_name) { }

    int id() const { return _id; }
    const std::string & name() const { return _name; }

  private:
    int _id;
    std::string _name;
};

class Goods
{ public:
    typedef std::vector<Goods> Container;

    Goods(const int c_id, const int c_gid, const float c_cp, const float c_p, int _q, const char *name, const bool wu) :
            _id(c_id), _grp(c_gid), _quant(_q), _cprice(c_cp), _price(c_p), _name(name), _was_updated(wu) { }

    int id() const { return _id; }
    const std::string & name() const { return _name; }
    float cprice() const { return _cprice; }
    float price() const { return _price; }
    int quantity() const { return _quant; }
    bool was_updated() const { return _was_updated; }

  private:
    int _id, _grp, _quant;
    float _cprice, _price;
    std::string _name;
    bool _was_updated;
};

struct BuyAccountInfo_t
{
    BuyAccountInfo_t(const char* _date, const char* _ident,
                     const char* _quant, const char* _price) :
                        date(_date), ident(_ident), quantity(_quant), price(_price) { }

    std::string date, ident, quantity, price;
};

typedef std::vector<BuyAccountInfo_t> buy_account_info_t;

struct SellAccountInfo_t : public BuyAccountInfo_t
{
    SellAccountInfo_t(const char *_date, const char *_ident,
                      const char *_quant, const char *_price, const char *_profit) :
                        BuyAccountInfo_t(_date, _ident, _quant, _price), profit(_profit) { }

    std::string profit;
};

typedef std::vector<SellAccountInfo_t> sell_account_info_t;

class Debtor
{ public:
    typedef std::vector<Debtor> Container;

    Debtor(const int c_id) throw() : _id(c_id) { }
    Debtor(const int c_id, const char* const c_name) throw() : _id(c_id), _name(c_name) { }

    int id() const throw() { return _id; }
    const std::string & name() const throw() { return _name; }

    void setId(const int new_id) throw() { _id = new_id; }

    template <class StrType>
    void setName(const StrType str) { _name = str; }

  private:
    unsigned int _id;
    std::string _name;
};

class DebtorInfo : public Debtor
{ public:
    DebtorInfo() throw() : Debtor(-1) { }
    DebtorInfo(const int c_id, const char* const c_name, const char* const c_address,
               const char* const h_phone, const char* const m_phone) throw() : Debtor(c_id, c_name),
                _address(c_address), _phone(h_phone), _mob(m_phone) { }

    template <class StrType> void setAddress(const StrType str) { _address = str; }
    template <class StrType> void setHomePhone(const StrType _ph) { _phone = _ph; }
    template <class StrType> void setMobilePhone(const StrType _ph) { _mob = _ph; }

    const std::string address() const throw() { return _address; }
    const std::string homePhone() const throw() { return _phone; }
    const std::string mobilePhone() const throw() { return _mob; }

  private:
    std::string _address, _phone, _mob;
};

class Debt
{ public:
    typedef std::vector<Debt> Container;

    Debt() : _id(-1), _quantity(0), _price(0.00) { }
    Debt(const int c_id, const int c_quant, const float c_price, const float c_cost_price) :
            _id(c_id), _quantity(c_quant), _price(c_price), _cost_price(c_cost_price) { }

    int id() const { return _id; }
    int quantity() const { return _quantity; }
    float price() const { return _price; }
    float cost_price() const { return _cost_price; }

  private:
    int _id, _quantity;
    float _price, _cost_price;
};

class DB
{ public:
    DB();

    bool open(const std::string & host, const std::string & un, const std::string & up,
              const std::string & db_name, unsigned int & port, const std::string & unix_socket)
    {
        return _connected = mysql_real_connect(&_db, host.size() ? host.c_str() : 0, un.c_str(), up.c_str(),
                                               db_name.c_str(), port, unix_socket.size() ? unix_socket.c_str() : 0, 0);
    }

    void close();

    bool opened() const { return _connected; }

    const char *error_str() { return mysql_error(&_db); }
    int error_code() { return mysql_errno(&_db); }

    std::auto_ptr<char> encode_sql_statement(const char *str, const unsigned int size)
    {
        std::auto_ptr<char> to(new char [size * 2 + 1]);

        mysql_real_escape_string(&_db, to.get(), str, size);
        return to;
    }

    void getGoodsGroups(GoodsGroup::Container & cont);

    /**
     *  Drop goods group including all goods, buy, sell and debt log
     *
     *  @param  group_id    Goods group identifier
     *  @param  recursive   Don't check dependencies, remove all data ...
     *  @return 0 if removed success,
     *          1 if found dependencies and recursive param is false,
     *          2 on SQL error
    */
    int dropGoodsGroup(const int group_id, bool recursive = false);
    int createGoodsGroup(const char *group_name);

    void getGoods(const int group_id, Goods::Container & cont);

    /**
     *  Drop goods including all buy, sell and debt log
     *
     *  @param  str_id    Goods identifier
     *  @param  recursive   Don't check dependencies, remove all data ...
     *  @return 0 if removed success,
     *          1 if found dependencies and recursive param is false,
     *          2 on SQL error
    */
    int dropGoods(const char *str_id, bool recursive = false);
    bool createGoods(const int group_id, const std::string & goods_id,
                     const std::string & goods_name);

    bool updateGoodsIdent(const char *old_id, const char *new_id);
    bool updateGoodsName(const char *gid, const std::string & name);
    bool updateGoodsPrice(const char *gid, const char *price);
    bool buyGoods(const char *gid, std::string & quantity, std::string & price);
    bool sellGoods(const char *gid, int & quant, int max_quant);

    void getBuyInfo(buy_account_info_t & binfo, const int goods_id,
                    const std::string & from_date, const std::string & to_date,
                    const std::string & from_time, const std::string & to_time);

    void getSellInfo(sell_account_info_t & sinfo, const int goods_id,
                     const std::string & from_date, const std::string & to_date,
                     const std::string & from_time, const std::string & to_time);

    void getDebtorList(Debtor::Container & cont);

    bool dropDebtor(const int id);

    /**
     *  Creating new debtor ...
     *
     *  @param  info    Debtor information
     *  @return         Debtor identifier,
     *                  -1 on name dublicate,
     *                  -2 on other SQL error
    */
    int createDebtor(const DebtorInfo & info);

    bool getDebtorInfo(const int id, DebtorInfo & debtor);

    bool updateDebtorName(const int id, const std::string & name)
        { return updateDebtorsTable(id, "NAME", name); }

    bool updateDebtorAddress(const int id, const std::string & address)
        { return updateDebtorsTable(id, "ADDRESS", address); }

    bool updateDebtorHomePhone(const int id, const std::string & phone)
        { return updateDebtorsTable(id, "H_PHONE", phone); }

    bool updateDebtorMobilePhone(const int id, const std::string & phone)
        { return updateDebtorsTable(id, "M_PHONE", phone); }

    bool giveDebt(const int debtor_id, const char *goods_id, const char *price, const char *cost_price, const int quant);
    void getDebts(const int debtor_id, Debt::Container & cont);

    /**
     *  Check if any debtor has debt with specified identifier
     *
     *  @param  goods_id    Goods identifier
     *  @return 0 if this goods is not linked with any debtor,
     *          1 if this goods linked with one or more debtor,
     *          2 on SQL error
    */
    int isGoodsHasDebtor(const char *goods_id) { return getCount("DEBTS", "DEBTOR", "GOODS_ID", goods_id); }

    /**
     *  Check if debtor with sepcified identifier has any goods
     *
     *  @param  debtor_id   Debtor identifier
     *  @return 0 if this debtor hasn't any goods
     *          1 if has any ..
     *          2 on SQL error
    */
    int isDebtorHasGoods(const int debtor_id)
    {
        char Str[16];

        INT_TO_STR(Str, debtor_id);
        return getCount("DEBTS", "DEBTOR", "DEBTOR", Str);
    }

    bool sellDillersGoods(const char* goods_id, const int quant);

    // To revert diller's bid we must to buy goods from a diller ...
    bool revertDillersGoods(const char* goods_id, const char* cost_price, const int quant);

  private:
    MYSQL _db;
    bool _connected;

    bool updateDebtorsTable(const int id, const char *var, const std::string & value);
    bool updateGoodsIdentTable(const char *old_id, const char *new_id,
                               const char *table, const char *column);

    bool deleteFromTable(const char *table, const char *col, const char *value);
    int getCount(const char *table, const char *count_column, const char *where_column, const char *expr);
};

#endif  // _CSHOP_DATABASE_H_

