/*
 *  Copyright (c) 2004
 *      Progressive Software Engineering Company
 */

#include <qaction.h>
#include <qtable.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qpopupmenu.h>
#include <qinputdialog.h>

#include <cstdio>
#include <string>

#include "common.h"
#include "database.h"
#include "debtorwidget.h"

using std::string;

DebtorWidget::DebtorWidget(DB *db, const int deb_id, QWidget *parent) throw() :
    QWidget(parent), _db(db), debtor_ident(deb_id)
{
    QVBoxLayout *layout = new QVBoxLayout(this);

    layout->addWidget((debt_head = new DebtorInfoHatWidget(this)));
    connect(debt_head, SIGNAL(dataChanged(const DebtorInfoHatWidget::DataType)),
            this, SLOT(dataChanged(const DebtorInfoHatWidget::DataType)));

    // ---

    table = new QTable(0, 5, this);

    table->setSelectionMode(QTable::SingleRow);

    table->verticalHeader()->hide();
    table->setLeftMargin(0);

    QHeader *table_header = table->horizontalHeader();

    table_header->setLabel(IdCol, ENCODE("Идентификатор"));
    table->setColumnWidth(IdCol, 50);

    table_header->setLabel(QuantityCol, ENCODE("Кол-во"));
    table->setColumnWidth(QuantityCol, 50);

    table_header->setLabel(CPriceCol, ENCODE("Цена возвр."));
    table->setColumnWidth(CPriceCol, 70);

    table_header->setLabel(PriceCol, ENCODE("Цена"));
    table->setColumnWidth(PriceCol, 70);

    table_header->setLabel(PriceSumCol, ENCODE("Сумма"));
    table->setColumnWidth(PriceSumCol, 70);

    layout->addWidget(table);

    // ---

    DebtorInfo info;

    if (_db->getDebtorInfo(debtor_ident, info))
    {
        debt_head->name(ENCODE(info.name().c_str()));
        debt_head->address(ENCODE(info.address().c_str()));
        debt_head->homePhone(ENCODE(info.homePhone().c_str()));
        debt_head->mobilePhone(ENCODE(info.mobilePhone().c_str()));
    }

    // ---

    table_popup = new QPopupMenu(this);

    rasAct = new QAction(ENCODE("Расчитать"), ENCODE("&Расчитать"), CTRL+Key_R, this);
    connect(rasAct, SIGNAL(activated()), this, SLOT(slot_flushGoods()));
    rasAct->addTo(table_popup);

    revAct = new QAction(ENCODE("Вернуть"), ENCODE("&Вернуть"), CTRL+Key_V, this);
    connect(revAct, SIGNAL(activated()), this, SLOT(slot_revertGoods()));
    revAct->addTo(table_popup);

    connect(table, SIGNAL(contextMenuRequested(int, int, const QPoint &)),
            this, SLOT(slot_contextMenu(int, int, const QPoint &)));

    // ---

    Debt::Container debts;

    _db->getDebts(debtor_ident, debts);

    table->setNumRows(debts.size());

    for (u_int i = 0; i < debts.size(); i++)
        insertRow(i, debts[i]);
}

void DebtorWidget::insertRow(const int row, const Debt & debt)
{
    char Str[16];

    INT_TO_STR(Str, debt.id());
    table->setItem(row, IdCol, new QTableItem(table, QTableItem::Never, Str));

    INT_TO_STR(Str, debt.quantity());
    table->setItem(row, QuantityCol, new QTableItem(table, QTableItem::Never, Str));

    FLOAT_TO_PRICE_STR(Str, debt.cost_price());
    table->setItem(row, CPriceCol, new QTableItem(table, QTableItem::Never, Str));

    FLOAT_TO_PRICE_STR(Str, debt.price());
    table->setItem(row, PriceCol, new QTableItem(table, QTableItem::Never, Str));

    FLOAT_TO_PRICE_STR(Str, (debt.price() * debt.quantity()));
    table->setItem(row, PriceSumCol, new QTableItem(table, QTableItem::Never, Str));
}

void DebtorWidget::dataChanged(const DebtorInfoHatWidget::DataType type)
{
    string Str;

    switch (type)
    {
        case DebtorInfoHatWidget::NameData:
        {
            if (_db->updateDebtorName(debtor_ident, Str = debt_head->name().utf8()))
                emit debtorNameChanged(Str);
        } break;

        case DebtorInfoHatWidget::AddressData:
        {
            _db->updateDebtorAddress(debtor_ident, Str = debt_head->address().utf8());
        } break;

        case DebtorInfoHatWidget::hPhoneData:
        {
            _db->updateDebtorHomePhone(debtor_ident, Str = debt_head->homePhone().utf8());
        } break;

        case DebtorInfoHatWidget::mPhoneData:
        {
            _db->updateDebtorMobilePhone(debtor_ident, Str = debt_head->mobilePhone().utf8());
        } break;
    }
}

void DebtorWidget::slot_contextMenu(int row, int col, const QPoint & pos)
{
    bool isSingleItem = row >= 0;

    rasAct->setEnabled(isSingleItem);
    revAct->setEnabled(isSingleItem);

    table_popup->popup(pos);
}

void DebtorWidget::slot_revertGoods()
{
    QTableItem *quant_col = table->item(table->currentRow(), QuantityCol);

    bool ok;
    int max_quant = strtol(quant_col->text(), 0, 10),
        quant = QInputDialog::getInteger(ENCODE("Возврат товара ..."), ENCODE("Сколько единиц товара вернуть:"),
                                            1, 1, max_quant, 1, &ok, this);

    if (ok)
    {
        if (_db->revertDillersGoods(table->item(table->currentRow(), IdCol)->text(),
                                    table->item(table->currentRow(), CPriceCol)->text(), quant))
        {
            quant = max_quant - quant;

            if (quant > 0)
            {
                char Str[16];

                INT_TO_STR(Str, quant);
                quant_col->setText(Str);
                table->updateCell(table->currentRow(), QuantityCol);
            }
            else
                table->removeRow(table->currentRow());
        }
    }
}

void DebtorWidget::slot_flushGoods()
{
    QTableItem *quant_col = table->item(table->currentRow(), QuantityCol);

    bool ok;
    int max_quant = strtol(quant_col->text(), 0, 10),
        quant = QInputDialog::getInteger(ENCODE("Продано ..."), ENCODE("Сколько единиц данного товара продал диллер:"),
                                            1, 1, max_quant, 1, &ok, this);

    if (ok)
    {
        if (_db->sellDillersGoods(table->item(table->currentRow(), IdCol)->text(), quant))
        {
            quant = max_quant - quant;

            if (quant > 0)
            {
                char Str[16];

                INT_TO_STR(Str, quant);
                quant_col->setText(Str);
                table->updateCell(table->currentRow(), QuantityCol);
            }
            else
                table->removeRow(table->currentRow());
        }
    }
}

