/*
 *  Copyright (c) 2004
 *      Progressive Software Engineering Company
 */

#include <qheader.h>
#include <qpopupmenu.h>
#include <qaction.h>
#include <qlineedit.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qvalidator.h>
#include <qinputdialog.h>
#include <qmessagebox.h>
#include <qcombobox.h>
#include <qhbox.h>

#include <climits>
#include <memory>

#include "goodstable.h"
#include "common.h"
#include "topwin.h"
#include "goodsdialog.h"

AddGoodsDialog::AddGoodsDialog(QWidget *parent) : QDialog(parent)
{
    setCaption(ENCODE("Создание продукта"));

    QGridLayout *layout = new QGridLayout(this, 1, 1, 5, 0);

    QGroupBox *groupBox = new QGroupBox(this);
    layout->addWidget(groupBox, 0, 0);

    layout = new QGridLayout(groupBox, 1, 1, 5, 5);

    goodsIdEdit = new QSpinBox(0, INT_MAX, 1, groupBox);

    QLabel *label = new QLabel(ENCODE("Идент.:"), groupBox);
    label->setAlignment(DEF_LAB_ALIGN);

    layout->addWidget(label, 0, 0);
    layout->addMultiCellWidget(goodsIdEdit, 0, 0, 1, 2);

    goodsNameEdit = new QLineEdit(groupBox);

    label = new QLabel(ENCODE("Наимен.:"), groupBox);
    label->setAlignment(DEF_LAB_ALIGN);

    layout->addWidget(label, 1, 0);
    layout->addMultiCellWidget(goodsNameEdit, 1, 1, 1, 2);

    QSpacerItem *spacer = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);
    layout->addItem(spacer, 3, 1);

    QFrame *line = new QFrame(groupBox);
    line->setFrameShape(QFrame::HLine);
    layout->addMultiCellWidget(line, 4, 4, 0, 2);

    QWidget *buttonGroup = new QWidget(groupBox);

    QHBoxLayout *box_layout = new QHBoxLayout(buttonGroup, 0, 5);

    OkButton = new QPushButton(ENCODE("Создать"), buttonGroup);
    OkButton->setEnabled(false);
    box_layout->addWidget(OkButton);

    QPushButton *button = new QPushButton(ENCODE("Отменить"), buttonGroup);
    box_layout->addWidget(button);

    layout->addWidget(buttonGroup, 5, 2);

    adjustSize();
    clearWState(WState_Polished);

    connect(goodsNameEdit, SIGNAL(textChanged(const QString &)), this, SLOT(checkInput()));
    connect(button, SIGNAL(clicked()), this, SLOT(reject()));
    connect(OkButton, SIGNAL(clicked()), this, SLOT(accept()));
}

void AddGoodsDialog::checkInput()
{
    OkButton->setEnabled(goodsNameEdit->text().length());
}

void AddGoodsDialog::getData(std::string & goods_id, std::string & goods_name) const
{
    goods_id = goodsIdEdit->text().utf8();
    goods_name = goodsNameEdit->text().utf8();
}

DebtorSelection::DebtorSelection(DB *db, const int max_value, QWidget *parent) :
    QDialog(parent), _DillerIdent(-1)
{
    setCaption(ENCODE("Передача диллеру"));

    QGridLayout *layout = new QGridLayout(this, 1, 1, 5, 5);

    // ---

    QLabel *label = new QLabel(ENCODE("Диллер:"), this);
    label->setAlignment(DEF_LAB_ALIGN);

    layout->addWidget(label, 0, 0);

    dillers_box = new QComboBox(false, this);
    layout->addWidget(dillers_box, 0, 1);
    connect(dillers_box, SIGNAL(activated(int)), this, SLOT(slot_activated(int)));

    db->getDebtorList(debtors);

    for (Debtor::Container::const_iterator i = debtors.begin(); i != debtors.end(); i++)
        dillers_box->insertItem(ENCODE(i->name().c_str()));

    // ---

    label = new QLabel(ENCODE("Количество:"), this);

    label->setAlignment(DEF_LAB_ALIGN);
    layout->addWidget(label, 1, 0);

    layout->addWidget((quant_spin = new QSpinBox(0, max_value, 1, this)), 1, 1);
    connect(quant_spin, SIGNAL(valueChanged(int)), this, SLOT(slot_spinValueChanged(int)));

    // ---

    QSpacerItem *spacer = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum);

    layout->addItem(spacer, 2, 0);

    QHBox *hBox = new QHBox(this);

    hBox->setSpacing(3);

    okButton = new QPushButton(ENCODE("Передать"), hBox);

    okButton->setEnabled(false);
    connect(okButton, SIGNAL(clicked()), this, SLOT(slot_confirm()));

    QPushButton *button = new QPushButton(ENCODE("Отменить"), hBox);
    connect(button, SIGNAL(clicked()), this, SLOT(reject()));

    layout->addWidget(hBox, 2, 1);

    // ---

    adjustSize();
    clearWState(WState_Polished);
}

void DebtorSelection::slot_activated(int index)
{
    okButton->setEnabled(index >= 0 && quant_spin->value() > 0);
}

void DebtorSelection::slot_spinValueChanged(int val)
{
    okButton->setEnabled(val > 0 && dillers_box->currentItem() >= 0);
}

void DebtorSelection::slot_confirm()
{
    _DillerIdent = debtors[dillers_box->currentItem()].id();

    accept();
}

bool DebtorSelection::getSellDillerInfo(DB *db, const int max_value, int & ident, int & quant, QWidget *parent)
{
    std::auto_ptr<DebtorSelection> dialog(new DebtorSelection(db, max_value, parent));

    if (dialog->exec() == QDialog::Accepted)
    {
        ident = dialog->dillerIdent();
        quant = dialog->goodsQuantity();

        return true;
    }

    return false;
}

GoodsTable::GoodsTable(QWidget *parent, DB *c_db, const int c_gg, TopWin *tw) :
    QTable(0, 6, parent), _db(c_db), group_id(c_gg), _tw(tw)
{
    setSelectionMode(QTable::SingleRow);

    verticalHeader()->hide();
    setLeftMargin(0);

    QHeader *table_header = horizontalHeader();

    table_header->setLabel(IdCol, ENCODE("Идент."));
    setColumnWidth(IdCol, 50);

    table_header->setLabel(NameCol, ENCODE("Наимен."));
    setColumnWidth(NameCol, 200);

    table_header->setLabel(CPriceCol, ENCODE("Себе ст."));
    setColumnWidth(CPriceCol, 70);

    table_header->setLabel(PercCol, ENCODE("Процент"));
    setColumnWidth(PercCol, 70);

    table_header->setLabel(PriceCol, ENCODE("Стоимость"));
    setColumnWidth(PriceCol, 80);

    table_header->setLabel(QuantityCol, ENCODE("Кол-во"));
    setColumnWidth(QuantityCol, 70);

    // setup popup menu
    popupMenu = new QPopupMenu(this);

    QAction *action = new QAction(ENCODE("Добавить"), ENCODE("Добавить"), CTRL+Key_N, this);
    connect(action, SIGNAL(activated()), this, SLOT(slot_addGoods()));
    action->addTo(popupMenu);

    delGoodsAct = new QAction(ENCODE("Удалить"), ENCODE("У&далить"), CTRL+Key_D, this);
    connect(delGoodsAct, SIGNAL(activated()), this, SLOT(slot_dropGoods()));
    delGoodsAct->addTo(popupMenu);

    popupMenu->insertSeparator();

    action = new QAction(ENCODE("Перейти"), ENCODE("Перейти"), CTRL+Key_G, this);
    connect(action, SIGNAL(activated()), this, SLOT(slot_gotoRow()));
    action->addTo(popupMenu);

    popupMenu->insertSeparator();

    sellGoodsAct = new QAction(ENCODE("Продать"), ENCODE("Продать"), CTRL+Key_S, this);
    connect(sellGoodsAct, SIGNAL(activated()), this, SLOT(slot_sellGoods()));
    sellGoodsAct->addTo(popupMenu);

    buyGoodsAct = new QAction(ENCODE("Закупить"), ENCODE("Закупить"), CTRL+Key_B, this);
    connect(buyGoodsAct, SIGNAL(activated()), this, SLOT(slot_buyGoods()));
    buyGoodsAct->addTo(popupMenu);

    giveDebtAct = new QAction(ENCODE("Передать диллеру"), ENCODE("Передать диллеру"), CTRL+Key_P, this);
    connect(giveDebtAct, SIGNAL(activated()), this, SLOT(slot_giveDebt()));
    giveDebtAct->addTo(popupMenu);

    popupMenu->insertSeparator();

    view_sp_col_id = popupMenu->insertItem(ENCODE("Показывать все колонки"), this,
                                           SLOT(slot_showCols()), CTRL+SHIFT+Key_H);
    popupMenu->setItemChecked(view_sp_col_id, true);
    // ---

    connect(this, SIGNAL(contextMenuRequested(int, int, const QPoint &)),
            this, SLOT(slot_contextMenu(int, int, const QPoint &)));

    connect(this, SIGNAL(currentChanged(int, int)), this, SLOT(slot_storeValue(int, int)));
    connect(this, SIGNAL(valueChanged(int, int)), this, SLOT(slot_updateValue(int, int)));

    // load data
    Goods::Container cont;

    _db->getGoods(group_id, cont);

    for (Goods::Container::size_type s = 0; s < cont.size(); s++)
    {
        int rws = numRows();

        setNumRows(rws + 1);
        insert(rws, cont[s]);
    }

    if (cont.size())
        setCurrentCell(0, 0);

    // ---
}

void GoodsTable::paintCell(QPainter *p, int row, int col, const QRect & cr, bool selected, const QColorGroup & cg)
{
    if (col == PriceCol)
    {
        PriceTableItem *Pit = dynamic_cast<PriceTableItem *>(item(row, col));

        if (Pit && Pit->was_updated())
        {
            QColorGroup bc(cg);

            bc.setColor(QColorGroup::Base, QColor(255, 197, 61));
            QTable::paintCell(p, row, col, cr, selected, bc);
            return;
        }
    }
    else if (col == QuantityCol)
    {
        QTableItem *TItem = item(row, col);

        if (TItem && strtol(TItem->text(), 0, 10) <= 3)
        {
            QColorGroup bc(cg);

            bc.setColor(QColorGroup::Base, QColor(80, 255, 80));
            QTable::paintCell(p, row, col, cr, selected, bc);
            return;
        }
    }
    else if (col == PercCol)
    {
        QTableItem *TItem = item(row, col);

        if (TItem && strtol(TItem->text(), 0, 10) <= 0)
        {
            QColorGroup bc(cg);

            bc.setColor(QColorGroup::Base, QColor(255, 0, 0));
            QTable::paintCell(p, row, col, cr, selected, bc);
            return;
        }
    }

    QTable::paintCell(p, row, col, cr, selected, cg);
}

void GoodsTable::slot_gotoRow()
{
    bool ok;
    int goods_id = QInputDialog::getInteger(ENCODE("Поиск по номеру товара"), ENCODE("Введите номер:"),
                    0, 0, INT_MAX, 1, &ok, this);

    if (ok)
    {
        char Str[32];

        (void)snprintf(Str, sizeof(Str) - 1, "%d", goods_id);

        for (int i = 0; i < numRows(); i++)
            if (!item(i, 0)->text().compare(Str))
            {
                setCurrentCell(i, 0);
                break;
            }
    }
}

void GoodsTable::slot_contextMenu(int row, int col, const QPoint & pos)
{
    bool isSingleItem = row >= 0;

    delGoodsAct->setEnabled(isSingleItem);
    buyGoodsAct->setEnabled(isSingleItem);

    bool canSell = isSingleItem ? strtol(item(row, QuantityCol)->text(), 0, 10) > 0 : false;

    sellGoodsAct->setEnabled(canSell);
    giveDebtAct->setEnabled(canSell);

    popupMenu->popup(pos);
}

void GoodsTable::slot_addGoods()
{
    std::auto_ptr<AddGoodsDialog> dialog(new AddGoodsDialog(this));

    if (dialog->exec() == QDialog::Accepted)
    {
        std::string goods_id, goods_name;

        dialog->getData(goods_id, goods_name);

        if (_db->createGoods(group_id, goods_id, goods_name))
        {
            insert(goods_id, goods_name);
            _tw->writeMessage("Новый товар добавлен");
        }
        else
            _tw->writeSQLError(_db->error_code(), _db->error_str());
    }
}

void GoodsTable::slot_dropGoods()
{
    int row = currentRow();
    QTableItem *id = item(row, IdCol);

    if (id && QMessageBox::question(this, ENCODE("Удаление ..."),
                                    ENCODE("Вы действительно хотите удалить продукт ?"),
                                    ENCODE("Да"), ENCODE("Нет"), QString::null, 1, 1) == 0)
    {
        int ret = _db->dropGoods(id->text());

        if (!ret)
        {
            removeRow(row);
            setCurrentCell(row, 0);
            _tw->writeMessage("Товар удален");
        }
        else if (ret == 1)
            QMessageBox::information(this, ENCODE("Обнаружена зависимость ..."),
                ENCODE("Извините, но вы не можете товар, пока он находятся на продаже у диллеров !"));
        else
            QMessageBox::information(this, ENCODE("Ошибка ..."),
                ENCODE("Произошла ошибка СУБД, невозможно удалить товар."));
    }
}

void GoodsTable::slot_showCols()
{
    if (popupMenu->isItemChecked(view_sp_col_id))
    {
        hideColumn(CPriceCol);
        hideColumn(PercCol);
        popupMenu->setItemChecked(view_sp_col_id, false);
    }
    else
    {
        showColumn(CPriceCol);
        showColumn(PercCol);
        popupMenu->setItemChecked(view_sp_col_id, true);
    }
}

void GoodsTable::slot_storeValue(int row, int col)
{
    QTableItem *s_item = item(row, col);

    if (s_item)
        storred_value = s_item->text();
}

void GoodsTable::slot_giveDebt()
{
    int row = currentRow();
    QTableItem *id, *pr, *qnt, *cost_price;

    if ((id = item(row, IdCol)) && (pr = item(row, PriceCol)) && (qnt = item(row, QuantityCol)) &&
        (cost_price = item(row, CPriceCol)))
    {
        int ident, quant;

        if (DebtorSelection::getSellDillerInfo(_db, strtol(qnt->text(), 0, 10), ident, quant, this))
        {
            if (_db->giveDebt(ident, id->text(), pr->text(), cost_price->text(), quant))
            {
                // update quantity table col ...
                quant = strtol(qnt->text(), 0, 10) - quant;

                char Str[32];

                (void)snprintf(Str, sizeof(Str) - 1, "%d", quant);
                qnt->setText(Str);
                updateCell(row, QuantityCol);
                _tw->writeMessage("Товар передан диллеру");
            }
            else
                _tw->writeMessage("Ошибка передачи товара диллеру");
        }
    }
}

void GoodsTable::insert(const int row, const Goods & goods)
{
    char Str[32];

    (void)snprintf(Str, sizeof(Str) - 1, "%d", goods.id());
    setItem(row, IdCol, new QTableItem(this, QTableItem::OnTyping, Str));

    setItem(row, NameCol, new QTableItem(this, QTableItem::OnTyping, ENCODE(goods.name().c_str())));

    (void)snprintf(Str, sizeof(Str) - 1, "%.2f", goods.cprice());
    setItem(row, CPriceCol, new QTableItem(this, QTableItem::Never, Str));

    double cp = goods.cprice(), perc = cp > 0 ? (goods.price() - cp) * 100 / cp : 0;

    (void)snprintf(Str, sizeof(Str) - 1, "%.2f", perc);
    setItem(row, PercCol, new QTableItem(this, QTableItem::Never, Str));

    (void)snprintf(Str, sizeof(Str) - 1, "%.2f", goods.price());
    setItem(row, PriceCol, new PriceTableItem(this, QTableItem::OnTyping, Str, goods.was_updated()));

    (void)snprintf(Str, sizeof(Str) - 1, "%d", goods.quantity());
    setItem(row, QuantityCol, new QTableItem(this, QTableItem::Never, Str));
}

void GoodsTable::insert(const std::string & goods_id, const std::string & goods_name)
{
    int row = numRows();

    setNumRows(row + 1);

    setItem(row, IdCol, new QTableItem(this, QTableItem::OnTyping, goods_id.c_str()));
    setItem(row, NameCol, new QTableItem(this, QTableItem::OnTyping, ENCODE(goods_name.c_str())));
    setItem(row, CPriceCol, new QTableItem(this, QTableItem::Never, "0.00"));
    setItem(row, PercCol, new QTableItem(this, QTableItem::Never, "0.00"));
    setItem(row, PriceCol, new PriceTableItem(this, QTableItem::OnTyping, "0.00", true));
    setItem(row, QuantityCol, new QTableItem(this, QTableItem::Never, "0"));
}

void GoodsTable::updatePercent(int row)
{
    QTableItem *TItem = item(row, CPriceCol),
               *PItem = item(row, PriceCol);

    if (TItem && PItem)
    {
        double cp = strtod(TItem->text(), 0),
               perc = cp > 0 ? (strtod(PItem->text(), 0) - cp) * 100 / cp : 0;
        char Str[32];

        (void)snprintf(Str, sizeof(Str) - 1, "%.2f", perc);

        if ((TItem = item(row, PercCol)))
        {
            TItem->setText(Str);
            updateCell(row, PercCol);
        }
    }
}

void GoodsTable::slot_updateValue(int row, int col)
{
    QTableItem *TItem = item(row, col);

    if (TItem)
    {
        switch (col)
        {
            case IdCol:
            {
                if (isDigit(TItem->text(), TItem->text().length()))
                {
                    if (_db->updateGoodsIdent(storred_value.utf8(), TItem->text()))
                    {
                        storred_value = TItem->text();
                        _tw->writeMessage("Идентификатор товара изменен");
                        return;
                    }
                }

                TItem->setText(storred_value);
            } break;

            case NameCol:
            {
                QTableItem *IdItem = item(row, IdCol);

                if (IdItem)
                {
                    std::string ng;

                    ng.assign(TItem->text().utf8());

                    if (_db->updateGoodsName(IdItem->text(), ng))
                    {
                        storred_value = TItem->text();
                        _tw->writeMessage("Наименование товара изменено");
                        return;
                    }
                }

                TItem->setText(storred_value);
            } break;

            case PriceCol:
            {
                QTableItem *IdItem = item(row, IdCol);

                if (IdItem)
                {
                    PriceTableItem *PItem = dynamic_cast<PriceTableItem *>(TItem);

                    if (PItem)
                    {
                        if (_db->updateGoodsPrice(IdItem->text(), TItem->text()))
                        {
                            storred_value = TItem->text();
                            PItem->set_updated(true);
                            updatePercent(row);
                            _tw->writeMessage("Стоимость товара изменена");
                            return;
                        }
                    }
                }

                TItem->setText(storred_value);
            } break;
        }
    }
}

void GoodsTable::slot_buyGoods()
{
    int row = currentRow();

    QTableItem *id, *cost_price, *quant;

    if ((id = item(row, IdCol)) && (cost_price = item(row, CPriceCol)) &&
        (quant = item(row, QuantityCol)))
    {
        std::string quantity, price;

        if (BuyGoodsDialog::getData(this, quantity, price))
        {
            if (_db->buyGoods(id->text(), quantity, price))
            {
                cost_price->setText(price.c_str());
                quant->setText(quantity.c_str());

                updateCell(row, CPriceCol);
                updateCell(row, QuantityCol);

                updatePercent(row);

                _tw->writeMessage("Произведена закупка товара");
            }
        }
    }
}

void GoodsTable::slot_sellGoods()
{
    int row = currentRow();

    QTableItem *id, *quant;

    if ((id = item(row, IdCol)) && (quant = item(row, QuantityCol)))
    {
        bool ok;
        int max_q = strtol(quant->text(), 0, 10),
            goods_q = QInputDialog::getInteger(ENCODE("Продажа товара"), ENCODE("Количество:"),
                            0, 0, max_q, 1, &ok, this);
        if (ok && goods_q > 0)
        {
            if (_db->sellGoods(id->text(), goods_q, max_q))
            {
                char Str[32];

                (void)snprintf(Str, sizeof(Str) - 1, "%d", goods_q);

                quant->setText(Str);
                updateCell(row, QuantityCol);

                std::string str = "Товар продан (";

                (str += id->text()) += ')';
                _tw->writeMessage(str.c_str());
            }
        }
    }
}

