/*
 *  Copyright (c) 2004
 *      Progressive Software Engineering Company
 */

#include <qheader.h>
#include <qpopupmenu.h>
#include <qaction.h>
#include <qinputdialog.h>
#include <qmessagebox.h>
#include <qpixmap.h>

#include <string>

#include "common.h"
#include "menuwin.h"
#include "goodstable.h"
#include "debtorwidget.h"
#include "adddebtor.h"

using std::string;

MenuIndexWidget::MenuIndexWidget(TopWin *top_win, QWidget *parent) : QListView(parent),
    _db(0), _tw(top_win), goods_item(0), dolg_item(0), cur_grp(0), cur_diller(0)
{
    header()->hide();

    addColumn("");
    setResizeMode(AllColumns);
    setRootIsDecorated(true);

    setSortColumn(-1);

    // --- setup popup menus ---
    groups_popup = new QPopupMenu(this);

    QAction *action = new QAction(ENCODE("&Создать"), CTRL+SHIFT+Key_C, groups_popup);
    connect(action, SIGNAL(activated()), this, SLOT(slot_create_group()));
    action->addTo(groups_popup);

    group_popup = new QPopupMenu(this);

    action = new QAction(ENCODE("&Удалить"), CTRL+SHIFT+Key_D, group_popup);
    connect(action, SIGNAL(activated()), this, SLOT(slot_drop_group()));
    action->addTo(group_popup);

    debtors_popup = new QPopupMenu(this);

    action = new QAction(ENCODE("&Создать"), CTRL+SHIFT+Key_C, debtors_popup);
    connect(action, SIGNAL(activated()), this, SLOT(slot_create_debtor()));
    action->addTo(debtors_popup);

    debtor_popup = new QPopupMenu(this);

    action = new QAction(ENCODE("&Удалить"), CTRL+SHIFT+Key_D, debtor_popup);
    connect(action, SIGNAL(activated()), this, SLOT(slot_drop_debtor()));
    action->addTo(debtor_popup);

    // ---

    connect(this, SIGNAL(contextMenuRequested(QListViewItem *, const QPoint &, int)),
            this, SLOT(slot_contextMenuRequested(QListViewItem *, const QPoint &, int)));

    connect(this, SIGNAL(currentChanged(QListViewItem *)),
            this, SLOT(slot_dclicked(QListViewItem *)));
}

void MenuIndexWidget::slot_contextMenuRequested(QListViewItem *item, const QPoint & pos, int col)
{
    // ---

    if (dynamic_cast<GoodsGroupsItem *>(item))
    {
        groups_popup->popup(pos);
        return;
    }

    // ---

    GoodsGroupItem *grp = dynamic_cast<GoodsGroupItem *>(item);

    if (grp)
    {
        cur_grp = grp;
        group_popup->popup(pos);
        return;
    }

    // ---

    DillerItem *diller = dynamic_cast<DillerItem *>(item);

    if (diller)
    {
        cur_diller = diller;
        debtor_popup->popup(pos);
        return;
    }

    // ---

    if (dynamic_cast<DillersGroupItem *>(item))
    {
        debtors_popup->popup(pos);
        return;
    }
}

void MenuIndexWidget::slot_dclicked(QListViewItem *item)
{
    GoodsGroupItem *grp = dynamic_cast<GoodsGroupItem *>(item);

    if (grp)
    {
        _tw->setupMainWidget(new GoodsTable(_tw->centralWidget(), _db, grp->id(), _tw));
        return;
    }

    DillerItem *diller = dynamic_cast<DillerItem *>(item);

    if (diller)
    {
        DebtorWidget *dbw = new DebtorWidget(_db, diller->id(), _tw->centralWidget());

        connect(dbw, SIGNAL(debtorNameChanged(const std::string &)), this, SLOT(slot_dillerNameChanged(const std::string &)));

        _tw->setupMainWidget(dbw);
        return;
    }

    _tw->setupMainWidget();
}

void MenuIndexWidget::slot_dillerNameChanged(const string & name) throw()
{
    DillerItem *diller = dynamic_cast<DillerItem *>(currentItem());

    if (diller)
        diller->setText(0, ENCODE(name.c_str()));
}

void MenuIndexWidget::slot_create_group()
{
    bool ok;
    QString text = QInputDialog::getText(ENCODE("Создание группы товаров"),
                                         ENCODE("Название:"), QLineEdit::Normal,
                                         QString::null, &ok, this);
    if (ok && !text.isEmpty())
    {
        const int n_id = _db->createGoodsGroup(text.utf8());

        if (n_id > 0)
        {
            (void) new GoodsGroupItem(goods_item, n_id, text.utf8());
            goods_item->sortChildItems(0, true);
        }
    }
}

void MenuIndexWidget::slot_drop_group()
{
    if (cur_grp)
    {
        QString str = ENCODE("Вы действительно хотите удалить группу \"");

        str += ENCODE(cur_grp->name().c_str());
        str += "\" ?";

        if (QMessageBox::question(this, ENCODE("Удаление группы товаров"), str,
                                  ENCODE("Да"), ENCODE("Нет"), QString::null, 1, 1) == 0)
        {
            int res = _db->dropGoodsGroup(cur_grp->id());

            if (!res)
            {
                delete cur_grp;
                cur_grp = 0;

                goods_item->sortChildItems(0, true);
            }
            else if (res == 1)
                QMessageBox::information(this, ENCODE("Обнаружена зависимость ..."),
                    ENCODE("Извините, но вы не можете удалить группу, включая все входящие в неё товары, "
                           "пока они находятся на продаже у диллеров !"));
            else
                QMessageBox::information(this, ENCODE("Ошибка ..."),
                    ENCODE("Произошла ошибка СУБД, невозможно удалить группу товаров."));
        }
    }
}

void MenuIndexWidget::slot_drop_debtor()
{
    if (cur_diller)
    {
        int ret = _db->isDebtorHasGoods(cur_diller->id());

        if (!ret)
        {
            if (QMessageBox::question(this, ENCODE("Удаление диллера"),
                    ENCODE("Вы действительно хотите удалить диллера и всю связаную с ним информацию ?"),
                    ENCODE("Да"), ENCODE("Нет"), QString::null, 1, 1) == 0)
            {
                if (_db->dropDebtor(cur_diller->id()))
                {
                    delete cur_diller;
                    cur_diller = 0;
                }
            }
        }
        else if (ret == 1)
            QMessageBox::information(this, ENCODE("Информация"),
                ENCODE("Вы не можете удалить диллера, не расчетавшегося с долгами"));
        else
            QMessageBox::information(this, ENCODE("Ошибка"),
                ENCODE("Ошибка СУБД, невозможно удалить дилера."));
    }
}

void MenuIndexWidget::slot_create_debtor()
{
    AddDebtorDialog::Ptr dialog(new AddDebtorDialog(this));

    if (dialog->exec() == QDialog::Accepted)
    {
        DebtorInfo debtor;

        debtor.setName(static_cast<string>(dialog->info()->name().utf8()));
        debtor.setAddress(static_cast<string>(dialog->info()->address().utf8()));
        debtor.setHomePhone(static_cast<string>(dialog->info()->homePhone().utf8()));
        debtor.setMobilePhone(static_cast<string>(dialog->info()->mobilePhone().utf8()));

        int ret = _db->createDebtor(debtor);

        if (ret >= 0)
        {
            debtor.setId(ret);
            (void) new DillerItem(dolg_item, debtor);
            dolg_item->sortChildItems(0, true);
        }
        else if (ret == -1)
            QMessageBox::information(this, ENCODE("Повторение ..."),
                        ENCODE("Извините, но диллер с таким именем уже существует."));
        else
            QMessageBox::information(this, ENCODE("Ошибка"),
                ENCODE("Ошибка СУБД, невозможно добавить дилера."));
    }
}

void MenuIndexWidget::setDatabase(DB *db)
{
    if (db)
    {
        insertItem((goods_item = new GoodsGroupsItem(this, db)));
        insertItem((dolg_item = new DillersGroupItem(this, goods_item, db)));
        setCurrentItem(goods_item);
    }
    else
    {
        delete goods_item;
        delete dolg_item;

        goods_item = 0;
        dolg_item = 0;
    }

    _db = db;
}

MenuTabWidget::MenuTabWidget(TopWin *tw, QWidget *parent) : QTabWidget(parent), _tw(tw)
{
    setTabPosition(Top);
    insertTab((_index = new MenuIndexWidget(_tw, this)), ENCODE("Индекс"));
}

MenuDockWin::MenuDockWin(TopWin *tw, QWidget *parent, Place place) : QDockWindow(place, parent), _tw(tw)
{
    setCaption(ENCODE("Меню"));
    setOrientation(Horizontal);
    setResizeEnabled(true);
    setVerticallyStretchable(true);
    setCloseMode(Always);
    setWidget((_menuTabs = new MenuTabWidget(_tw, this)));
}

GoodsGroupsItem::GoodsGroupsItem(MenuIndexWidget *parent, DB *db) :
    QListViewItem(parent, ENCODE("Товары"))
{
    setPixmap(0, QPixmap("pic/goods_ico_24x24.gif"));

    GoodsGroup::Container cont;

    db->getGoodsGroups(cont);

    for (unsigned int i = 0; i < cont.size(); i++)
    {
        GoodsGroup::Container::const_reference ref = cont[i];

        (void) new GoodsGroupItem(this, ref.id(), ref.name().c_str());
    }

    sortChildItems(0, true);
}

GoodsGroupItem::GoodsGroupItem(GoodsGroupsItem *parent, const int c_id, const char *c_name) :
    QListViewItem(parent, ENCODE(c_name)), _id(c_id), _name(c_name)
{
    setPixmap(0, QPixmap("pic/perfume_24x24.jpg"));
}

DillersGroupItem::DillersGroupItem(QListView *parent, QListViewItem *after, DB *db) :
    QListViewItem(parent, after, ENCODE("Диллеры")), _db(db)
{
    setPixmap(0, QPixmap("pic/peoples_24x24.png"));

    Debtor::Container debtors;

    _db->getDebtorList(debtors);

    for (Debtor::Container::const_iterator i = debtors.begin(); i != debtors.end(); i++)
        (void) new DillerItem(this, *i);

    sortChildItems(0, true);    // sort debtors ...
}

DillerItem::DillerItem(QListViewItem *parent, const Debtor & dbr) :
    QListViewItem(parent, ENCODE(dbr.name().c_str())), _id(dbr.id())
{
    setPixmap(0, QPixmap("pic/personal_24x24.png"));
}

