/*
 *  Copyright (c) 2004
 *      Progressive Software Engineering Company
 */

#include <qpixmap.h>
#include <qaction.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qstatusbar.h>
#include <qstring.h>
#include <qsize.h>
#include <qlayout.h>
#include <qiconset.h>
#include <qapplication.h>
#include <qtimer.h>
#include <qdockarea.h>

#include <string>

#include "common.h"
#include "topwin.h"
#include "connwin.h"
#include "msgwin.h"
#include "menuwin.h"
#include "accountinfo.h"
#include "accountwin.h"

TopWin::TopWin() : QMainWindow(), cent_lw(0), _db()
{
    QWidget *screen = QApplication::desktop()->screen(0);

    resize(QSize(int(screen->width() * 0.7), int(screen->height() * 0.7)));

    setCaption(ENCODE("Panther - smart shopping"));
    setIcon(QPixmap("pic/logo_32x32.png"));
    setCentralWidget(new QWidget(this));

    cent_layout = new QGridLayout(centralWidget(), 1, 1, 0, 0);

    // setup messages frame
    messages = new MsgDockWin(this);

    connect(messages, SIGNAL(visibilityChanged(bool)),
            this, SLOT(slot_messagesFrameVisibilityChanged(bool)));

    addDockWindow(messages, DockBottom);
    messages->Tabs()->DBMessages()->setPicture("pic/mysql.jpg");
    // ---

    // setup menu frame
    menu = new MenuDockWin(this, this);

    connect(menu, SIGNAL(visibilityChanged(bool)), this, SLOT(slot_menuFrameVisibilityChanged(bool)));
    addDockWindow(menu, DockLeft);
    // ---

    // setup file menu
    QPopupMenu *filePopup = new QPopupMenu(this);

    // setup account menu
    accMenu = new QPopupMenu(filePopup);

    QAction *action = new QAction(ENCODE("Закупки"), 0, this);
    connect(action, SIGNAL(activated()), this, SLOT(slot_buyAccount()));
    action->addTo(accMenu);

    action = new QAction(ENCODE("Продажи"), 0, this);
    connect(action, SIGNAL(activated()), this, SLOT(slot_sellAccount()));
    action->addTo(accMenu);

    filePopup->insertItem(ENCODE("Отчеты"), accMenu);
    // ---

    filePopup->insertSeparator();

    QAction *quit_action = new QAction(QIconSet(QPixmap("pic/exit.png")), ENCODE("&Выйти"), CTRL+Key_Q, this);
    connect(quit_action, SIGNAL(activated()), this, SLOT(slot_quit()));
    quit_action->addTo(filePopup);

    menuBar()->insertItem(ENCODE("&Файл"), filePopup);
    // ---

    // set db menu
    QPopupMenu *dbPopup = new QPopupMenu(this);

    conn_action = new QAction(this);
    connect(conn_action, SIGNAL(activated()), this, SLOT(slot_conn()));

    conn_action->setIconSet(QIconSet(QPixmap("pic/mysql.jpg")));
    conn_action->addTo(dbPopup);

    menuBar()->insertItem(ENCODE("&СУБД"), dbPopup);
    // ---

    // setup view menu
    viewMenu = new QPopupMenu(this);

    view_menu_id = viewMenu->insertItem(ENCODE("Меню"), this,
                                            SLOT(slot_showMenuWin()), CTRL+SHIFT+Key_M);
    viewMenu->setItemChecked(view_menu_id, true);

    view_messages_id = viewMenu->insertItem(ENCODE("Сообщения"), this,
                                            SLOT(slot_showMessagesWin()), CTRL+SHIFT+Key_O);
    viewMenu->setItemChecked(view_messages_id, true);

    menuBar()->insertItem(ENCODE("&Окно"), viewMenu);
    // ---

    // --- help menu ---
    QPopupMenu *helpPopup = new QPopupMenu(this);

    action = new QAction(ENCODE("&О программе"), CTRL+Key_A, this);
    connect(action, SIGNAL(activated()), this, SLOT(slot_about()));
    action->addTo(helpPopup);

    menuBar()->insertItem(ENCODE("&Помощь"), helpPopup);
    // ---

    connWin = new ConnWin(this, true);

    setConnectionStatus(false, false);
    statusBar()->message(ENCODE("Готово"), 2000);

    clearWState(WState_Polished);

    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(timerDone()));
    timer->start(0, true);
}

void TopWin::timerDone()
{
    delete timer;
    timer = 0;

    slot_conn();
}

void TopWin::setupMainWidget(QWidget *wg)
{
    delete cent_lw;
    cent_lw = wg;

    if (cent_lw)
    {
        cent_layout->addWidget(cent_lw, 0, 0);
        cent_lw->show();
    }
}

void TopWin::setConnectionStatus(bool connected, bool rep)
{
    QString msg;

    if (connected)
    {
        conn_action->setText(ENCODE("Прервать соединение"));
        conn_action->setMenuText(ENCODE("&Прервать соединение"));
        conn_action->setAccel(CTRL+SHIFT+Key_D);
        menu->Tabs()->Index()->setDatabase(&_db);
        statusBar()->message((msg = ENCODE("Соединение с СУБД установлено")), 2000);
        accMenu->setEnabled(true);
    }
    else
    {
        conn_action->setText(ENCODE("Устаночить соединение"));
        conn_action->setMenuText(ENCODE("&Установить соединение"));
        conn_action->setAccel(CTRL+SHIFT+Key_C);
        menu->Tabs()->Index()->setDatabase(0);
        statusBar()->message((msg = ENCODE("Соединение с СУБД разоравно")), 2000);
        accMenu->setEnabled(false);
    }

    if (rep)
        messages->Tabs()->DBMessages()->write(msg);
}

void TopWin::setFrameShown(int menu_id, QDockWindow *frame)
{
    if (viewMenu->isItemChecked(menu_id))
    {
        frame->hide();
        viewMenu->setItemChecked(menu_id, false);
    }
    else
    {
        frame->show();
        viewMenu->setItemChecked(menu_id, true);
    }
}

void TopWin::slot_quit()
{
    close();
}

void TopWin::slot_conn()
{
    if (_db.opened())
    {
        if (QMessageBox::question(this, ENCODE("Закрытие соединения с СУБД"),
                                  ENCODE("Вы действительно хотите закрыть соединение с СУБД ?"),
                ENCODE("Да"), ENCODE("Нет"), QString::null, 1, 1) == 0)
        {
            _db.close();
            setConnectionStatus(false);
        }
    }
    else if (connWin->exec() == QDialog::Accepted)
    {
        std::string host, un, up, db_name, unix_socket;
        unsigned int port = connWin->port();

        connWin->host(host);
        connWin->login(un);
        connWin->password(up);
        connWin->database(db_name);
        connWin->socket(unix_socket);

        _db.open(host, un, up, db_name, port, unix_socket) ?
            setConnectionStatus(true) : writeSQLError(_db.error_code(), _db.error_str());
    }
}

void TopWin::slot_showMenuWin()
{
    setFrameShown(view_menu_id, menu);
}

void TopWin::slot_showMessagesWin()
{
    setFrameShown(view_messages_id, messages);
}

void TopWin::slot_menuFrameVisibilityChanged(bool visible)
{
    viewMenu->setItemChecked(view_menu_id, visible);
}

void TopWin::slot_messagesFrameVisibilityChanged(bool visible)
{
    viewMenu->setItemChecked(view_messages_id, visible);
}

void TopWin::writeSQLError(const int errno, const char *error)
{
    char int_str[32];

    snprintf(int_str, sizeof(int_str), "%d", errno);

    std::string StrErr = "Ошибка ";

    StrErr.append(int_str).append(": ").append(error);
    messages->Tabs()->DBMessages()->write(StrErr);
    QMessageBox::critical(this, ENCODE("Ошибка СУБД"), ENCODE(StrErr.c_str()), "OK");
}

void TopWin::writeMessage(const char *message)
{
    QString msg = ENCODE(message);

    messages->Tabs()->LogMessages()->write(msg);
    statusBar()->message(msg, 2000);
}

#define DNF_MSG() \
    QMessageBox::information(this, ENCODE("Не найдено"), \
                             ENCODE("Извините, по Вашему запросу не найдено данных"))

void TopWin::slot_sellAccount()
{
    int goods_id;
    std::string from_date, to_date, from_time, to_time;

    if (AccountInfoDialog::getInfo(goods_id, from_date, to_date, from_time, to_time, this))
    {
        sell_account_info_t sai;

        _db.getSellInfo(sai, goods_id, from_date, to_date, from_time, to_time);

        if (sai.size())
        {
            SellAccountWindow *acw = new SellAccountWindow(sai, this);

            std::string caption = "Отчет о продажах с ";

            caption.append(from_date).append(" / ").append(from_time).append(" по ").append(to_date).append(
                        " / ").append(to_time);

            acw->setCaption(ENCODE(caption.c_str()));
            writeMessage(caption.c_str());
        }
        else
            DNF_MSG();
    }
}

void TopWin::slot_buyAccount()
{
    int goods_id;
    std::string from_date, to_date, from_time, to_time;

    if (AccountInfoDialog::getInfo(goods_id, from_date, to_date, from_time, to_time, this))
    {
        buy_account_info_t bai;

        _db.getBuyInfo(bai, goods_id, from_date, to_date, from_time, to_time);

        if (bai.size())
        {
            BuyAccountWindow *acw = new BuyAccountWindow(bai, this);

            std::string caption = "Отчет о закупках с ";

            caption.append(from_date).append(" / ").append(from_time).append(" по ").append(
                to_date).append(" / ").append(to_time);

            acw->setCaption(ENCODE(caption.c_str()));
            writeMessage(caption.c_str());
        }
        else
            DNF_MSG();
    }
}

#undef DNF_MSG

void TopWin::slot_about()
{
    QMessageBox::information(this, ENCODE("О программе"), \
        ENCODE("Panther CShop - smart shopping.\n"
               "Автор: Владислав Лазаренко <snail@ilyichevsk.net>"));
}

void TopWin::closeEvent(QCloseEvent *e)
{
    if (_db.opened())
    {
        if (QMessageBox::question(this, ENCODE("Завершение работы ..."),
                                  ENCODE("Вы действительно хотите завершить работу программы ?"),
                                  ENCODE("Да"), ENCODE("Нет"), QString::null, 1, 1) == 0)
        {
            _db.close();
            setConnectionStatus(false);
            e->accept();
        }
        else
            e->ignore();
    }
    else
        e->accept();
}

