#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <cgi/utils.h>
#include <osdep.h>

#ifndef INTERNALS_DEFINED
#define INTERNALS_DEFINED

struct FP_INTERNALS
{
	long buffer_ptr;
	long buffer_read;
	long last_compared;
	int  just_after_boundary;
	int  continued;
	int  read_header;
	int  read_data;
	int  content_type;
	int  request_method;
	int  data_parsed;
	int  signature;
	char *query_string;
	char *saved_query_string;
};

#else

#error	"Wrong include order! Can't proceed!"

#endif

#include <cgi/form_parser.h>

#define CT_UNDEFINED	0
#define	CT_MULTIPART	1
#define CT_URLENCODED	2

#define CONTENT_DISPOSITION	"content-disposition:"
#define CONTENT_TYPE		"content-type:"

// Prototypes go here...

inline void set_form_parser_error(struct FORM_PARSER *parser, int error);
inline void set_form_parser_corrupted(struct FORM_PARSER *parser, char *data);
void print_form_parser_state(struct FORM_PARSER *parser);
int inline is_oversized(struct FORM_PARSER *parser, int size);
int parse_header(struct FORM_PARSER *parser, char *data);
int determine_content_type(struct FORM_PARSER *parser);
int determine_request_method(struct FORM_PARSER *parser);
int inline form_parser_initialized(struct FORM_PARSER *parser);

extern char** environ;

void clear_form_parser(struct FORM_PARSER *parser)
{
	if(!parser)
		return;
	if(parser->field_name)
	{
		free(parser->field_name);
		parser->field_name = NULL;
	}
	if(parser->field_data)
	{
		free(parser->field_data);
		parser->field_data = NULL;
	}
	if(parser->file_name)
	{
		free(parser->file_name);
		parser->file_name = NULL;
	}
	if(parser->content_type)
	{
		free(parser->content_type);
		parser->content_type = NULL;
	}
	if(parser->corrupted)
	{
		free(parser->corrupted);
		parser->corrupted = NULL;
	}
	parser->field_length = 0;
	parser->is_file = 0;
//	parser->internal.read_header = 1;
//	parser->internal.read_data = 0;
	parser->parser_error = E_OK;
}

void init_form_parser(struct FORM_PARSER *parser, int method)
{
	struct FORM_PARSER_CONFIG saved_config;
	char **envp;
	FILE *in;

	if(!parser)
		return;

	envp = parser->envp;
	in = parser->in;

	memcpy(&saved_config, &(parser->config), sizeof(struct FORM_PARSER_CONFIG));
	memset(parser, 0, sizeof(struct FORM_PARSER));
	memcpy(&(parser->config), &saved_config, sizeof(struct FORM_PARSER_CONFIG));

	if(envp)
		parser->envp = envp;
	else
		parser->envp = environ;

	if(in)
		parser->in = in;
	else
		parser->in = stdin;

	parser->internal.buffer_ptr = parser->config.buffer_size;

	if(method == POST_DATA)
	{
		determine_content_type(parser);
		determine_request_method(parser);
	}

	parser->internal.signature = (int)(parser->config.buffer) ^ 
					parser->config.buffer_size ^ (int)(parser->config.boundary);
}

void destroy_form_parser(struct FORM_PARSER *parser)
{
	if(!parser)
		return;
	clear_form_parser(parser);
	if(parser->config.auto_boundary)
		if(parser->config.boundary)
		{
			free(parser->config.boundary);
		}
	free(parser);
}

void reset_form_parser(struct FORM_PARSER *parser)
{
	clear_form_parser(parser);
	init_form_parser(parser, POST_DATA);
}

void set_form_parser_error(struct FORM_PARSER *parser, int error)
{
	parser->parser_error = error;
}

void set_form_parser_corrupted(struct FORM_PARSER *parser, char *data)
{
	if(!parser->parser_error)
		set_form_parser_error(parser, E_CORRUPT);
	parser->corrupted = (char *)malloc(strlen(data) + 1);
	if(parser->corrupted)
		strcpy(parser->corrupted, data);
	else
		set_form_parser_error(parser, E_NOMEM);
}

int inline form_parser_initialized(struct FORM_PARSER *parser)
{
	if(!parser)
		return 1;
	if(parser->internal.signature == ((int)(parser->config.buffer) ^ parser->config.buffer_size ^ 
			(int)(parser->config.boundary)))
		return 1;
	return 0;
}

int parse_header(struct FORM_PARSER *parser, char *data)
{
	char *word;
	char *header = data;

	while(*header)
	{
		skip_blanks(&header);
		if(skip_word(&header, CONTENT_DISPOSITION))
		{
			skip_blanks(&header);
			if(!skip_word(&header, "form-data"))
			{
				set_form_parser_corrupted(parser, header);
				return 0;
			}
			while(*header != '\r')
			{
				skip_chars(&header, " \t;");
				if(skip_word(&header, "name="))
				{
					if(!get_word(&word, &header))
					{
						set_form_parser_error(parser, E_NOMEM);
						set_form_parser_corrupted(parser, header);
						return 0;
					}
					parser->field_name = word;
					word = NULL;
				}
				else if(skip_word(&header, "filename="))
				{
					if(!get_word(&word, &header))
					{
						set_form_parser_error(parser, E_NOMEM);
						set_form_parser_corrupted(parser, header);
						return 0;
					}
					parser->is_file = 1;
					parser->file_name = word;
					word = NULL;
				}
				else
				{
					set_form_parser_corrupted(parser, header);
					return 0;
				}
			}

			if(!skip_word(&header, "\r\n"))
			{
				set_form_parser_corrupted(parser, header);
				return 0;
			}
		}
		else if(skip_word(&header, CONTENT_TYPE))
		{
			skip_blanks(&header);
			if(!get_word(&word, &header))
			{
				set_form_parser_corrupted(parser, header);
				return 0;
			}
			
			parser->content_type = word;
			word = NULL;
			
			skip_to(&header, "\r");
			if(!skip_word(&header, "\r\n"))
			{
				set_form_parser_corrupted(parser, header);
				return 0;
			}
		}
		else
		{
			set_form_parser_corrupted(parser, header);
			return 0;
		}
	}
	
	return 1;
}

struct FORM_PARSER *create_form_parser()
{
	struct FORM_PARSER *parser = NULL;
	parser = (struct FORM_PARSER *)malloc(sizeof(struct FORM_PARSER));
	if(!parser)
		return NULL;
	bzero(parser, sizeof(struct FORM_PARSER));
	parser->internal.data_parsed = 0;
	parser->internal.signature = 0;
	parser->config.max_size = 0;
	parser->config.always_read_header = 1;
	parser->config.auto_boundary = 0;
	parser->config.boundary = NULL;
	return parser;
}

void print_form_parser_state(struct FORM_PARSER *parser)
{
	if(!parser)
		return;
	printf("Internals:	\n");
	printf("	buffer_ptr:	%ld\n", parser->internal.buffer_ptr);
	printf("	buffer_read:	%ld\n", parser->internal.buffer_read);
	printf("	last_compared:	%ld\n\n", parser->internal.last_compared);
}

int determine_content_type(struct FORM_PARSER *parser)
{
	char *ctenv, *ctenvptr, *boundary;
#define	STRING_MULTIPART	"multipart/form-data"

	if(!parser)
		return 0;

	ctenv = fast_get_env(parser->envp, "CONTENT_TYPE");

	if(!ctenv)
	{
		parser->internal.content_type = CT_URLENCODED;
	}
	else if(strncasecmp(ctenv, STRING_MULTIPART, strlen(STRING_MULTIPART)) == 0)
	{
		parser->internal.content_type = CT_MULTIPART;
		ctenvptr = strcasestr(ctenv, "boundary=");

		if(!ctenvptr)
		{
			set_form_parser_error(parser, E_CORRUPT);
			return 0;
		}

		ctenvptr += 9;
		if(parser->config.auto_boundary)
		{
			if(!get_word(&boundary, &ctenvptr))
			{
				return 0;
			}

			if(parser->config.boundary)
			{
				free(parser->config.boundary);
				parser->config.boundary = NULL;
			}
			
			if(!asprintf(&(parser->config.boundary), "--%s", boundary) < 0)
			{
				set_form_parser_error(parser, E_NOMEM);
				return 0;
			}
			
			free(boundary);
		}
	}
	else
	{
		parser->internal.content_type = CT_URLENCODED;
	}
	return 1; // success
}

int determine_request_method(struct FORM_PARSER *parser)
{
	char *rmenv;
#define	STRING_GET	"GET"
#define STRING_POST	"POST"

	if(!parser)
		return 0;
	rmenv = fast_get_env(parser->envp, "REQUEST_METHOD");

	// GET data is present always, not depending on request method
	parser->internal.request_method = GET_DATA;

	if(rmenv != NULL)
		if(strcasecmp(rmenv, STRING_POST) == 0)
		{
			// POST data is present too...
			parser->internal.request_method |= POST_DATA;
		}

	return 1; // success
}

int inline is_oversized(struct FORM_PARSER *parser, int size)
{
	if(parser->config.max_size == 0)
		return 0;
	if((parser->config.max_size >= size) || 
		(parser->internal.read_header && parser->config.always_read_header))
			return 0;
	return 1;
}

int parse_form_urlencoded(struct FORM_PARSER *parser)
{
	char *data = NULL, *realloced;
	int curlen = 0, curpos = 0;

	if(!parser)
		return 0;
	if(parser->is_last)
		return 0;

	if(!form_parser_initialized(parser))
	{
		set_form_parser_error(parser, E_NOTINITIALIZED);
		return 0;
	}

	parser->internal.read_header = 1;
	parser->internal.read_data = 0;

	if(parser->internal.buffer_ptr != parser->config.buffer_size)
	{
		// As we want to resume main loop as like we didn't returned from here, we need
		// such a horrible hack to avoid wasting bytes of data...
		// I know, that's really wrong! But it is really needed here, sorry!
		goto JUMP_INSIDE_PARSE_FORM_LOOP;
	}

	while((parser->internal.buffer_read = fread(parser->config.buffer, 1, parser->config.buffer_size, parser->in)))
	{
		realloced = (char *)realloc(data, curpos + parser->internal.buffer_read);
		if(!realloced)
		{
			free(data);
			set_form_parser_error(parser, E_NOMEM);
			return 0;
		}
		data = realloced;
		curlen = curpos + parser->internal.buffer_read;

		for(parser->internal.buffer_ptr = 0; parser->internal.buffer_ptr < parser->internal.buffer_read; parser->internal.buffer_ptr ++)
		{
			if(parser->config.buffer[parser->internal.buffer_ptr] == '=')
			{
				if(parser->internal.read_data)
				{
					parser->config.buffer[parser->internal.buffer_read] = 0;
					set_form_parser_corrupted(parser, parser->config.buffer);
					return 0;
				}
				
				data[curpos] = 0;
				realloced = url_decode(data, &curpos);
				free(data);
				data = realloced;
				realloced = realloc(data, curpos + 1);
				if(!realloced)
				{
					set_form_parser_error(parser, E_NOMEM);
					return 0;
				}
				parser->field_name = realloced;

				curpos = 0;
				curlen = parser->internal.buffer_read - parser->internal.buffer_ptr;
				data = malloc(curlen);
				
				if(!data)
				{
					set_form_parser_error(parser, E_NOMEM);
					return 0;
				}

				parser->internal.read_header = 0;
				parser->internal.read_data = 1;
				continue;
			}
			else if(parser->config.buffer[parser->internal.buffer_ptr] == '&')
			{
				if(parser->internal.read_header)
				{
					parser->config.buffer[parser->internal.buffer_read] = 0;
					set_form_parser_corrupted(parser, parser->config.buffer);
					return 0;
				}
				
				data[curpos] = 0;
				realloced = url_decode(data, &curpos);
				free(data);
				data = realloced;
				realloced = realloc(data, curpos + 1);
				if(!realloced)
				{
					set_form_parser_error(parser, E_NOMEM);
					return 0;
				}
				parser->field_data = realloced;
				parser->field_length = curpos;//strlen(parser->field_data);

				return 1;

JUMP_INSIDE_PARSE_FORM_LOOP:

				curpos = 0;
				curlen = parser->internal.buffer_read - parser->internal.buffer_ptr;
				data = malloc(curlen);
				
				if(!data)
				{
					set_form_parser_error(parser, E_NOMEM);
					return 0;
				}
				
				parser->internal.read_header = 1;
				parser->internal.read_data = 0;
				continue;
			}

			data[curpos] = parser->config.buffer[parser->internal.buffer_ptr];
			curpos ++;
		}
	}
	if(parser->internal.read_data)
	{
		data[curpos] = 0;
		realloced = url_decode(data, &curpos);
		free(data);
		data = realloced;
		realloced = realloc(data, curpos + 1);
		if(!realloced)
		{
			set_form_parser_error(parser, E_NOMEM);
			return 0;
		}
		parser->field_data = realloced;
		parser->field_length = curpos;//strlen(parser->field_data);
		parser->is_last = 1;

		return 1;
	}
	else
	{
		set_form_parser_corrupted(parser, "(need more data)");
		return 0;
	}
}

int parse_form_get(struct FORM_PARSER *parser)
{
	char *data = NULL, *realloced, *query_string;
	int curlen = 0, curpos = 0;

	if(!parser)
		return 0;

	clear_form_parser(parser);


	if(!parser->internal.query_string)
	{
		query_string = fast_get_env(parser->envp, "QUERY_STRING");

		if(!query_string)
			return 0;

		if(!(parser->internal.query_string = query_string))// strdup(query_string)))
		{
			set_form_parser_error(parser, E_NOMEM);
			return 0;
		}

		parser->internal.saved_query_string = parser->internal.query_string;

		if(strlen(parser->internal.query_string) == 0)
		{
			return 0;
		}
	}

	if(parser->is_last || !parser->internal.query_string)
		return 0;

	if(!form_parser_initialized(parser))
	{
		set_form_parser_error(parser, E_NOTINITIALIZED);
		return 0;
	}

	parser->internal.read_header = 1;
	parser->internal.read_data = 0;

	if(parser->internal.buffer_ptr != parser->config.buffer_size)
	{
		// As we want to resume main loop as like we didn't returned from here, we need
		// such a horrible hack to avoid wasting bytes of data...
		// I know, that's really wrong! But it is really needed here, sorry!
		goto JUMP_INSIDE_PARSE_FORM_GET_LOOP;
	}

	while((parser->internal.buffer_read = MIN(strlen(parser->internal.query_string), 
												parser->config.buffer_size)))
	{
		parser->internal.buffer_read = MIN(strlen(parser->internal.query_string), parser->config.buffer_size);
		memcpy(parser->config.buffer, parser->internal.query_string, parser->internal.buffer_read);
		parser->internal.query_string += parser->internal.buffer_read;

		realloced = (char *)realloc(data, curpos + parser->internal.buffer_read);
		if(!realloced)
		{
			free(data);
			set_form_parser_error(parser, E_NOMEM);
			return 0;
		}
		data = realloced;
		curlen = curpos + parser->internal.buffer_read;

		for(parser->internal.buffer_ptr = 0; parser->internal.buffer_ptr < parser->internal.buffer_read; parser->internal.buffer_ptr ++)
		{
			if(parser->config.buffer[parser->internal.buffer_ptr] == '=')
			{
				if(parser->internal.read_data)
				{
					parser->config.buffer[parser->internal.buffer_read] = 0;
					set_form_parser_corrupted(parser, parser->config.buffer);
					return 0;
				}
				
				data[curpos] = 0;
				realloced = url_decode(data, &curpos);
				free(data);
				data = realloced;
				realloced = realloc(data, curpos + 1);
				if(!realloced)
				{
					set_form_parser_error(parser, E_NOMEM);
					return 0;
				}
				parser->field_name = realloced;

				curpos = 0;
				curlen = parser->internal.buffer_read - parser->internal.buffer_ptr;
				data = malloc(curlen);
				
				if(!data)
				{
					set_form_parser_error(parser, E_NOMEM);
					return 0;
				}

				parser->internal.read_header = 0;
				parser->internal.read_data = 1;
				continue;
			}
			else if(parser->config.buffer[parser->internal.buffer_ptr] == '&')
			{
				if(parser->internal.read_header)
				{
					parser->config.buffer[parser->internal.buffer_read] = 0;
					set_form_parser_corrupted(parser, parser->config.buffer);
					return 0;
				}
				
				data[curpos] = 0;
				realloced = url_decode(data, &curpos);
				free(data);
				data = realloced;
				realloced = realloc(data, curpos + 1);
				if(!realloced)
				{
					set_form_parser_error(parser, E_NOMEM);
					return 0;
				}
				parser->field_data = realloced;
				parser->field_length = curpos;//strlen(parser->field_data);

				return 1;

JUMP_INSIDE_PARSE_FORM_GET_LOOP:

				curpos = 0;
				curlen = parser->internal.buffer_read - parser->internal.buffer_ptr;
				data = malloc(curlen);
				
				if(!data)
				{
					set_form_parser_error(parser, E_NOMEM);
					return 0;
				}
				
				parser->internal.read_header = 1;
				parser->internal.read_data = 0;
				continue;
			}

			data[curpos] = parser->config.buffer[parser->internal.buffer_ptr];
			curpos ++;
		}
	}

	if(parser->internal.read_data)
	{
		data[curpos] = 0;
		realloced = url_decode(data, &curpos);
		free(data);
		data = realloced;
		realloced = realloc(data, curpos + 1);
		if(!realloced)
		{
			set_form_parser_error(parser, E_NOMEM);
			return 0;
		}
		parser->field_data = realloced;
		parser->field_length = curpos;//strlen(parser->field_data);
		parser->is_last = 1;

		return 1;
	}
	else
	{
//		set_form_parser_corrupted(parser, "(need more data)");
		return 0;
	}
}

int parse_form_multipart(struct FORM_PARSER *parser)
{
	char *crlf = "\r\n";
	char *data = NULL, *realloced;
	int curpos = 0, curlen = 0;
	int cmpsize, cr_was_before = 0, crlf_was_before = 0;

	if(!parser)
		return 0;

	clear_form_parser(parser);

	if(parser->is_last)
		return 0;

	if(!form_parser_initialized(parser))
	{
		set_form_parser_error(parser, E_NOTINITIALIZED);
		return 0;
	}

	if(parser->internal.buffer_ptr != parser->config.buffer_size)
	{
		// As we want to resume main loop as like we didn't returned from here, we need
		// such a horrible hack to avoid wasting bytes of data...
		// I know, that's really wrong! But it is really needed here, sorry!
		goto JUMP_INSIDE_PARSE_FORM_MULTIPART_LOOP;
	}
	
	while((parser->internal.buffer_read = fread(parser->config.buffer, 1, parser->config.buffer_size, parser->in)))
	{
		if(!is_oversized(parser, curpos))
		{
			realloced = (char *)realloc(data, curpos + parser->internal.buffer_read);
			if(!realloced)
			{
				free(data);
				set_form_parser_error(parser, E_NOMEM);
				return 0;
			}
			data = realloced;
			curlen = curpos + parser->internal.buffer_read;
		}

		for(parser->internal.buffer_ptr = 0; parser->internal.buffer_ptr < parser->internal.buffer_read; parser->internal.buffer_ptr ++)
		{
			if(parser->internal.just_after_boundary)
			{
				// We just have read boundary. We expect to find \r\n or -- here.
				// This would be simple, if we could be assured that we don't have a part
				// (or whole) of this in next chunk of data that we have not actually
				// read yet...
				int just_after_boundary = parser->internal.just_after_boundary;
				if(!strncmp(parser->config.buffer + parser->internal.buffer_ptr, 
							&crlf[2 - just_after_boundary], 
							MIN(parser->internal.buffer_read - parser->internal.buffer_ptr, 
								just_after_boundary)))
				{
					// This is rather \r\n. We couldn't be assured, if we haven't compared all 2
					// symbols. But if it so, we could try to compare the rest later...
					parser->internal.just_after_boundary -= MIN(parser->internal.buffer_read - parser->internal.buffer_ptr, just_after_boundary);
					parser->internal.buffer_ptr += MIN(parser->internal.buffer_read - parser->internal.buffer_ptr, just_after_boundary);

					if(parser->internal.just_after_boundary)
						// We've found \r. Let's see if the next symbol is \n
						parser->internal.continued = 1;
					else
						// Looks like that was really \r\n - that's really good, cause we can start
						// to read new field then!
						parser->internal.continued = 0;
				}
				else if(!strncmp(parser->config.buffer + parser->internal.buffer_ptr, "--", MIN(parser->internal.buffer_read - parser->internal.buffer_ptr, just_after_boundary)))
				{
					// It seems like this is -- - the end of input... Let's see...
					parser->internal.just_after_boundary -= MIN(parser->internal.buffer_read - parser->internal.buffer_ptr, just_after_boundary);
					parser->internal.buffer_ptr += MIN(parser->internal.buffer_read - parser->internal.buffer_ptr, just_after_boundary);

					if(parser->internal.continued)
					{
						// Hmm... the first was \r, and the next is -... It's really weird...
						// Ok, let's think someone want to hack us, and make them down!..
						parser->config.buffer[parser->internal.buffer_read] = 0;
						set_form_parser_corrupted(parser, parser->config.buffer);
						return 0;
					}

					if(!parser->internal.just_after_boundary)
					{
						// We have compared all 2 symbols and it really was a '--',
						// So, the input is over, let's tell them, out there...
						parser->is_last = 1;
						return 0;
					}
				}
				else
				{
					// There should be \r\n or --, otherwise something is wrong!
					parser->config.buffer[parser->internal.buffer_read - 1] = 0;
					set_form_parser_corrupted(parser, parser->config.buffer);
					return 0;
				}
			}

			// Let's see whether there is boundary string somewhere...
			cmpsize = MIN(strlen(parser->config.boundary) - parser->internal.last_compared, parser->internal.buffer_read - parser->internal.buffer_ptr);

			if(strncmp((parser->config.buffer) + (parser->internal.buffer_ptr), (parser->config.boundary) + (parser->internal.last_compared), 
				cmpsize) == 0)
			{
				// Here is something that maybe a boundary! If it matches at first symbols, 
				// but there is end of buffer, the next time we should test the rest!
				if(cmpsize < (strlen(parser->config.boundary) - parser->internal.last_compared))
				{
					parser->internal.last_compared += parser->internal.buffer_read - parser->internal.buffer_ptr;
					parser->internal.buffer_ptr = parser->internal.buffer_read;
					continue;	// read next bytes of something that looks like a boundary!...
				}

				// Yes, there's a boundary!
				parser->internal.just_after_boundary = 2;
				parser->internal.continued = 0;
				parser->internal.buffer_ptr += cmpsize - 1;
				parser->internal.last_compared = 0;

				if(parser->internal.read_data)
				{
					// If we read field data, let's save it to parser and tell
					// them out there...
					if(is_oversized(parser, curpos - 2))
					{
						if(!parser->parser_error)
							set_form_parser_error(parser, E_DATATOOBIG);
						parser->field_length = curpos - 2;
					}
					else
					{

						if((curpos < 2) || (strncmp(&data[curpos - 2], "\r\n", 2)))
						{
							// We should have \r\n in the end of field data,
							// as we need something to strip out ;)
							// If we don't, we should throw away such data
							data[curpos] = 0;
							set_form_parser_corrupted(parser, data);
							free(data);
							return 0;
						}

						curpos -= 2;

						// Hoorah! Let's waste \r\n in the end of data!
						data[curpos] = 0;

						// Free unused memory, if there is any (so, there is!)...
						realloced = (char *)realloc(data, curpos + 1);
						if(!realloced)
						{
							free(data);
							set_form_parser_error(parser, E_NOMEM);
							return 0;
						}
						data = realloced;
					
						parser->field_data = data;
						parser->field_length = curpos;
					}
					
					// Ok, we've read the whole field, let's tell about that...
					return 1;

					// Whoops! We're here again, like we've never left ;)
JUMP_INSIDE_PARSE_FORM_MULTIPART_LOOP:

					// Free all memory that was used by previous field data and
					// let's start new field!
					curlen = parser->internal.buffer_read - parser->internal.buffer_ptr;
					data = (char*)malloc(curlen);
					if(!data)
					{
						set_form_parser_error(parser, E_NOMEM);
						return 0;
					}
					curpos = 0;
				}
				else if(parser->internal.read_header)
				{
					// If we read field header, we shouldn't be here anyhow, so the
					// data is corrupted!
					parser->config.buffer[parser->internal.buffer_read - 1] = 0;
					set_form_parser_corrupted(parser, parser->config.buffer + parser->internal.buffer_ptr);
					return 0;
				}
				// Anyhow, now we should start to read the header of new field!
				parser->internal.read_header = 1;
				parser->internal.read_data = 0;
			}
			else if(parser->internal.read_header)
			{
				// Ok, we currently read header's data
				if(parser->internal.last_compared && !is_oversized(parser, curpos))
				{
					// The boundary we wanted to find actually was not a boundary.
					// Anyhow, the data is wasted. But we know that all that data is
					// the same like first (n) symbols of boundary, so we can restore it...
					realloced = realloc(data, curlen + parser->internal.last_compared);
					if(!realloced)
					{
						free(data);
						set_form_parser_error(parser, E_NOMEM);
						return 0;
					}
					data = realloced;

					strncpy(data + curpos, parser->config.boundary, parser->internal.last_compared);
					curlen += parser->internal.last_compared;
					curpos += parser->internal.last_compared;
					parser->internal.last_compared = 0;
				}
			
				if(parser->config.buffer[parser->internal.buffer_ptr] == '\n')
				{
					if(cr_was_before)
					{
						if(crlf_was_before)
						{
							// If there is \r\n\r\n in the header, that means the header is over!
							// Let's parse it and start to read data!
							data[curpos-1] = 0;

							if(!is_oversized(parser, curpos))
							{
								realloced = (char *)realloc(data, curpos);
								if(!realloced)
								{
									free(data);
									set_form_parser_error(parser, E_NOMEM);
									return 0;
								}
								data = realloced;
	
								if(!parse_header(parser, data))
								{
									// Hmm... we don't know what actually, but something was wrong in
									// the header... let's trust the Specialized Header Parser that told us
									// about that...

									return 0;
								}
							}
							else
							{
								// all we do then is just skip to the next header...
								set_form_parser_error(parser, E_HEADERTOOBIG);
							}

							// Seems that everything is OK by now...
							free(data);
							data = NULL; // dispose and nullify data, then re-allocate new chunk...
							curlen = parser->internal.buffer_read - parser->internal.buffer_ptr;
							data = malloc(curlen);
							if(!data)
							{
								set_form_parser_error(parser, E_NOMEM);
								return 0;
							}
							curpos = 0;
							// Ok, now we should read the data, not a header!
							parser->internal.read_data = 1;
							parser->internal.read_header = 0;
							continue;
						}
						else
						{
							crlf_was_before = 1;
						}
					}
					else
					{
						crlf_was_before = 0;
					}
				}
				else if(parser->config.buffer[parser->internal.buffer_ptr] == '\r')
				{
					cr_was_before = 1;
				}
				else
				{
					cr_was_before = 0;
					crlf_was_before = 0;
				}

				if(is_oversized(parser, curpos))
				{
					if(parser->internal.last_compared)
					{
						curpos += parser->internal.last_compared;
						parser->internal.last_compared = 0;
					}
					curpos ++;
					continue;
				}

				// Simply add the next symbol to the header string...
				data[curpos] = parser->config.buffer[parser->internal.buffer_ptr];
				curpos ++;
			}
			else if(parser->internal.read_data)
			{
				if(is_oversized(parser, curpos))
				{
					if(parser->internal.last_compared)
					{
						curpos += parser->internal.last_compared;
						parser->internal.last_compared = 0;
					}
					curpos ++;
					continue;
				}
				if(parser->internal.last_compared)
				{
					// The boundary we wanted to find actually was not a boundary.
					// Anyhow, the data is wasted. But we know that all that data is
					// the same like first (n) symbols of boundary, so we can restore it...
					realloced = realloc(data, curlen + parser->internal.last_compared);
					if(!realloced)
					{
						free(data);
						set_form_parser_error(parser, E_NOMEM);
						return 0;
					}
					strncpy(data + curpos, parser->config.boundary, parser->internal.last_compared);
					curlen += parser->internal.last_compared;
					curpos += parser->internal.last_compared;
					parser->internal.last_compared = 0;
				}
				// Add next symbol to the data
				data[curpos] = parser->config.buffer[parser->internal.buffer_ptr];
				curpos ++;
			}
			else
			{
				// Hmm... We don't read header... We don't read data...
				// SO WHAT ACTUALLY WE READ????!!!! Arghh... that's hackers again?..
				parser->config.buffer[parser->internal.buffer_read] = 0;
				set_form_parser_corrupted(parser, parser->config.buffer);
				return 0;
			}
		}
	}
	// No more data, but there was no -- in the end. That actually is rather warning than error...
	// But this should never happen if everything is right... So, actually it is rather an error!
	set_form_parser_corrupted(parser, "(unexpected EOF)");
	return 0;
}

int parse_post_data(struct FORM_PARSER *parser)
{
	int result;

	if(!parser)
		return 0;

	if(!(parser->internal.request_method & POST_DATA))
		return 0; // we have nothing to parse

	if(parser->internal.content_type == CT_MULTIPART)
		result = parse_form_multipart(parser);
	else
		result = parse_form_urlencoded(parser);
	
	if(!result)
		parser->internal.data_parsed |= POST_DATA;
	
	return result;
}

int parse_get_data(struct FORM_PARSER *parser)
{
	int result;

	if(!parser)
		return 0;

	result = parse_form_get(parser);
	
	if(!result)
		parser->internal.data_parsed |= GET_DATA;
	
	return result;
}

