#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <ctype.h>

#include <cgi/cgi.h>
#include <cgi/cookies.h>
#include <cgi/session.h>
#include <cgi/sess_file.h>
#include <osdep.h>

void session_load_var(t_session *session, t_session_var *var);
void session_load_vars(t_session *session);
void session_save_var(t_session *session, t_session_var *var);
void session_save_vars(t_session *session);
char *session_generate_id();

void session_register_var(struct s_cgi_context *ctx, char *name, void *var_ptr, 
							t_serializer serializer, t_deserializer deserializer)
{
	t_session *session = ctx->session;
	t_session_var *var, **last;

	if(!session)
	{
		session = ctx->session = session_create();
		session->cgi_ctx = ctx;
	}

	if(!session || !name || !var_ptr)
		return;
	
	last = &session->vars;
	var = session->vars;

	while(var)
	{
		last = &var->next;
		var = var->next;
	}
	
	*last = var = malloc(sizeof(t_session_var));
	memset(var, 0, sizeof(t_session_var));
	
	var->name = strdup(name);
	var->serialize = serializer;
	var->deserialize = deserializer;
	
	var->var = var_ptr;
	
	if(session->sessid)
		session_load_var(session, var);
};

t_session *session_create()
{
	t_session *session;

	session = malloc(sizeof(t_session));
	memset(session, 0, sizeof(t_session));

	session->handler = &file_handler;
	session->timeout = SESSION_TIMEOUT;

	return session;
};

void session_load_var(t_session *session, t_session_var *var)
{
	t_serialized *serialized;

	serialized = session->saved_vars;

	while(serialized)
	{
		if(!strcmp(serialized->name, var->name))
		{
			var->deserialize(serialized, var->var);
			return;
		}

		serialized = serialized->next;
	}
}

void session_load_vars(t_session *session)
{
	t_serialized serialized, *last, *next;
	t_session_var *var;

	if((last = session->saved_vars))
	{
		while(last)
		{
			next = last->next;

			if(last->data)
				free(last->data);

			if(last->name)
				free(last->name);

			free(last);

			last = next;
		}
	}

	if(session->handler)
		session->handler->stream_open(session);
	else
		return;

	if(!session->stream)
		return;

	memset(&serialized, 0, sizeof(serialized));
	last = NULL;

	while(session->handler->load_var(session, NULL, &serialized) > 0)
	{
		next = malloc(sizeof(t_serialized));
		memcpy(next, &serialized, sizeof(serialized));
//		memset(&serialized, 0, sizeof(serialized));

		if(last)
			last->next = next;
		else
			session->saved_vars = next;

		last = next;
	}

	session->handler->stream_close(session);

	var = session->vars;

	while(var)
	{
		session_load_var(session, var);
		var = var->next;
	}
}

void session_save_var(t_session *session, t_session_var *var)
{
	t_serialized *serialized, *last = NULL;

	serialized = session->saved_vars;

	while(serialized)
	{
		if(!strcmp(serialized->name, var->name))
		{
			if(serialized->data)
				free(serialized->data);

			serialized->size = 0;
			var->serialize(serialized, var->var);

			return;
		}

		last = serialized;
		serialized = serialized->next;
	}

	serialized = malloc(sizeof(t_serialized));
	bzero(serialized, sizeof(t_serialized));
	serialized->name = strdup(var->name);
	var->serialize(serialized, var->var);

	if(last)
		last->next = serialized;
	else
		session->saved_vars = serialized;
}

void session_save_vars(t_session *session)
{
	t_session_var *var;
	t_serialized *serialized;

	var = session->vars;

	if(session->handler)
	{
		session->handler->stream_open(session);
		session->handler->stream_clean(session);
	}
	else
	{
		return;
	}

	while(var)
	{
		session_save_var(session, var);
		var = var->next;
	}

	serialized = session->saved_vars;

	while(serialized)
	{
		session->handler->save_var(session, serialized->name, serialized);
		serialized = serialized->next;
	}

	session->handler->stream_close(session);
}

#define SESSID_LENGTH	32

inline char *session_generate_id(char sessid[SESSID_LENGTH+1])
{
//	static char sessid[33] = {[32] = 0};
	register int	i;
	unsigned char c;

	sessid[32] = '\0';

	for(i = 0; i < SESSID_LENGTH; i ++)
	{
		do
		{
			c = (unsigned char)(random() % 255);
		} while(!isalnum(c));
		
		sessid[i] = c;
	}

	return sessid;
}

void session_timeout(struct s_cgi_context *ctx, long timeout)
{
	if(!ctx->session)
	{
		ctx->session = session_create();
	}
	
	ctx->session->timeout = timeout;
}

t_session *session_start(struct s_cgi_context *ctx)
{
	t_session *session = ctx->session;
	char *sessid = NULL;
	int created = 0;
	int new_sessid = 0;

	if(!session)
	{
		session = session_create();
		ctx->session = session;
		session->cgi_ctx = ctx;
		created = 1;
	}

	if(ctx)
		sessid = cookie_get(ctx, SESSION_COOKIE);

	srandom(time(NULL));
	
	if(!sessid)
	{
		new_sessid = 1;
		sessid = (char *)malloc(sizeof(char) * (SESSID_LENGTH + 1));

		do
		{
			sessid = session_generate_id(sessid);
			session->sessid = sessid;
			
			if(!session->handler)
				break;
		
		} while(session->handler->stream_exists(session));

		if(ctx)
			cookie_set(ctx, SESSION_COOKIE, sessid, -1, "/");

		session->sessid = strdup(sessid);
		session->is_new = 1;
	}
	else
	{
		session->sessid = strdup(sessid);
		session->is_new = !session->handler->stream_exists(session);
	}

	session_load_vars(session);
	
	if(session->handler)
		if((long)(random() % 1000) <= (long)(1000 * GC_PROB))
		{
			session->handler->collect_garbage(session);
		}
	
	return session;
};

void session_save(t_session *session)
{
	session_save_vars(session);
};

void session_destroy(struct s_cgi_context *ctx)
{
	t_session *session = ctx->session;

	if(session->handler)
		session->handler->stream_destroy(session);

	cookie_unset(ctx, SESSION_COOKIE);
};

void session_free(t_session *session)
{
	t_session_var *var, *next;
	t_serialized *serialized, *s_next;

	var = session->vars;
	serialized = session->saved_vars;

	while(var)
	{
		next = var->next;

		if(var->name)
			free(var->name);
		
		free(var);
		var = next;
	}

	while(serialized)
	{
		if(serialized->name)
			free(serialized->name);
		
		if(serialized->data)
			free(serialized->data);
		
		s_next = serialized->next;
		free(serialized);
		
		serialized = s_next;
	}

	if(session->stream)
		if(session->handler)
			session->handler->stream_close(session);

	if(session->sessid)
		free(session->sessid);
	
	free(session);
};

