#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <assert.h>
#include <sys/time.h>

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <osdep.h>

#ifdef HAVE_LIBGEN_H
#	include <libgen.h>
#endif

#include <cgi/template.h>
#include <cgi/html_tree.h>
#include <cgi/html_parser.h>

/* 
 * Command processor entries 
 */

// Command processor entry type

struct s_command_processor_entry 
{
	char *name;
	t_command_processor	processor;
	struct s_command_processor_entry *next;
};

// Local prototypes go here

void cmd_include(t_tpl_section *section, TAG* node, char **text, char **tagstart, char **tagend, char *param);
void cmd_strip_tags(t_tpl_section *section, TAG* node, char **text, char **tagstart, char **tagend, char *param);
void cmd_html_special_chars(t_tpl_section *section, TAG* node, char **text, char **tagstart, char **tagend, char *param);

int process_command(t_tpl_section *section, TAG *node, char **where, char **start, char **end);
int process_node(t_tpl_section *section, TAG *node, char **where, t_replacement *replacers);


// List of custom processors

struct s_command_processor_entry *custom_processors = NULL;


// List of built-in command processors

struct s_command_processor_entry std_processors[] = {
		{"include", &cmd_include},
		{"strip_tags", &cmd_strip_tags},
		{"htmlencode", &cmd_html_special_chars}
	};

long std_processors_count = sizeof(std_processors) / \
							sizeof(struct s_command_processor_entry);


/*
 * Valid tags looks like: 
 *	{name} - placeholder
 *	{name:val1|val2|...|valN} - option
 *	{#cmd:param} - command execution
 */

char *tags = "{}";
char *cmdpref = "#";

void register_template_command_processor(char *cmd, 
										 t_command_processor processor)
{
	struct s_command_processor_entry *entry, **next;

	if(!cmd)
		return;

	next = &custom_processors;
	entry = custom_processors;
	
	while(entry)
	{
		if(!strcasecmp(entry->name, cmd))
		{
			entry->processor = processor;
			return;
		}
		next = &entry->next;
		entry = entry->next;
	}
	
	*next = malloc(sizeof(struct s_command_processor_entry));
	bzero(*next, sizeof(struct s_command_processor_entry));
	(*next)->name = strdup(cmd);
	(*next)->processor = processor;
}

void unregister_template_command_processor(char *cmd)
{
	struct s_command_processor_entry *entry, **next;

	if(!cmd)
		return;

	next = &custom_processors;
	entry = custom_processors;
	
	while(entry)
	{
		if(!strcasecmp(entry->name, cmd))
		{
			*next = entry->next;
			free(entry->name);
			free(entry);
			return;
		}
		next = &entry->next;
		entry = entry->next;
	}
}

void free_template_command_processors()
{
	struct s_command_processor_entry *entry, *next;

	entry = custom_processors;
	
	while(entry)
	{
		next = entry->next;

		free(entry->name);
		free(entry);

		entry = next;
	}
}

char *get_replacement_text(char *pch, t_replacement *replacers)
{
	t_replacement *cur;
	char *text = NULL;
	char *tmp, *options, *eol;
	int is_option = 0;
	register char *c = NULL;
	long optcount = 1, optnum;

	tmp = index(pch, ':');
	if(tmp)
	{
		*tmp = 0;
		is_option = 1;
		c = options = tmp + 1;
		
		while(*c)
		{
			if(*c == '|')
				optcount ++;

			c ++;
		}

		c = options;
	}

	cur = replacers;
	
	while(cur)
	{
		if(!strcasecmp(cur->name, pch))
		{
			switch(cur->type)
			{
				case SRT_REPLACEMENT:
					if(cur->value.replacement)
						text = strdup(cur->value.replacement);
					return text;
					break;

				case SRT_OPTION:
					optnum = cur->value.option % optcount;

					if(!c)
						return strdup("");

					while(optnum)
					{
						if(*c == '|')
							optnum --;
						c ++;
					}
					
					eol = c;
					
					while(*eol)
					{
						if(*eol == '|')
						{
							*eol = 0;
							break;
						}
						eol ++;
					}
					
					text = strdup(c);
					*eol = '|';
					*tmp = ':';

					return text;
					break;

				default:
					/* NOTE: this should NEVER happen */
					assert(0);
			}
		}
		
		cur = cur->next;
	}

	if(tmp)
		*tmp = ':';
	
	return NULL;
}

int process_command(t_tpl_section *section, TAG *node, char **where, char **start, char **end)
{
	struct s_command_processor_entry *entry;
	char *tmp, *oldtext;
	char *param = NULL;
	int processed = 0;
	int i;

	oldtext = *where;
	tmp = index(*start, ':');

	if(tmp)
	{
		**end = 0;
		param = strdup(tmp + 1);
		**end = tags[1];
	}

	*tmp = 0;

	for(i = 0; i < std_processors_count; i ++)
	{
		if(!strcasecmp(std_processors[i].name, *start + 2))
		{
			*tmp = ':';
			std_processors[i].processor(section, node, where, start, end, param);
			processed = 1;
			break;
		}
	}

	if(!processed)
	{
		entry = custom_processors;
	
		while(entry)
		{
			if(!strcasecmp(entry->name, *start + 2))
			{
				entry->processor(section, node, where, start, end, param);
				processed = 1;
				break;
			}
			entry = entry->next;
		}
	}

	if(param)
		free(param);

	*tmp = ':';

	if(*where != oldtext)
	{
		node->is_changed = 1;
		free(oldtext);
	}
	
	return processed;
}

int process_node(t_tpl_section *section, TAG *node, char **where, t_replacement *replacers)
{
	char *buf, *openptr, *closeptr, *newtext, *rep, *tag;
	long open[20];// = NULL;
	register long openlen = 0, cur = 0;//, *tmp;
	long buflen, replen, newlen, closelen;
//	int	copied = 0;

	if(!where)
		return 0;
	if(!*where)
		return 0;
	
	buf = *where;

	while(buf[cur])
	{
		if(buf[cur] == '{')
		{
			openlen ++;

			if(openlen > 20)
			{
				return 0;
			}
			
			open[openlen-1] = cur;
		}

		if(buf[cur] == '}' && openlen > 0)
		{
			openptr = &buf[open[openlen-1]];
			closeptr = &buf[cur];

			if(buf[open[openlen-1]+1] == '#')
			{
				// Process command - returns false if nothing has changed
				if(process_command(section, node, &buf, &openptr, &closeptr))
				{
					node->is_changed = 1;
					cur = closeptr - buf;

					open[openlen-1] = cur+1;
					
					// In case of overflow (not for real life)
					assert(open[openlen-1] > cur);
					
					while(open[openlen-1] > cur && openlen > 0)
						openlen --;
					
					continue;
				}
			}
			else
			{
				// Insert placeholder
				buf[cur] = 0;
				buf[open[openlen-1]] = 0;

				tag = &buf[open[openlen-1]+1];
				
				rep = get_replacement_text(tag, replacers);

				if(!rep)
				{
					buf[cur] = '}';
					buf[open[openlen-1]] = '{';
				}
				else
				{
					node->is_changed = 1;
	
					buflen   = strlen(buf);
					replen   = strlen(rep);
					closelen = strlen(closeptr + 1);
					newlen  = (buflen + replen + closelen) + 1;
					newtext = malloc(newlen); 
		
					strcpy(newtext, buf);
					strcat(newtext, rep);
					strcat(newtext, closeptr + 1);
	
					free(buf);
					free(rep);
	
					buf = newtext;
					cur = buflen + replen - 1;
				}
			}

			openlen --;
		}

		cur ++;
	}

	*where = buf;

	return 1;
}

void cmd_include(t_tpl_section *section, TAG* node, char **text, char** tagstart, char** tagend, char *param)
{
	t_template *tpl;
	char *tag1, *tag2;
	TAG	*tmp;
	char *filename, *dname;
	long len;

	if(!node->is_text_node)
		return;

	if(!param)
		return;

	dname = dirname_r(section->template->__file);
	
	len = strlen(dname);
	filename = malloc(len + strlen(param) + 2);

	strcpy(filename, dname);
	filename[len] = DIR_SEPARATOR;
	filename[len + 1] = '\0';
	strcat(filename, param);

	tpl = template_create(filename);

	free(filename);
	free(dname);

	if(!tpl)
		return;

	**tagstart = 0;
	
	tag1 = strdup(*text);
	tag2 = strdup((*tagend) + 1);

	tmp = createTextNode();
	tmp->text = tag2;
	
	insertAfter(node, tmp);
	insertAfter(node, copyTree(tpl->document));

	template_free(tpl);

	*text = tag1;
	*tagstart = *tagend = (tag1 + strlen(tag1));
};

void cmd_strip_tags(t_tpl_section *section, TAG* node, char **text, char** tagstart, char** tagend, char *param)
{
	char *stripped, *tag;
	long len;

	if(!param)
		return;

	stripped = strip_tags(param);
	
	len = strlen(stripped);
	
	tag = malloc(strlen(*text) + len + strlen(*tagend) + 1);
	
	*tag = '\0';

	**tagstart = 0;
	strcat(tag, *text);
	strcat(tag, stripped);
	strcat(tag, (*tagend)+1);

	free(stripped);

	*text = tag;
	*tagstart = *tagend = (tag + len);
};

void cmd_html_special_chars(t_tpl_section *section, TAG* node, char **text, char** tagstart, char** tagend, char *param)
{
	char *stripped, *tag;
	long len;

	if(!param)
		return;

	stripped = html_special_chars(param);
	
	len = strlen(stripped);
	
	tag = malloc(strlen(*text) + len + strlen(*tagend) + 1);
	
	*tag = '\0';

	**tagstart = 0;
	strcat(tag, *text);
	strcat(tag, stripped);
	strcat(tag, (*tagend)+1);

	free(stripped);

	*text = tag;
	*tagstart = *tagend = (tag + len);
};

t_tpl_section *template_section_reset(t_tpl_section *section, TAG *node)
{
	TAG *next = NULL;//, *parent, *prevsibling, *nextsibling;
	t_tpl_section *last = NULL;
	t_replacement *replace, *rnext;

	rnext = replace = section->replacements;

	while(replace)
	{
		rnext = replace->next;

		if(replace->name)
			free(replace->name);

		if(replace->value.replacement && (replace->type == SRT_REPLACEMENT))
			free(replace->value.replacement);

		free(replace);
		
		replace = rnext;
	}

	if(!section->document)
		section->document = copyBranch(node);

	if(!node)
	{
		section->__safe_copy = NULL;
		last = section->sections;

		while(last)
		{
			template_section_reset(last, NULL);

			last = last->next;
		}	

		return section;
	}

	node->is_visible = 0;

	section->__safe_copy = node;

	section->replacements = NULL;

	if(section->document->children)
		next = findNode(section->document->children, SECTION_TAG);

	last = section->sections;

	while(next && last)
	{
		template_section_reset(last, next);

		last = last->next;
		
		next = findNextPlain(next);
	}	

	return section;
}

t_tpl_section *template_section_create(t_tpl_section *root, TAG *node)
{
	TAG *next = NULL;//, *parent, *nextsibling, *prevsibling;
	t_tpl_section *last = NULL, *section, *current;
	char *name;

	node->is_visible = 0;

	section = malloc(sizeof(t_template));
	bzero(section, sizeof(t_template));

	section->template = root;

	section->document = copyBranch(node);
	section->__safe_copy = node;
	
	name = getAttribute(node, "name");
	if(name)
		section->name = strdup(name);

	next = findNode(section->document->children, SECTION_TAG);

	while(next)
	{
		current = template_section_create(root, next);

		if(!last)
			section->sections = current;
		else
			last->next = current;

		last = current;
		next = findNextPlain(next);
	}	

	return section;
}

t_template *template_create(char *filename)
{
	t_template *tpl;
	t_tpl_section *current, *last = NULL;
	TAG *node;
	FILE *file;

	tpl = malloc(sizeof(t_template));
	bzero(tpl, sizeof(t_template));

	if(filename)
	{
		file = fopen(filename, "r");
		if(!file)
		{
			free(tpl);
			return NULL;
		}
		node = html_parse_file(file);
		fclose(file);
	}
	else
	{
		node = html_parse_file(NULL);
	}

	tpl->__file = strdup(filename);
	tpl->template = tpl;
	tpl->__safe_copy = node;
	tpl->document = copyTree(node);
	
	node = findNode(tpl->document, SECTION_TAG);

	while(node)
	{
		current = template_section_create(tpl, node);

		if(!last)
			tpl->sections = current;
		else
			last->next = current;

		last = current;
		node = findNextPlain(node);
	}
	
	return tpl;
}

void template_set(t_tpl_section *section, char *name, char *fmt, ...)
{
	char *replace;
	t_replacement *replacement = NULL, *cur, **next = NULL;
	t_tpl_section *subsection;
	int existing = 0;//, size;
	va_list v;

	if(!section)
		return;

	va_start(v, fmt);
	vasprintf(&replace, fmt, v);
	va_end(v);

	cur = section->replacements;

	if(!cur)
	{
		replacement = malloc(sizeof(t_replacement));
		section->replacements = replacement;
	}
	else
	{
		while(cur)
		{
			if(!strcasecmp(cur->name, name))
			{
				if(cur->type == SRT_REPLACEMENT)
					free(cur->value.replacement);
				replacement = cur;
				existing = 1;
				break;
			}
			
			next = &cur->next;
			cur = cur->next;
		}

		if(!replacement)
		{
			replacement = malloc(sizeof(t_replacement));
			*next = replacement;
		}
	}
	
	if(!existing)
	{
		bzero(replacement, sizeof(t_replacement));

		if(name)
			replacement->name = strdup(name);
	}
	
	replacement->value.replacement = replace;
	replacement->type = SRT_REPLACEMENT;

	subsection = section->sections;

	while(subsection)
	{
		template_set(subsection, name, "%s", replace);
		subsection = subsection->next;
	}
};

void template_opt(t_tpl_section *section, char *name, long option)
{
	t_replacement *replacement = NULL, *cur, **next = NULL;
	t_tpl_section *subsection;
	int existing = 0;

	if(!section)
		return;

	cur = section->replacements;

	if(!cur)
	{
		replacement = malloc(sizeof(t_replacement));
		section->replacements = replacement;
	}
	else
	{
		while(cur)
		{
			if(!strcasecmp(cur->name, name))
			{
				if(cur->type == SRT_REPLACEMENT)
					free(cur->value.replacement);
				replacement = cur;
				existing = 1;
				break;
			}
			
			next = &cur->next;
			cur = cur->next;
		}

		if(!replacement)
		{
			replacement = malloc(sizeof(t_replacement));
			*next = replacement;
		}
	}
	
	if(!existing)
	{
		bzero(replacement, sizeof(t_replacement));

		replacement->name = strdup(name);
	}
	
	replacement->value.option = option;
	replacement->type = SRT_OPTION;

	subsection = section->sections;

	while(subsection)
	{
		template_opt(subsection, name, option);
		subsection = subsection->next;
	}
};

void inline template_section_parse(t_tpl_section *tpl)
{
	TAG *node;
	ATTR *attr;

	if(!tpl)
		return;

	node = tpl->document;

	/* 
	 * Parse section - replace placeholders and
	 * process commands
	 */	
	do
	{

		if(node->is_text_node)
		{

			process_node(tpl, node, &node->text, tpl->replacements);

		}
		else
		{

			process_node(tpl, node, &node->tag_name, tpl->replacements);
		
			attr = node->attributes;
			
			while(attr)
			{
				process_node(tpl, node, &attr->name, tpl->replacements);
				process_node(tpl, node, &attr->value, tpl->replacements);
			
				attr = attr->next;
			}
		}

	} while((node = recurseNodes(node)));
}

void template_section_out(t_tpl_section *tpl)
{
	TAG *node;

	if(!tpl)
		return;

	/* Parse section */
	template_section_parse(tpl);

	if(tpl->document)
	{
		insertBefore(tpl->__safe_copy, extractFrom(tpl->document));
		deleteNode(tpl->document);
	}

	tpl->document = NULL;

	/* Reset section after output */	
	node = tpl->__safe_copy;
	template_section_reset(tpl, node);
};

void template_out(t_tpl_section *tpl, FILE *out)
{
//	TAG *node;
//	ATTR *attr;

	if(!tpl)
		return;

	/* Parse section */
	template_section_parse(tpl);
	
	/* Output document tree */	
	dumpTree(tpl->document, out);
};

void template_insert(t_tpl_section *dest, char *where, t_tpl_section *src)
{
	TAG *node;
	TAG *parent, *nextsibling, *prevsibling;
	TAG *tmp;
	char *left, *right;
	char *part1, *part2;
	char *tag;
	char save;
	t_tpl_section *tpl = dest;

	template_section_parse(src);
	
	parent = tpl->document->parent;
	nextsibling = tpl->document->next_sibling;
	prevsibling = tpl->document->prev_sibling;
	tpl->document->parent = NULL;
	tpl->document->next_sibling = NULL;
	tpl->document->prev_sibling = NULL;

	node = tpl->document;
	
	tag = malloc(strlen(where) + 3);
	*tag = tags[0];
	save = tags[1];
	tag[1] = 0;
	strcat(tag, where);
	strcat(tag, tags + 1);

	do
	{
		if(node->is_text_node)
		{
			left = strcasestr(node->text, tag);
			if(!left)
				continue;

			right = left + strlen(tag);
			*left = 0;
			
			part1 = strdup(node->text);
			part2 = strdup(right);
			
			*left = tags[0];
			
			free(node->text);
			node->text = part1;
			
			tmp = createTextNode();
			tmp->text = part2;
			
			insertAfter(node, tmp);
			insertAfter(node, copyTree(src->document));
			
			node = node->next_sibling->next_sibling;
		}
	} while((node = recurseNodes(node)));

	free(tag);
	
	tpl->document->parent = parent;
	tpl->document->next_sibling = nextsibling;
	tpl->document->prev_sibling = prevsibling;
};

t_tpl_section *template_find_section(t_template *tpl, char *name)
{
	t_tpl_section *cur, *found;

	if(!tpl)
		return NULL;

	cur = tpl->sections;

	while(cur)
	{
		if(!strcasecmp(cur->name, name))
		{
			return cur;
		}
		else if((found = template_find_section(cur, name)))
		{
			return found;
		}
		
		cur = cur->next;
	}
	
	return NULL;
}

void template_free(t_tpl_section *section)
{
	t_replacement *replace, *rnext;
	t_tpl_section *cur, *snext;

	cur = section->sections;
	
	while(cur)
	{
		snext = cur->next;
		template_free(cur);
		cur = snext;
	}

	if(section->document)
		deleteTree(section->document);

	if(section->__safe_copy)
	{
		removeChild(section->__safe_copy->parent, section->__safe_copy);
		deleteTree(section->__safe_copy);
	}

	rnext = replace = section->replacements;

	if(section->__file)
		free(section->__file);

	while(replace)
	{
		rnext = replace->next;

		if(replace->name)
			free(replace->name);

		if(replace->value.replacement && (replace->type == SRT_REPLACEMENT))
			free(replace->value.replacement);

		free(replace);
		
		replace = rnext;
	}

	section->replacements = NULL;

	if(section->name)
		free(section->name);

	free(section);	
}

