/*
 * Copyright (c) 2002 RusNet #c channel
 * All rights reserved
 *
 * This file is a part of mproxy package, and implements a UNIX bouncer.
 * See descr.txt for more details
 *
 * $Id$
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <sys/file.h>

#include <arpa/inet.h>
#include <netinet/in.h>

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <time.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <netdb.h>
#include <ctype.h>

#include "base64.h"
#include "common.h"

/* debug tracing */
#ifdef __DEBUG__
#define dprintf(x)	(void) fprintf x
#else
#define dprintf(x)
#endif /* __DEBUG__ */

/*
 * Prototypes
 */
static void sighndlr(int signo);
static void checkmail(FILE *fp);
static void usend(struct header *sp, unsigned char *data, int datalen);
static void urecv(struct tdata *tp);
static void checkconns(void);
static struct tdata *addconn(struct header *sp);
static void delconn(struct tdata *tp);
static void usage(const char *prog);
static void edaemon(void);

/*
 * Globals
 */
static int __quit;			/* quit indicator */
static struct tdata *__connlist;	/* connection list */

/*
 * Forward data to the socket identified by id
 */
void
usend(struct header *sp, unsigned char *data, int datalen)
{
	struct tdata *tp, *tmp;

	/* find out what connection should receive the */ 
	for (tp = NULL, tmp = __connlist; tmp != NULL; tmp = tmp->next) {
		if (tmp->id == sp->id) {
			tp = tmp;
			break;
		}
	}

	/* not found, this is a new connection request */
	if (tp == NULL && (datalen <= 0 || (tp = addconn(sp)) == NULL))
		return;

	/* is it a disconnect request ? */
	if (datalen <= 0) {
		dprintf((stderr, "usend: (%s) -> %s:%d requested"
		    " disconnect\n", (char *) tp->data,
		    inet_ntoa(tp->sa.sin_addr), ntohs(tp->sa.sin_port)));
		delconn(tp);
		return;
	}

	dprintf((stderr, "usend: %s:%d %d bytes => %s:%d\n",
	    (char *) tp->data, tp->id, datalen,
	    inet_ntoa(tp->sa.sin_addr), ntohs(tp->sa.sin_port)));

	/* forwad data to the remote server */
	(void) send(tp->sock, data, datalen, 0);
}

/*
 * monitor all connections for incoming data
 */
void
checkconns(void)
{
	struct tdata *tp, *tmp;
	struct timeval tv = { 0L, 100000L };	/* 0.1 sec */
	fd_set rfds;

	FD_ZERO(&rfds);

	/* add all sockets to a descriptors set */
	for (tp = __connlist; tp != NULL; tp = tp->next)
		FD_SET(tp->sock, &rfds);

	/* examine them */
	if (select(1024, &rfds, NULL, NULL, &tv) > 0)
		FOREACH_CONN_SAFE(__connlist, tp, tmp)
			if (FD_ISSET(tp->sock, &rfds))
				urecv(tp);
}

/*
 * Connection descriptor passed has new data arrived.
 * mail the data to the email address stored in tp->data
 */
void
urecv(struct tdata *tp)
{
	char ibuf[BSIZ], shellcmd[BSIZ * 2], b64[BSIZ * 2];
	int nbytes;

	/*
	 * read once. if data remains in the socket, subsequent calls
	 * will exctract it
	 */
	nbytes = recv(tp->sock, ibuf, sizeof(ibuf), 0);

	/* if error, mark as disconnected */
	if (nbytes < 0)
		nbytes = 0;

	/* null-terminate data */
	ibuf[nbytes] = '\0';

	/* crypt the data */
	cryptdecrypt(ibuf, nbytes);

	/* pack the data */
	(void) base64encode(ibuf, nbytes, b64);

	/* mail it */
	(void) snprintf(shellcmd, sizeof(shellcmd),
	    "echo \"" MAGIC " %d/%d/%s/%s/%d : %s\" | mail -s kuku %s",
	    tp->msgno++, tp->id, "x", "x", 0, b64, (char *) tp->data);

	if (system(shellcmd) != 0) {
		dprintf((stderr, "urecv: mail command failed\n"));
	} else {
		dprintf((stderr, "urecv: %s:%d %d bytes => %s\n",
		    inet_ntoa(tp->sa.sin_addr), ntohs(tp->sa.sin_port),
		    nbytes, (char *) tp->data));
	}
	
	/* if 0 received, server has disconnected */
	if (nbytes == 0) {
		dprintf((stderr, "urecv: %s:%d (%s) disconnected\n",
		    inet_ntoa(tp->sa.sin_addr), ntohs(tp->sa.sin_port),
		    (char *) tp->data));
		delconn(tp);
	}
}

/*
 * Register new connection
 */
struct tdata *
addconn(struct header *sp)
{
	struct tdata *new;
	struct hostent *hep;

	/* allocate new connection descriptor */
	if ((new = calloc(1, sizeof(struct tdata))) == NULL) {
		dprintf((stderr, "addconn: calloc() failed\n"));
		return (NULL);
	}

	/* create socket */
	if ((new->sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
		dprintf((stderr, "addconn: socket() failed (%s)\n",
		    strerror(errno)));
		free(new);
		return (NULL);
	}

	/* resolve the remote server */
	new->sa.sin_family = AF_INET;
	new->sa.sin_port = htons(sp->port);
	if ((hep = gethostbyname(sp->server)) == NULL) {
		dprintf((stderr, "addconn: gethostbyname(%s) failed (%s)\n",
		    sp->server, strerror(errno)));
		free(new);
		(void) close(new->sock);
		return (NULL);
	}
	(void) memcpy(&new->sa.sin_addr, hep->h_addr_list[0], 4);
	
	/* try to connect to the remote server */
	if (connect(new->sock, (struct sockaddr *) &new->sa,
	    sizeof(struct sockaddr_in)) != 0) {
		dprintf((stderr, "addconn: connect(%s:%d) failed (%s)\n",
		    sp->server, sp->port, strerror(errno)));
		free(new);
		(void) close(new->sock);
		return (NULL);
	}

	/* connected. remember the email address and id */
	new->data = strdup(sp->email);
	new->id = sp->id;

	dprintf((stderr, "addconn: new connection requested (%s:%d -> %s:%d)\n",
	    sp->email, sp->id, sp->server, sp->port));

	/* add node to a list of opened connections */
	addnode(new, &__connlist);

	return (new);
}

/*
 * Drop the connection
 */
void
delconn(struct tdata *tp)
{
	/* close the socket */
	(void) close(tp->sock);

	/* Remove descriptor from connection list */
	delnode(tp, &__connlist);

	/* release descriptor */
	free(tp->data);
	free(tp);
	
	/* XXX send disconnect request to the windows box ? */
}

/*
 * process the incoming mail
 */
void
checkmail(FILE *fp)
{
	
	char line[BSIZ * 2], b64[BSIZ * 2], *p;
	int dlen;
	struct header header;
	struct stat st;

	if (fstat(fileno(fp), &st) != 0) {
		dprintf((stderr, "checkmail: fstat() failed (%s)",
		    strerror(errno)));
		return;
	}

	/* if there is no mail, return */
	if (st.st_size == 0)
		return;

	/* scan the mailbox line by line */
	while (fgets(line, sizeof(line), fp) != NULL) {

		/* check for signature */
		if (memcmp(line, MAGIC, strlen(MAGIC))
		    || parseheader(line, &header) != 0)
			continue;

		/* find the beginning of the message */
		if ((p = strchr(line, ':')) == NULL) {
			dprintf((stderr, "checkmail: corrupted message [%s]\n",
			    line));
			continue;
		}

		/* skip all whitespaces */
		for (++p; isspace(*p); )
			p++;

		/* decode message */
		dlen = base64decode(p, b64);

		/* decrypt message */
		cryptdecrypt(b64, dlen);

		/* forward the content to the remote server */
		usend(&header, b64, dlen);
	}
}

/*
 * The signal handler
 */
void sighndlr(int sig)
{
	dprintf((stderr, "sighndlr: caught signal %d, quitting\n", sig));
	__quit++;	/* mark quit */
}

/*
 * fork into background
 */
void
edaemon(void)
{
	pid_t pid;
	int fd;
	struct rlimit rlim;

	/* ignore all terminal-related signals */
	if (getppid() != 1) {
		signal(SIGTTOU, SIG_IGN);
		signal(SIGTTIN, SIG_IGN);
		signal(SIGTSTP, SIG_IGN);
	}

	/* fork, and parent exits*/
	if ((pid = fork()) != 0) {
		(void) printf("forked with pid [%d]\n", pid);
		exit(0);
	}

	/* become a session leader */
	setsid();

	/* close all file descriptors - especially stdin/stdout etc */
	getrlimit(RLIMIT_NOFILE, &rlim);
	for (fd = 0; fd < rlim.rlim_max; fd++)
		(void) close(fd);

	/* go to root, so don't make a problems to umount */
	(void) chdir("/");
}

/*
 * Display usage info and exit
 */
void
usage(const char *prog)
{
	(void) fprintf(stderr, "usage:\n%s [-d] [-l <file>]\n-d:\tdon't detach "
	    "from terminal\n-l:\tlog into file\n", prog);
	exit(1);
}

/*
 * The entry point
 */
int
main(int argc, char **argv)
{
	struct tdata *tp, *tmp;
	FILE *fp;
	char *mbox, *logfile;
	extern char *optarg;
	int ch, debug;

	debug = 0;
	logfile = NULL;

	while ((ch = getopt(argc, argv, "dl:")) != -1)
		switch (ch) {
			case 'd':
				debug++;
				break;
			case 'l':
				logfile = optarg;
				break;
			case '?':
				/* FALLTROUGH */
			default:
				usage(argv[0]);
		}

	/* if not in debug mode, detach from terminal */
	if (!debug)
		edaemon();

	/* log into alternative logfile, if requested */
	if (logfile != NULL)
		(void) freopen(logfile, "a+", stderr);

	if ((mbox = getenv("MAIL")) == NULL) {
		fprintf(stderr, "MAIL environment variable is not set\n");
		exit(1);
	}

	/* set the signal handlers */
	(void) signal(SIGINT, sighndlr);
	(void) signal(SIGTERM, sighndlr);

	/* fall into serving loop */
	for (fp = NULL, __quit = 0; __quit == 0; ) {
		/* sleep */
		(void) usleep(100000U);	/* 0.1 sec */

		/* open a mailbox */
		if (fp == NULL && (fp = fopen(mbox, "r+")) == NULL) {
			dprintf((stderr, "main: fopen(%s) failed (%s)",
			    mbox, strerror(errno)));
			continue;
		}

		/* lock it exclusively */
		if (flock(fileno(fp), LOCK_EX) != 0) {
			dprintf((stderr, "main: flock(%s) failed (%s)",
			    mbox, strerror(errno)));
			continue;
		}

		/* handle incoming mail */
		checkmail(fp);

		/* erase the file content */
		(void) ftruncate(fileno(fp), 0);

		/* release the lock */
		(void) flock(fileno(fp), LOCK_UN);

		/* close file */
		(void) fclose(fp);
		fp = NULL;

		/* examine all connections for incoming data */
		checkconns();
	}

	/* deallocate all connections */
	FOREACH_CONN_SAFE(__connlist, tp, tmp) {
		dprintf((stderr, "main: closing %s:%d -> %s:%d\n",
		    (char *) tp->data, tp->id,
		    inet_ntoa(tp->sa.sin_addr), ntohs(tp->sa.sin_port)));
		delconn(tp);
	}

	return (0);
}
