/*
 * Copyright (c) 2002 #c channel on RusNet
 * All rights reserved
 *
 * Authors:
 *   Sergey Lyubka aKa devnull: architecture & lead programmer
 *
 * Description:
 *   this is a part of `mailproxy' package.
 *   mailproxy allows to tunnel TCP connection using SMTP tunnel
 *   this file contains source code for the win32 client.
 *
 *   In order to provide the service, two threads are started.
 *   first thread, function `unpacker', listens for incoming mail, verifying
 *   the mailbox. If incoming mail contains the data from the remote mail
 *   address, this thread puts the data into the listening socket.
 *   second thread, fuction `listener', opens a listening
 *   socket. On every connect, listener spawns a thread 'packer' that packs
 *   all data received on that socket into a mail messages and sends them
 *   to the remote mail address
 *
 * Compilation:
 *   o MSVC:
 *     cl winclnt.c base64.c common.c user32.lib advapi32.lib ws2_32.lib /ML
 *     (make sure you have environment variables set
 *     set include=<DRIVE>:\PATH\TO\MSVC\include
 *     set lib=<DRIVE>:\PATH\TO\MSVC\lib)
 *
 * $Id$
 */

#include <windows.h>
#include <winsock.h>
#include <winreg.h>
#include <process.h>
#include <stdio.h>
#include <stdarg.h>
#include <commctrl.h>

#include "base64.h"
#include "common.h"

/* Local listenint port */
#define	SMTPPORT	25
#define	POP3PORT	110

/*
 * structure that holds global data
 */
struct gdata {
	CRITICAL_SECTION syncr;	/* synchronization object */
	char	mailserv[BSIZ], 
		mailuser[BSIZ],
		mailpasswd[BSIZ],
		email[BSIZ],	/* remote email */
		server[BSIZ],	/* remote server */
		port[BSIZ],	/* remote port */
		lport[BSIZ];	/* local port */
	int	sock;		/* listening socket. -1 means disconnected */
	unsigned flags;
	struct tdata *list;	/* list of active connections */
	
};
/* valid flags  */
#define FL_DISCONNECT		(1 << 0)	/* force disconnect */
#define	FL_UNPACKER_ON		(1 << 1)	/* unpacker thread is running */
#define	FL_LISTENER_ON		(1 << 2)	/* listener thread is running */
#define	FL_CONNECTED		(1 << 3)	/* state is 'connected' */
/* macros to synchronize global data access */
#define LOCKDATA(data)		EnterCriticalSection(&((data)->syncr))
#define UNLOCKDATA(data)	LeaveCriticalSection(&((data)->syncr))


/*
 * Control identifiers
 */
#define ID_GRP_LOCAL		50
#define ID_GRP_REMOTE		51

#define ID_BTN_QUIT		100
#define ID_BTN_CONNECT		101
#define ID_BTN_LOG		102

#define ID_EDIT_MAILSERVER	150
#define ID_EDIT_MAILPASSWD	151
#define ID_EDIT_MAILUSER	152
#define ID_EDIT_EMAIL		153
#define ID_EDIT_SERVER		154
#define ID_EDIT_PORT		155
#define ID_EDIT_LPORT		156

#define ID_STATIC_MAILSERVER	200
#define ID_STATIC_MAILPASSWD	201
#define ID_STATIC_MAILUSER	202
#define ID_STATIC_EMAIL		203
#define ID_STATIC_SERVER	204
#define ID_STATIC_PORT		205
#define ID_STATIC_LPORT		206
#define ID_STATIC_COPYRIGHT	207

#define	ID_STATUSBAR		300

/*
 * Prototypes
 */
static BOOL CALLBACK DlgProc(HWND, UINT, WPARAM, LPARAM);
static UINT showerr(UINT type, const char *fmt, ...);
static void initdlg(HWND);
static const char *reggetval(int id);
static int reginit(void);
static const char *idtoname(int id);
static void setregval(HWND hDlg, int id);
static void savedlg(HWND hDlg, struct gdata *dp);
static void setguistate(HWND hDlg, struct gdata *dp);
static void listener(void *);
static void unpacker(void *);
static void packer(void *);
static void mail(struct tdata *, char *, int);
static char *mailcmd(int, const char *fmt, ...);
static int mailconn(const char *mailserv, short port);
static void scanbox(int sock, struct gdata *dp);
static void killsock(int *sock);
static int pop3login(const char *serv, const char *user, const char *passwd);
static char *nbread(int sock);
static int unpack(char *reply, struct gdata *dp);
static void trace(const char *fmt, ...);

/*
 * Global variables
 */
static const char *regpath = "Software\\RusNet\\mproxy";
static HWND hbar;	/* statusbar window */

/*
 * Put the trace message onto the status bar
 */
void
trace(const char *fmt, ...)
{
	HWND hDlg;
	FILE *fp;
	char msg[BSIZ * 3];
	va_list ap;

	/* windows lacks `vsnprintf'. XXX possible buffer overflow */
	va_start(ap, fmt);
	(void) vsprintf(msg, fmt, ap);
	va_end(ap);

	if ((hDlg = GetParent(hbar)) != NULL) {
		if (GetWindowLong(hDlg, GWL_USERDATA)!= 0) {
			/* debug checkbox is checked, print to a file */
			if ((fp = fopen("mroxy.log", "a+")) != NULL) {
				(void) fprintf(fp, "%s\n", msg);
				(void) fclose(fp);
			}
		}
	}


	/* display the prepared content */
	SetWindowText(hbar, msg);
}

/*
 * non-blocking read from the socket
 */
char *
nbread(int sock)
{
	static char ibuf[BSIZ * 2];	/* XXX oops. tough restriction! */
	fd_set fdset;
	int n, ntotal;
	struct timeval tv = { 1L, 1L }; /* no more than 1 second */

	FD_ZERO(&fdset);
	FD_SET(sock, &fdset);

	n = ntotal = 0;

	while (select(sock + 1, &fdset, NULL, NULL, &tv) > 0) {
		/* got something */
		n = recv(sock, ibuf + ntotal, sizeof(ibuf) - ntotal, 0);

		/* if error, exit from here */
		if (n < 0) {
			ntotal = 0;
			break;
		}
		
		ntotal += n;

		/* for pop3, '.' means end of reply */
		if (ntotal > 2 && memcmp(&ibuf[ntotal - 3], ".\r\n", 3) == 0)
			break;

		if (ntotal >= (sizeof(ibuf) - 1)) {
			trace("nbread: buffer wrapped! corrupted stream");
			ntotal = 0;	/* XXX wrap buffer */
		}

		FD_ZERO(&fdset);
		FD_SET(sock, &fdset);
	}

	/* null - terminate it `*/
	ibuf[ntotal] = '\0';
	
	trace("nbread:\n[%s]\n", ibuf);

	return (ibuf);
}

/*
 * shuts down the socket
 */
void
killsock(int *sock)
{
	(void) closesocket(*sock);
	(void) shutdown(*sock, 2);
	*sock = -1;
}

/*
 * send an SMTP command over a socket, return NULL on error
 */
char *
mailcmd(int sock, const char *fmt, ...)
{
	int nbytes;
	char cmd[BSIZ * 3];
	va_list ap;
	
	/* windows lacks `vsnprintf'. XXX possible buffer overflow */
	va_start(ap, fmt);
	(void) vsprintf(cmd, fmt, ap);
	va_end(ap);
	
	/* append the line feed */
	(void) strcat(cmd, "\r\n");
	nbytes = strlen(cmd);

	trace("mailcmd:\n[%s]\n", cmd);

	(void) send(sock, cmd, nbytes, 0);

	return (nbread(sock));
}

/*
 * connect to the mail server for sending mail
 */
int
mailconn(const char *mailserv, short port)
{
	struct sockaddr_in sa;
	struct hostent *hep;
	int fd;
	
	/* TODO: error checking */
	fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	sa.sin_family = AF_INET;
	sa.sin_port = htons(port);
	
	if ((hep = gethostbyname(mailserv)) == NULL) {
		showerr(MB_OK | MB_ICONEXCLAMATION,
			"gethostbyname(%s) failed", mailserv);
		return (-1);
	}
	(void) memcpy(&sa.sin_addr.s_addr, hep->h_addr_list[0], 4);
	
	if (connect(fd, (struct sockaddr *) &sa, sizeof(sa)) != 0) {
		showerr(MB_OK | MB_ICONEXCLAMATION,
			"connect to %s:%d failed", mailserv, port);
		return (-1);
	}

	/* read the greeting from server */
	(void) nbread(fd);

	return (fd);
}

/*
 * forward the data to the mailer. this function should be reentrant
 */
void
mail(struct tdata *tp, char *obuf, int len)
{
	int rval;
	static int sock = -1;	/* socket to the mailer */
	char b64[BSIZ * 2];
	struct gdata *dp = (struct gdata *) tp->data;

	/*
	 * initialize the socket to the mailer.
	 * FIXME: race is here. sock should be protected (mutex or so)
	 */
	if (sock == -1) 
		sock = mailconn(dp->mailserv, SMTPPORT);

	/* crypt the data */
	cryptdecrypt(obuf, len);

	/* base64 - encode the data we want to send */
	(void) base64encode(obuf, len, b64);

	/* try to connect to the mailer */
	do {
		rval = strlen(mailcmd(sock, "HELO localhost"));
		if (rval == 0)
			sock = mailconn(dp->mailserv, SMTPPORT);
	} while (rval == 0);

	mailcmd(sock, "MAIL FROM: %s", dp->mailuser);
	mailcmd(sock, "RCPT TO: %s", dp->email);
	mailcmd(sock, "DATA");
	mailcmd(sock, "From: Vasya Pupkin\r\n"
	    "Subject: ku ku\r\n\r\n" MAGIC " %d/%d/%s/%s/%s:%s\r\n.",
	    tp->msgno++, tp->id, dp->mailuser, dp->server, dp->port, b64);
}

/*
 * somebody connected to the listener socket, and new thread spwaned
 * for serving this connection. this is the entry point for the new thread
 */
void
packer(void *arg)
{
	struct tdata tdata;
	struct gdata *dp;
	char ibuf[BSIZ];
	int nbytes;

	/* 
	 * make a copy of thread data immediately. every thread gets its
	 * private stack area, to tdata is allocated in private thread's space
	 * XXX: race is possible here, 'cause the content of *arg
	 * may be changed by the caller (which is unlikely anyway)
	 */
	tdata = * (struct tdata *) arg;
	dp = (struct gdata *) tdata.data;

	/* add ourseleves to the list of active connections */
	LOCKDATA(dp);
	addnode(&tdata, &dp->list);
	UNLOCKDATA(dp);

	tdata.id = GetCurrentThreadId();

	trace("packer: thread %d connected", tdata.id);

	do {
		nbytes = recv(tdata.sock, ibuf, sizeof(ibuf), 0);
		if (nbytes > 0)
			mail(&tdata, ibuf, nbytes);
	} while (nbytes > 0);

	/*
	 * if nbytes == 0, client disconnected; if < 0, error.
	 * in either case, cleanup and return, finishing the thread
	 */
	killsock(&tdata.sock);

	trace("packer: thread %d disconnected", tdata.id);

	/* remove ourselves from the list of active connections */
	LOCKDATA(dp);
	delnode(&tdata, &dp->list);
	UNLOCKDATA(dp);
}

/*
 * the `listener' thread entry point
 */
void
listener(void *arg)
{
	struct gdata *dp = arg;
	int sock, quit = 0;
	struct sockaddr_in sa;
	fd_set fdset;
	struct timeval tv = { 0L, 100000L };	/* 0.1 sec */
	struct tdata tdata;
	
	/* are we started already ? */
	LOCKDATA(dp);
	if (dp->flags & FL_LISTENER_ON)
		quit++;
	else
		dp->flags |= FL_LISTENER_ON;	/* no, set 'running' flag */
	UNLOCKDATA(dp);
	
	if (quit) {
		showerr(MB_OK, "listener is running already");
		return;	/* yes, we're running, return */
	}

	/* create listening socket */
	sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (sock == INVALID_SOCKET) {
		showerr(MB_OK | MB_ICONEXCLAMATION, "invalid_sock");
		return;
	}

	/* bind socket to a local port */
	sa.sin_family = AF_INET;
	sa.sin_port = htons((short) atoi(dp->lport));
	sa.sin_addr.s_addr = INADDR_ANY;
	if (bind(sock, (struct sockaddr *) &sa, sizeof(sa)) != 0) {
		showerr(MB_OK | MB_ICONEXCLAMATION, "bind() failed");
		return;
	}

	/* queue no more than 1 connection */
	if (listen(sock, 1) != 0) {
		showerr(MB_OK | MB_ICONEXCLAMATION, "listen() failed");
		return;
	}

	/* mark globally that we are connected */
	LOCKDATA(dp);
	dp->sock = sock;
	dp->flags |= FL_CONNECTED;
	UNLOCKDATA(dp);

	/* enter an infinite loop */
	for (quit = 0; quit == 0;) {
		FD_ZERO(&fdset);
		FD_SET(sock, &fdset);

		/* XXX: should i lock the data before select ? guess, no */
		if (select(sock + 1, &fdset, NULL, NULL, &tv) > 0) {
			(void) memset(&tdata, 0, sizeof(struct tdata));
			/* somebody connects */
			tdata.data = dp;
			tdata.socklen = sizeof(struct sockaddr_in);
			tdata.sock = accept(sock, (struct sockaddr *)
			    &tdata.sa, &tdata.socklen);
			if (tdata.sock != INVALID_SOCKET)
				/* start serving new connection */
				_beginthread(packer, 0L, &tdata);
		}

		/* disconnect, if requested */
		LOCKDATA(dp);
		if (dp->flags & FL_DISCONNECT) {
			killsock(&dp->sock);
			quit++;				/* schedule quit */
		}
		UNLOCKDATA(dp);
	}

	/* wait unpacker thread to finish */
	for (quit = 0; quit == 0;) {
		Sleep(100);
		LOCKDATA(dp);
		if (!(dp->flags & FL_UNPACKER_ON)) {
			quit++;
		}
		UNLOCKDATA(dp);
	}
	
	/*and clear running flag */
	LOCKDATA(dp);
	dp->flags &= ~(FL_LISTENER_ON | FL_CONNECTED | FL_DISCONNECT);
	UNLOCKDATA(dp);
}

/*
 * look what thread is to receive the
 * data, and write data into thread's socket
 * return 0, if unpacked and delivered successfully, -1 otherwise
 */
int
unpack(char *reply, struct gdata *dp)
{
	char *p;
	char b64[BSIZ * 2];
	unsigned dlen;
	int rval;
	struct tdata *tp, *tmp;
	struct header header;

	rval = -1;	/* mark not - successful return code */

	for (p = reply; p != (char *) 1; p = strchr(p, '\n') + 1) {
		if (memcmp(p, MAGIC, strlen(MAGIC))
		    || parseheader(p, &header) != 0)
			continue;
		
		/*
		 * if header is correct, mark succesfull parsing and
		 * queue mesage deletion
		 */
		rval = 0;
		trace("unpack: got data for thread %d", header.id);
	
		/* find the beginning of the message */
		if ((p = strchr(p, ':')) == NULL) {
			trace("corrupted message received [%s]", p);
			break;
		}

		/* skip all whitespaces */
		for (++p; isspace(*p); )
			p++;

		/* decode message */
		dlen = base64decode(p, b64);

		/* decrypt the data */
		cryptdecrypt(b64, dlen);

		/* find the thread - receiver */
		LOCKDATA(dp);
		for (tp = NULL, tmp = dp->list; tmp != NULL; tmp = tmp->next) {
			if (tmp->id == header.id) {
				tp = tmp;
				break;
			}
		}
		UNLOCKDATA(dp);

		if (tp == NULL) {
			trace("unpack: unexistent thread %d", header.id);
			break;
		}

		if (dlen == 0) {
			/* remote server disconnected */
			trace("unpack: got disconnection for %d", tp->id);
			LOCKDATA(dp);
			delnode(tp, &dp->list);
			UNLOCKDATA(dp);
			break;
		}

		/* send the data to a thread's socket */
		(void) send(tp->sock, b64, dlen, 0);
		break;
	}

	return (rval);
}

/*
 * scan the pop3 box, retrieve the mail
 */
void
scanbox(int sock, struct gdata *dp)
{
	char *reply, *p, *list;
	static unsigned cnt;
	int msgno, msglen;		/* message number and message length */

	if (cnt++ % 100 == 0)
		trace("unpacker: waiting for data...");

	reply = mailcmd(sock, "LIST");
	list = strdup(reply);

	for (p = list;p != (char *) 1 && *p != '\0' && *p != '.';
	    p = strchr(p, '\n') + 1) {
		if (sscanf(p, "%d %d", &msgno, &msglen) != 2)
			continue;
		reply = mailcmd(sock, "RETR %d", msgno);
		if (unpack(reply, dp) == 0) 
			(void) mailcmd(sock, "DELE %d", msgno);
	}

	free(list);
}

/*
 * authenticate with POP3 server and enter TRANSACTION state.
 * return 0 on success, -1 on error.
 * sock is the opened socket to the POP3 server
 */
int
pop3login(const char *mailserv, const char *user, const char *passwd)
{
	int sock;
	char b64[BSIZ], *reply, *p, stripped[BSIZ];
	
	/* connect to a POP3 server */
	if ((sock = mailconn(mailserv, POP3PORT)) == -1) {
		showerr(MB_OK | MB_ICONEXCLAMATION,
		    "cannot connect to the %s:%d", mailserv, POP3PORT);
		return (-1);
	}

	/* strip everything after @ */
	strncpy(stripped, user, sizeof(stripped));
	if ((p = strchr(stripped, '@')) != NULL)
		*p = '\0';
	
	/* first, try to auth using user/pass */
	reply = mailcmd(sock, "USER %s", stripped);
	reply = mailcmd(sock, "PASS %s", passwd);
	if (memcmp(reply, "+OK", 3) == 0)
		return (sock);

	/* authenticate using AUTH LOGIN method */
	reply = mailcmd(sock, "AUTH LOGIN");	/* Username: */
	
	/* send the bsae64-encoded username */
	(void) base64encode(stripped, strlen(stripped), b64); 
	reply = mailcmd(sock, "%s", b64);
	
	/* send the base64-encoded password */
	(void) base64encode(passwd, strlen(passwd), b64);
	reply = mailcmd(sock, "%s", b64);

	if (memcmp(reply, "+OK", 3) == 0)
		return (sock);

	/* TODO: authenticate using cram-md5 method */
	
	/* if cannot auth, exit */
	showerr(MB_OK | MB_ICONEXCLAMATION,"cannot authenticate %s", stripped);

	return (-1);
}

/*
 * the `unpacker' thread entry point
 */
void
unpacker(void *arg)
{
	int quit = 0;
	struct gdata *dp = arg;
	static int sock;		/* socket to the POP3 server */
	
	/* are we started already ? */
	LOCKDATA(dp);
	if (dp->flags & FL_UNPACKER_ON)
		quit++;
	else
		dp->flags |= FL_UNPACKER_ON;	/* no, set 'running' flag */
	UNLOCKDATA(dp);
	
	if (quit) {
		showerr(MB_OK, "unpacker is running already");
		return;	/* yes, we're running, return */
	}

	/* fall into infinite loop */
	for (quit = 0; quit == 0;) {
	
		/* connect to a POP3 server */
		sock = pop3login(dp->mailserv, dp->mailuser, dp->mailpasswd);
		if (sock < 0) {
			LOCKDATA(dp);
			dp->flags |= FL_DISCONNECT;
			dp->flags &= ~FL_UNPACKER_ON;
			UNLOCKDATA(dp);
			killsock(&sock);
			return;
		}
		trace("logged on POP3 server");

		/* scan the mailbox */
		scanbox(sock, dp);

		/* disconnect from POP3 server */
		(void) mailcmd(sock, "QUIT");
		killsock(&sock);
		

		Sleep(500);
		LOCKDATA(dp);
		if (dp->flags & FL_DISCONNECT) {
			dp->flags &= ~FL_UNPACKER_ON;
			quit++;
		}
		UNLOCKDATA(dp);
	}
}

/*
 * reflect the connection state on the GUI
 */
void
setguistate(HWND hDlg, struct gdata *dp)
{
	static int prevstate;
	char title[BSIZ];
	int state;

	LOCKDATA(dp);
	state = dp->flags & FL_CONNECTED;
	UNLOCKDATA(dp);

	if (prevstate != state) {
		/* state has been changed, refresh the GUI */
		EnableWindow(GetDlgItem(hDlg, ID_EDIT_MAILSERVER), !state);
		EnableWindow(GetDlgItem(hDlg, ID_EDIT_MAILUSER), !state);
		EnableWindow(GetDlgItem(hDlg, ID_EDIT_MAILPASSWD), !state);
		EnableWindow(GetDlgItem(hDlg, ID_EDIT_EMAIL), !state);
		EnableWindow(GetDlgItem(hDlg, ID_EDIT_SERVER), !state);
		EnableWindow(GetDlgItem(hDlg, ID_EDIT_PORT), !state);
		EnableWindow(GetDlgItem(hDlg, ID_EDIT_LPORT), !state);
		EnableWindow(GetDlgItem(hDlg, ID_BTN_LOG), !state);
		SetDlgItemText(hDlg, ID_BTN_CONNECT, state ?
		    "disconnect" : "connect");
		(void) sprintf(title, "mproxy: %s", state ?
		    "listening on port " : "not connected");
		if (state)
			strcat(title, dp->lport);
		SetWindowText(hDlg, title);
	}
	prevstate = state;
}

/*
 * save the settings into registry and initialize global data
 */
void
savedlg(HWND hDlg, struct gdata *dp)
{
	setregval(hDlg, ID_EDIT_PORT);
	setregval(hDlg, ID_EDIT_SERVER);
	setregval(hDlg, ID_EDIT_MAILUSER);
	setregval(hDlg, ID_EDIT_MAILSERVER);
	setregval(hDlg, ID_EDIT_EMAIL);
	setregval(hDlg, ID_EDIT_LPORT);
	setregval(hDlg, ID_BTN_LOG);

	LOCKDATA(dp);
#define GETVAL(id, where) GetDlgItemText(hDlg, id, where, sizeof(where))
	GETVAL(ID_EDIT_EMAIL, dp->email);
	GETVAL(ID_EDIT_SERVER, dp->server);
	GETVAL(ID_EDIT_PORT, dp->port);
	GETVAL(ID_EDIT_MAILSERVER, dp->mailserv);
	GETVAL(ID_EDIT_MAILUSER, dp->mailuser);
	GETVAL(ID_EDIT_MAILPASSWD, dp->mailpasswd);
	GETVAL(ID_EDIT_LPORT, dp->lport);
	UNLOCKDATA(dp);
}

/*
 * save individual control's text into registry
 */
void
setregval(HWND hDlg, int id)
{
	char text[BSIZ];
	const char *name;
	DWORD len;
	HKEY key;
	LONG ret;

	/* handle check buttons specially */
	if (id == ID_BTN_LOG)
		sprintf(text, "%d", IsDlgButtonChecked(hDlg, id));
	else
		GetDlgItemText(hDlg, id, text, sizeof(text));

	name = idtoname(id);

	ret = RegCreateKeyEx(HKEY_LOCAL_MACHINE, regpath, 0L, NULL, 0L,
			KEY_ALL_ACCESS, NULL, &key, NULL);

	if (ret == ERROR_SUCCESS) {
		len = (DWORD) strlen(text);
		RegSetValueEx(key, name, 0L, REG_SZ, (BYTE *) text, len);
		RegCloseKey(key);
	}
}

/*
 * convert the numeric control id to the registry key name
 */
const char *
idtoname(int id)
{
	const char *name = "unknown";
	
	switch (id) {
		case ID_EDIT_MAILSERVER:	name = "MAIL server"; break;
		case ID_EDIT_MAILUSER:		name = "MAIL user"; break;
		case ID_EDIT_EMAIL:		name = "Email"; break;
		case ID_EDIT_SERVER:		name = "Remote server"; break;
		case ID_EDIT_PORT:		name = "Remote port";	break;
		case ID_EDIT_LPORT:		name = "Local port"; break;
		case ID_BTN_LOG:		name = "Log";	break;
		default: break;
	}

	return (name);
}
/*
 * retrieve the registry value for the control with identifier `id'
 */
const char *
reggetval(int id)
{
	static char value[BSIZ];
	HKEY key;
	LONG ret;
	DWORD type, len;
	const char *name;

	value[0] = '\0';

	/* try to open path */
	ret = RegCreateKeyEx(HKEY_LOCAL_MACHINE, regpath, 0L, NULL, 0L,
			KEY_ALL_ACCESS, NULL, &key, NULL);
	
	if (ret != ERROR_SUCCESS)
		return (value);

	name = idtoname(id);
	len = sizeof(value);
	ret = RegQueryValueEx(key, name, NULL, &type, (BYTE *) value, &len);
	
	if (ret != ERROR_SUCCESS)
		value[0] = '\0';

	RegCloseKey(key);
	
	return (value);
}

/*
 * initialize dialog. get values from the registry
 */
void
initdlg(HWND hDlg)
{
	SetDlgItemText(hDlg, ID_EDIT_MAILSERVER, reggetval(ID_EDIT_MAILSERVER));
	SetDlgItemText(hDlg, ID_EDIT_MAILUSER, reggetval(ID_EDIT_MAILUSER));
	SetDlgItemText(hDlg, ID_EDIT_EMAIL, reggetval(ID_EDIT_EMAIL));
	SetDlgItemText(hDlg, ID_EDIT_SERVER, reggetval(ID_EDIT_SERVER));
	SetDlgItemText(hDlg, ID_EDIT_PORT, reggetval(ID_EDIT_PORT));
	SetDlgItemText(hDlg, ID_EDIT_LPORT, reggetval(ID_EDIT_LPORT));
	CheckDlgButton(hDlg, ID_BTN_LOG, atoi(reggetval(ID_BTN_LOG)));

	/* set empty MAIL password */
	SetDlgItemText(hDlg, ID_EDIT_MAILPASSWD, "");
}

/*
 * Display a message box. Return button identifier has been pressed
 */
UINT
showerr(UINT type, const char *fmt, ...)
{
	char msg[BSIZ];
	va_list ap;
	
	/* windows lacks `vsnprintf'. XXX possible buffer overflow */
	va_start(ap, fmt);
	(void) vsprintf(msg, fmt, ap);
	va_end(ap);

	/* display the prepared content */
	return (MessageBox(NULL, msg, "Notification", type));
}

/*
 * The dialog box procedure.
 */
BOOL CALLBACK
DlgProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	WSADATA wdata;
	static struct gdata data;

	switch (msg) {
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case ID_BTN_QUIT:
					PostQuitMessage(0);
					break;
				case ID_BTN_CONNECT:
					/* connect/disconnect pressed */

					/* set debug flag, so trace() can 
					 * retrieve it */
					SetWindowLong(hDlg, GWL_USERDATA,
					IsDlgButtonChecked(hDlg, ID_BTN_LOG));
					LOCKDATA(&data);
					if (data.flags & FL_CONNECTED) {
						data.flags |= FL_DISCONNECT;
					} else {
						savedlg(hDlg, &data);
						_beginthread(listener,0,&data);
						_beginthread(unpacker,0,&data);
					}
					UNLOCKDATA(&data);
					break;
			}
			break;
		case WM_TIMER:
			setguistate(hDlg, &data);
			break;
		case WM_INITDIALOG:
			/* set nice icon */
			SendMessage(hDlg, WM_SETICON, (WPARAM) ICON_SMALL,
			    (LPARAM) LoadIcon(NULL, IDI_APPLICATION));
			/* initialize winsock library */
			if (WSAStartup(MAKEWORD(2,2), &wdata) != 0) {
				showerr(MB_OK | MB_ICONEXCLAMATION,
				"cannot init WinSock");
			}
			/* get previous values from the registry */
			initdlg(hDlg);
			/* init sync object */
			InitializeCriticalSection(&data.syncr);
			/* set disconnected state */
			data.sock = -1;
			/* set timer to reflect changes on GUI */
			SetTimer(hDlg, 0, 100, NULL);
			/* create the statusbar */
			hbar = CreateStatusWindow(WS_CHILD | WS_VISIBLE,
			    "Not connected", hDlg, ID_STATUSBAR);
			break;
		case WM_QUIT:
			WSACleanup();
			DeleteCriticalSection(&data.syncr);
			KillTimer(hDlg, 0);
			DestroyWindow(hDlg);
			break;
		case WM_CLOSE:
			PostQuitMessage(0);
			break;
		default:
			break;
	}

	return (FALSE);
}

/*
 * entry point
 */
int WINAPI
WinMain(HINSTANCE inst, HINSTANCE prev, LPSTR cmd, int show)
{
	/* describe the main dialog */
	struct {
		DLGTEMPLATE tmpl;		/* 18 bytes */
		WORD menu, class;
		wchar_t caption[6 + 1];
		WORD fontsiz;
		wchar_t fontface[6 + 1];
		
		DLGITEMTEMPLATE btn_local;	/* 18 bytes */
		WORD local_signature, local_class;
		wchar_t local_title[5 + 1]; WORD local_data;
		
		DLGITEMTEMPLATE btn_remote;
		WORD remote_signature, remote_class;
		wchar_t remote_title[7 + 1]; WORD remote_data;
		
		DLGITEMTEMPLATE mailserver;
		WORD mailserver_signature, mailserver_class;
		wchar_t mailserver_title[1 + 1]; WORD mailserver_data;
		
		DLGITEMTEMPLATE mailuser;
		WORD mailuser_signature, mailuser_class;
		wchar_t mailuser_title[1 + 1]; WORD mailuser_data;
		
		DLGITEMTEMPLATE mailpasswd;
		WORD mailpasswd_signature, mailpasswd_class;
		wchar_t mailpasswd_title[1 + 1]; WORD mailpasswd_data;
		
		DLGITEMTEMPLATE email;
		WORD email_signature, email_class;
		wchar_t email_title[1 + 1]; WORD email_data;
		
		DLGITEMTEMPLATE server;
		WORD server_signature, server_class;
		wchar_t server_title[1 + 1]; WORD server_data;
		
		DLGITEMTEMPLATE port;
		WORD port_signature, port_class;
		wchar_t port_title[1 + 1]; WORD port_data;
		
		DLGITEMTEMPLATE st_mailserver;
		WORD st_mailserver_signature, st_mailserver_class;
		wchar_t st_mailserver_title[11 + 1]; WORD st_mailserver_data;
		
		DLGITEMTEMPLATE st_mailuser;
		WORD st_mailuser_signature, st_mailuser_class;
		wchar_t st_mailuser_title[5 + 1]; WORD st_mailuser_data;
		
		DLGITEMTEMPLATE st_mailpasswd;
		WORD st_mailpasswd_signature, st_mailpasswd_class;
		wchar_t st_mailpasswd_title[13 + 1]; WORD st_mailpasswd_data;
		
		DLGITEMTEMPLATE st_email;
		WORD st_email_signature, st_email_class;
		wchar_t st_email_title[5 + 1]; WORD st_email_data;
		
		DLGITEMTEMPLATE st_server;
		WORD st_server_signature, st_server_class;
		wchar_t st_server_title[7 + 1]; WORD st_server_data;
		
		DLGITEMTEMPLATE st_port;
		WORD st_port_signature, st_port_class;
		wchar_t st_port_title[5 + 1]; WORD st_port_data;
		
		DLGITEMTEMPLATE lport;
		WORD lport_signature, lport_class;
		wchar_t lport_title[1 + 1]; WORD lport_data;
		
		DLGITEMTEMPLATE st_lport;
		WORD st_lport_signature, st_lport_class;
		wchar_t st_lport_title[21 + 1]; WORD st_lport_data;
		
		DLGITEMTEMPLATE btn_log;
		WORD log_signature, log_class;
		wchar_t log_title[27 + 1]; WORD log_data;
		
		DLGITEMTEMPLATE btn_connect;
		WORD connect_signature, connect_class;
		wchar_t connect_title[7 + 1]; WORD connect_data;
		
		DLGITEMTEMPLATE btn_quit;
		WORD quit_signature, quit_class;
		wchar_t quit_title[5 + 1]; WORD quit_data;
		
		DLGITEMTEMPLATE st_copyright;
		WORD copy_signature, copy_class;
		wchar_t copy_title[63 + 1]; WORD copy_data;
		
	} dlg = {
		{ WS_CAPTION | WS_POPUP | WS_SYSMENU | WS_VISIBLE | DS_SETFONT
		| WS_MINIMIZEBOX, 0L, 20, 200, 200, 300, 125, } ,
		0, 0, L"mproxy", 8, L"Tahoma",

		/* 'local' group box */
		{ WS_CHILD | WS_VISIBLE | BS_GROUPBOX, 0L,
		5, 5, 140, 60, ID_GRP_LOCAL, },
		0xffff, 0x080, L"local", 0,
		
		/* 'remote' group box */
		{ WS_CHILD | WS_VISIBLE | BS_GROUPBOX, 0L,
		150, 5, 140, 60, ID_GRP_LOCAL, },
		0xffff, 0x080, L"remote ", 0,
		
		/* mail server edit box  */
		{ WS_CHILD | WS_VISIBLE | WS_BORDER | WS_TABSTOP
		| ES_AUTOHSCROLL, 0L, 10, 15, 70, 12, ID_EDIT_MAILSERVER, },
		0xffff, 0x081, L" ", 0,
		
		/* mail user edit box  */
		{ WS_CHILD | WS_VISIBLE | WS_BORDER | WS_TABSTOP 
		| ES_AUTOHSCROLL, 0L, 10, 30, 70, 12, ID_EDIT_MAILUSER, },
		0xffff, 0x081, L" ", 0,
		
		/* mail password edit box  */
		{ WS_CHILD | WS_VISIBLE | WS_BORDER | WS_TABSTOP 
		| ES_AUTOHSCROLL | ES_PASSWORD,
		0L, 10, 45, 70, 12, ID_EDIT_MAILPASSWD, },
		0xffff, 0x081, L" ", 0,
		
		/* email edit box  */
		{ WS_CHILD | WS_VISIBLE | WS_BORDER | WS_TABSTOP 
		| ES_AUTOHSCROLL, 0L, 155, 15, 70, 12, ID_EDIT_EMAIL, },
		0xffff, 0x081, L" ", 0,
		
		/* server edit box  */
		{ WS_CHILD | WS_VISIBLE | WS_BORDER | WS_TABSTOP 
		| ES_AUTOHSCROLL, 0L, 155, 30, 70, 12, ID_EDIT_SERVER, },
		0xffff, 0x081, L" ", 0,
		
		/* port edit box  */
		{ WS_CHILD | WS_VISIBLE | WS_BORDER | WS_TABSTOP 
		| ES_AUTOHSCROLL, 0L, 155, 45, 70, 12, ID_EDIT_PORT, },
		0xffff, 0x081, L" ", 0,
		
		/* mail server static text  */
		{ WS_CHILD | WS_VISIBLE, 0L,
		85, 15, 40, 12, ID_STATIC_MAILSERVER, },
		0xffff, 0x082, L"Mail server", 0,
		
		/* mail user static text  */
		{ WS_CHILD | WS_VISIBLE, 0L,
		85, 30, 40, 12, ID_STATIC_MAILUSER, },
		0xffff, 0x082, L"Email", 0,
		
		/* mail password static text  */
		{ WS_CHILD | WS_VISIBLE, 0L,
		85, 45, 55, 12, ID_STATIC_MAILPASSWD, },
		0xffff, 0x082, L"Mail password", 0,
		
		/* email static text  */
		{ WS_CHILD | WS_VISIBLE, 0L,
		230, 15, 40, 12, ID_STATIC_EMAIL, },
		0xffff, 0x082, L"Email", 0,
		
		/* server static text  */
		{ WS_CHILD | WS_VISIBLE, 0L,
		230, 30, 40, 12, ID_STATIC_SERVER, },
		0xffff, 0x082, L"Server ", 0,
		
		/* port static text  */
		{ WS_CHILD | WS_VISIBLE, 0L,
		230, 45, 40, 12, ID_STATIC_PORT, },
		0xffff, 0x082, L"Port ", 0,
		
		/* local port edit box */
		{ WS_CHILD | WS_VISIBLE | WS_BORDER | WS_TABSTOP 
		| ES_AUTOHSCROLL | ES_NUMBER, 0L,
		10, 70, 40, 12, ID_EDIT_LPORT, },
		0xffff,	0x0081, L" ", 0,
		
		/* local port static text */
		{ WS_CHILD | WS_VISIBLE, 0L,
		55, 70, 60, 12, ID_STATIC_LPORT, },
		0xffff,	0x0082, L"Local port to listen", 0,
		
		/* log checkbox */
		{ WS_CHILD | WS_VISIBLE | BS_AUTOCHECKBOX | WS_TABSTOP, 0L,
		10, 85, 120, 12, ID_BTN_LOG, },
		0xffff,	0x0080, L"Log to a file (mproxy.log) ", 0,
		
		/* connect button. XXX: DWORD-align ! */
		{ WS_CHILD | WS_VISIBLE | BS_PUSHBUTTON | WS_TABSTOP, 0L,
		210, 85, 50, 12, ID_BTN_CONNECT, },
		0xffff,	0x0080, L"connect", 0,
		
		/* quit button. XXX: DWORD-align ! */
		{ WS_CHILD | WS_VISIBLE | BS_PUSHBUTTON  | WS_TABSTOP, 0L,
		270, 85, 20, 12, ID_BTN_QUIT, },
		0xffff,	0x0080, L"quit ", 0,

		/* copyright text */
		{ WS_CHILD | WS_VISIBLE, 0L,
		10, 100, 220, 24, ID_STATIC_COPYRIGHT, },
		0xffff, 0x082,
		L"Copyright (c) devnull @ #c RusNet channel, "
		L"http://www.c.org.ua",0,
	};

	/* display it */
	DialogBoxIndirect(inst, (LPDLGTEMPLATE) &dlg, NULL, DlgProc);
}
